/*****************************************************************************/
/* picp-pic16f876-skypic-2.c  Julio-2007                                     */
/*---------------------------------------------------------------------------*/
/* Servidor para programar microcontroladores pics                           */
/* Servidor: PICP                                                            */
/* Implementado para la tarjeta Skypic a 20MHZ                               */
/*---------------------------------------------------------------------------*/
/* Adaptado a la tarjeta Skypic. La programacion de los pics se realiza      */
/* por el PUERTO B de la Skypic (CT2). Si lo que se esta grabando es otra    */
/* skypic, se debe conectar un cable desde el puerto B de la skypic          */
/* grabadora al puerto de grabacion (CT4) de la skypic a grabar              */
/*---------------------------------------------------------------------------*/
/*  Juan Gonzalez <juan@iearobotics.com>                                     */
/*  Rafael Trevio <skasi.7@gmail.com>                                       */
/*---------------------------------------------------------------------------*/
/*  LICENCIA GPL                                                             */
/*****************************************************************************/

//-- Especificar el pic a emplear
#include <pic16f876a.h>

//--------------------------------------------------------------------------
//            CONSTANTES PARA LAS TRAMAS DEL PROTOCOLO  
//--------------------------------------------------------------------------

//-- Identificadores de los servicios
#define SPING  'P'   // Servicio de PING (0x50)
#define SID    'I'   // Servicio de identificacion (0x49)
#define SRST   'T'   // Servicio de reset (0x54)
#define SDATA  'D'   // Servicio de load data (0x44)
#define SBEG   'B'   // Servicio de begin cycle (0x42)
#define SINC   'A'   // Servicio de increment address (0x41)
#define SRD    'R'   // Servicio de read data (0x52)
#define SCONF  'C'   // Servicio de load configuration (0x43)
#define SPROG  'W'   // Servicio PROGRAMACION (0x57)
#define SJUMP  'J'   // Servicio de jump (0x50)

//-- Identificadores para la respuesta
#define RPING 'O'   // Respuesta al Servicio de PING (0x4F)
#define RSI   'I'   // RSI. Codigo de respuesta del servicio de identificacion
#define RRST  'T'   // Respuesta al Servicio de reset (0x54)
#define RDATA 'D'   // Respuesta al Servicio de load data (0x44)
#define RBEG  'B'   // Respuesta al Servicio de begin cycle (0x42)
#define RINC  'A'   // Respuesta al Servicio de increment address (0x41)
#define RCONF 'C'   // Respuesta al Servicio de load configuration (0x43)
#define RRD   'R'   // Respuesta al Servicio de read data (0x52)
#define RPROG 'W'   // Respuesta servicio PROG
#define RJUMP  'J'  // Respuesta al Servicio de jump (0x50)

//----- Datos devueltos por el servicio de identificacion
#define IS	0x40    // IS. Identificacion del servidor
#define IM	0x30    // IM. Identificacion del microcontrolador
#define IPV	0x12    // IPV. SKYPIC v2

//------------------------------------------------------------------------
//                CONSTANTES PARA EL TEMPORIZADOR   
//------------------------------------------------------------------------
//-- Definiciones
#define TICKS10 0x3D  // Valor con el que inicializar contador para
                      // conseguir TICKs de 10ms

//------------------------------------------------------------------------
//                CONSTANTES PARA EL PUERTO SERIE   
//------------------------------------------------------------------------
//-- CONSTANTES
#define B9600_4MHZ   0x19  //-- Vel. 9600 baudios cuando cristal de 4MHz
#define B9600_20MHZ  0x81  //-- vel. 9600 baudios cuando cristal de 20MHz

//-- Establecer la velocidad para un cristal de 20MHZ
//-- La skypic lleva 20Mhz
#define B9600 B9600_20MHZ


//-----------------------------------------------------------------------
//          CONSTANTES PARA LOS PINES DEL PUERTO B
//-----------------------------------------------------------------------
//- CLOCK --> RB6
//- DATA ---> RB7
//- RESET --> MCLR
	
//-- Se definen los bits. En la Skypic estos bits se sacan por el 
//-- puerto B y estan pensados para grabar otra skypic conectada
//-- por su conector CT4
#define CLOCK 1<<3		// Bit 3
#define DATA  1<<7		// Bit 7
#define RESET 1<<4  	// Bit 4

//------------------------------------------------------------------------
//       CODIGOS DE LOS COMANDOS DEL MONITOR DEL PIC 
//------------------------------------------------------------------------
#define READ_DATA   0x04  // Comando READ_DATA from Program Memory
#define LOAD_DATA   0x02  // Comando: LOAD DATA For Program Memory
#define BEG_ER_PRO  0x08  // Comando: BEGIN ERASE PROGRAMMING CICLE	
#define INC_ADDR    0x06  // comando: INCREMENT ADDRESS
#define LOAD_CONFIG 0x00  // Comando: LOAD CONFIGURATION


//-------------------------------------------------------------------------
//                   VARIABLES GLOBALES      
//-------------------------------------------------------------------------

//-- Lectura de palabras
unsigned char palabral;    //-- Parte baja
unsigned char palabrah;    //-- Parte alta	

/*--------------------------------------------------------------------------*/
/*                  FUNCIONES DEL PUERTO SERIE                              */
/*--------------------------------------------------------------------------*/

/****************************************************/
/* Configurar el puerto serie a N81 y 9600 Baudios  */
/****************************************************/
void sci_init(void)
{
  SPBRG = B9600; //-- 9600 baudios
  TXSTA = 0x24;  //-- Configurar transmisor
  RCSTA = 0x90;  //-- Configurar receptor 
}

/******************************************/
/* Recibir un caracter por el SCI         */
/*----------------------------------------*/
/* DEVUELVE:                              */
/*   -Caracter recibido                   */
/******************************************/
unsigned char sci_read(void)
{
  //-- Eserar hasta que llegue el dato
  while (!RCIF);
    
  return RCREG;
}

/*****************************************/
/* Transmitir un caracter por el SCI     */
/*---------------------------------------*/
/* ENTRADAS:                             */
/*   -car: Caracter a enviar             */
/*****************************************/
void sci_write(unsigned char car)
{
  //-- Esperar a que Flag de lista para transmitir se active
  while (!TXIF);
    
  //-- Hacer la transmision
  TXREG=car;
}

/*--------------------------------------------------------------------------*/
/*                  FUNCIONES PARA EL TEMPORIZADOR                          */
/*--------------------------------------------------------------------------*/
/************************************/
/* Configurar el temporizador 0     */
/************************************/
void timer0_configurar()
{
  //-- Usarlo en modo temporizador, prescaler = 256
  OPTION_REG=0x87;
}

/************************************************/
/* Hacer una pausa en unidades de 10ms          */
/* ENTRADA:                                     */
/*   -pausa: Valor de la pausa en decenas de ms */
/************************************************/
void delay0(unsigned char pausa)
{
  
  //-- Esperar hasta que trancurran pausa ticks de reloj
  while(pausa>0) {
    TMR0=TICKS10;   // Inicializar contador
    T0IF=0;         // Quitar flag overflow
    //-- Esperar hasta que bit T0IF se ponga a '1'. Esto
    //-- indica que han transcurrido 10ms
    while (!T0IF);
      
    //-- Ha transcurrido un tick
    pausa--;
  }    
}

/*------------------------------------------------------------------*/
/*  FUNCIONES DE IMPLEMENTACION DEL PROTOCOLO ICSP                  */
/*------------------------------------------------------------------*/

/*********************************************/
/* Enviar bits por DATA                      */
/* Se comienza por los menos significativos  */
/* ENTRADAS:                                 */
/*     -dato: Byte a enviar                  */
/*     -nbits: numero de bits a enviar       */
/*********************************************/
void write_bits(unsigned char dato, unsigned char nbits)
{
  unsigned char i;
  
  
  for (i=0; i<nbits; i++) {
    
    //-- Enviar bit menos significativo de dato
    if ((dato & 0x01)==0)
			PORTB = PORTB & ~(DATA);  //-- Enviar un cero
    else 
      PORTB = PORTB | DATA;     //-- Enviar un uno
    
    //-- Activar el reloj
    PORTB = PORTB | CLOCK;  //-- Flanco de subida
    PORTB = PORTB & ~CLOCK; //-- Flanco de bajada
    
    //-- Acceder al siguiente bit de dato
    dato = dato >> 1;
  }  
}

/**********************************************************************/
/* Leer a traves del ICSP                                             */
/* ENTRADA: Numero de bits a leer                                     */
/* DEVUELVE: los bits leidos. Se comienza a insertar los bits por el  */
/*    bit mas significativo                                           */
/**********************************************************************/
unsigned char read_bits(unsigned char nbits)
{
  unsigned char dato=0x00;
  unsigned char i;
  
  for (i=0; i<nbits; i++) {
    dato=dato>>1;
  
    //-- Flanco de reloj: subida y baja.
    //-- A continuacion el dato esta listo
    PORTB = PORTB | CLOCK;
    PORTB = PORTB & ~CLOCK;
  
    if ((PORTB & DATA)==0X00)
      dato=dato & ~0x80; //-- Recibido un 0
    else
      dato=dato | 0x80;  //-- Recibido un 1
  
  }
  
  return dato;
}


/**********************************/
/* Enviar un comando por el ICSP  */
/* ENTRADAS:                      */
/*   -cmd : Comando a enviar      */
/**********************************/
void send_cmd(unsigned char cmd)
{
  //-- Poner el reloj a cero
  PORTB = PORTB & ~CLOCK;
  write_bits(cmd,0x6);
  
  //-- Poner DATA a '1' para indicar fin del comando
  PORTB = PORTB | DATA;
  
  //-- Aqui debe haber al menos una espera de 1us
  //-- Usamos 4 instrucciones que no hacen nada
  //-- El puerto A es de entrada, por lo que si escribimos en el
  //-- no hace nada
  //-- A una frecuencia de 20Mhz, el pic tarda en ejecutar cada
  //-- instruccion 250ns. Al menos se tienen que ejecutar 4
  PORTA=0xff;
  PORTA=0x00;
  PORTA=0xff;
  PORTA=0x00;
}

/***************************************/
/* Enviar un dato por el ICSP          */
/* Son palabras de 14 bits             */
/* ENTRADAS:                           */
/*   -datoh: byte alto                 */
/*   -datol: byte bajo                 */
/***************************************/
void send_data(unsigned char datoh, unsigned char datol)
{
  //-- Bit de start 0
  write_bits(0x00,1);
  
  //-- 8 bits de menor peso
  write_bits(datol,8);
  
  //-- 6 bits de mayor peso + bit de stop
  //palabrah = palabrah & 0x3F;
  write_bits(datoh, 7);
}

/*************************************************************/
/* Recibir un dato de 14 bits por el ICSP                    */
/* Se actualizan las variables globales palabral y palabrah  */
/*************************************************************/
void recv_data()
{
  //unsigned char i;
  
  //-- Configurar pin DATA para entrada
  TRISB = TRISB | DATA;
 
  //-- Leer bit de start y descartarlo 
  read_bits(1);
  
  //-- Leer 8 bits menos significativos
  palabral=read_bits(8);
  
  //-- Leer los 6 bits mas significativos + el bit de stop
  palabrah=read_bits(7);
  
  //-- Alinear la parte alta para que los bits menos significativos
  //-- esten a la derecha. Los dos mas significativos se ponen a 0
  palabrah=(palabrah>>1)&0x3F; 

  //-- Configurar pin DATA para salida
  TRISB = TRISB & ~DATA;
}



/*--------------------------------------------------------------------------*/
/*                SERVICIOS DEL PICP                                        */
/*--------------------------------------------------------------------------*/

/*********************/
/* Servicio PING     */
/*********************/
void serv_ping() {
	sci_write(RPING);
}

/*******************************/
/* Servicio de Identificacion  */
/*******************************/
void serv_id() {
	sci_write(RSI);
	sci_write(IS);
	sci_write(IM);
	sci_write(IPV);
}

/***********************************/
/* Servicio de reset               */
/***********************************/
void serv_rst() {
 
  //-- Activar el reset y dejar las demas senales a 0
  PORTB = RESET;
  
  //-- Esperar 20 ms
  delay0(2);
	
	//-- Desactivar reset
	PORTB = 0;

  //-- Enviar trama de respuesta
	sci_write(RRST);
}

/************************************/
/* Servicio load data               */
/************************************/
void serv_load_data()
{
  unsigned char datol;
  unsigned char datoh;
  
  //-- Leer los bytes bajos y alto de la palabra a cargar en el PIC
  datol = sci_read();
  datoh = sci_read();
  
  //-- Enviar el comando de Load data
  send_cmd(LOAD_DATA);
  
  //-- Enviar la palabra
  send_data(datoh, datol);
  
  //-- Enviar la trama de respuesta al PC
  sci_write(RDATA); 
}

/***************************************/
/* Servicio de Begin programing cycle  */
/***************************************/
void serv_begin() 
{

  //-- Enviar comando
	send_cmd(BEG_ER_PRO);

  //-- Enviar la trama de respuesta
	sci_write(RBEG);
}


/************************************/
/* Servicio de Increment address    */
/************************************/
void serv_inc() 
{
  //-- Enviar el comando
	send_cmd(INC_ADDR);
  
  //-- Enviar la trama de respuesta
	sci_write(RINC); 
}


/***********************************/
/* Servicio read-data              */
/***********************************/
void serv_rd()
{
  //-- Enviar comando READ_DATA
  send_cmd(READ_DATA); 
  
  //-- Leer la palabra
  recv_data();
  
  //-- Enviar la trama de respuesta  
  sci_write(RRD);         // Cabecera de la trama
	sci_write(palabral); // byte bajo
	sci_write(palabrah); // byte alto
  
}


/*******************************************/
/* Servicio de carga de configuracion      */
/*******************************************/
void serv_conf() 
{

	send_cmd(LOAD_CONFIG); // Mensaje de configuracin
	send_data(0x00, 0x00); // Enva cualquier cosa
	
	sci_write(RCONF);   // Enva la respuesta
}


/***********************************/
/* Servicio de programacion:       */
/*  LOAD_DATA + BEGIN_PROG         */
/***********************************/
void serv_prog() 
{
  unsigned char datol;
  unsigned char datoh;
  
  //-- Leer los bytes bajos y alto de la palabra a cargar en el PIC
  datol = sci_read();
  datoh = sci_read();
  
  //-- Enviar el comando de Load data
  send_cmd(LOAD_DATA);
  
  //-- Enviar la palabra
  send_data(datoh, datol);
  
  //-- comando de escritura
	send_cmd(BEG_ER_PRO);
  
  //-- Enviar la trama de respuesta al PC
  sci_write(RPROG); 
}

/********************************************************************/
/* Servicio jump. Incrementar el puntero de direccion tantas veces  */
/* como se especifique                                              */
/********************************************************************/
void serv_jump() {
  unsigned char i;
  unsigned char datol;
  unsigned char datoh;
  
  //-- Leer los bytes bajos y alto del salto
  datol = sci_read();  //-- byte bajo
  datoh = sci_read();  //-- byte alto

  //-- La version actual del SDCC (2.6.0) no trabaja muy bien con
  //-- enteros de 16 bits, por eso esta implementacion se basa solo
  //-- en numero de 8 bits. Primero se hace un bucle en el que se
  //-- avanzan tantos bloques de 256 palabras como indique datoh
  //-- Con un segundo bucle se avanzan datol palabras

  for(; datoh>0; datoh--) {
    
    //-- Enviar un bloque de 256 comandos de incremento
    //-- Primero 255...
    for (i=0; i<0xFF; i++)
      send_cmd(INC_ADDR);
    
    //-- Y luego el ultimo
    send_cmd(INC_ADDR);
  }
  
  //-- Avanzar un bloque menor de 256
	for (; datol > 0; datol--)
		send_cmd(INC_ADDR); // inc_address
	
  //-- Enviar la trama de respuesta
	sci_write(RJUMP); 
}


//----------------------------
//- Comienzo del programa  
//----------------------------

void main(void)
{
	unsigned char car;
	
	//-- Configurar el puerto serie
	sci_init();

	//-- Configurar el puerto B
  //-- CLOCK y RESET son de salida
  //-- DATA es entrada/salida pero inicialemnte se pone como salida
  TRISB = ~(CLOCK | DATA | RESET);
  
  //-- Inicialmente todas las senales a 0
  PORTB = 0x00;
  
  //-- Configurar el temporizador 0
  timer0_configurar();
	
	while (1) {
		
    //-- Esperar a que llega la cabecera de una trama
		car = sci_read();
	
		switch (car) {
			case SPING: // servicio SPING
				serv_ping();
				break;
		
			case SID: // Servicio de identificacion
				serv_id();
				break;
				  
			case SRST: // Servicio de reset
				serv_rst();
				break;
   
			case SDATA: // Servicio de load data
				serv_load_data();
				break;

      case SBEG: // Servicio de begin cycle
				serv_begin();
				break;

      case SINC: // Servicio de increment address
				serv_inc();
				break;

      case SRD: // Servicio de read data
				serv_rd();
			break;
      
			case SCONF: // Servicio de load configuration
				serv_conf();
				break;
      
      case SPROG: // Servicio de programacion
        serv_prog();
        break;

      //-- Por hacer...
			case SJUMP: // Servicio de salto
				serv_jump();
				break;
				
			default:
				break;
		}
	}
}
