#!/usr/bin/env python
# -*- coding: iso-8859-15 -*-

#-------------------------------------------------------------------
#
#-  Modulo stargate para la grabacion de microcontroladores PIC
#
# LICENCIA GPL
#-------------------------------------------------------------------

# Description: libStargate is a library for comunication to stargates
# Copyright: (C) 2007 by Juan González Gómez
#            (C) 2007 by Rafael Treviño Menéndez
# Authors: Juan González Gómez     <juan@iearobotics.com>
#          Rafael Treviño Menéndez <skasi.7@gmail.com>

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

import sys
import time
import libStargate.Main


#--- Cadena identificativa del servidor PICP
PICP_IDENT = "PICP"

#-----------------------------------
#------ Cabeceras de la trama 
#-----------------------------------
TRAMA_RESET_CAB  = 'T'  #-- Servicio Reset
TRAMA_RRESET_CAB = 'T'
TRAMA_CONF_CAB   = 'C'  #-- Servicio Load Config
TRAMA_RCONF_CAB  = 'C'
TRAMA_DATA_CAB   = 'D'  #-- Servicio Load Data
TRAMA_RDATA_CAB  = 'D'
TRAMA_RD_CAB     = 'R'  #-- Servicio Read Data
TRAMA_RRD_CAB    = 'R'
TRAMA_INC_CAB     = 'A' #-- Servicio INC Address
TRAMA_RINC_CAB    = 'A'
TRAMA_BEG_CAB     = 'B' #-- Servicio BEGIN Cycle
TRAMA_RBEG_CAB    = 'B'
TRAMA_JUMP_CAB    = 'J' #-- Servicio JUMP
TRAMA_RJUMP_CAB   = 'J'


#------------------------------------------------------------------------
#-- Constantes usadas con el metodo writeProgram del PICP para indicar
#-- lo que va ocurriendo con la grabacion 
#------------------------------------------------------------------------
WRITING_START=1      #-- Comienzo de la escritura
WRITING_INC=2        #-- Escritura de una palabra
WRITING_END=3        #-- Fin de la escritura
VERIFYING_START=4    #-- Comienzo de la verificacion
VERIFYING_INC=5      #-- Verificacion de una palabra
VERIFYING_ERROR=6    #-- Error de verificacion
VERIFYING_END=7      #-- Fin de la verificacion

#------------------------
#-- Cadenas de error
#------------------------
ERROR_COMUNICACION = "ERROR DE COMUNICACION"
ERROR_VERIFICACION = "ERROR DE VERIFICACION"


#----------------------------------------------------------------------------
#- Abrir la conexion con el servidor PICP. Se utiliza para facilitar la
#- creacion de clientes: se abre el puerto serie, se comprueba la conexion
#- y se verifica que esta cargado el servidor correcto
#----------------------------------------------------------------------------
def Open_session(serialName,logCallback=libStargate.Main.default_log):
  
  #-- Abrir puerto serie 
  try:
    picp = libStargate.Picp.Open (serialName,logCallback)
  except:
    logCallback("Error al abrir puerto serie")
    return None
    
  #---------------------------------------------
  #-- Comprobar la conexion con el servidor   
  #---------------------------------------------
  
  #--- Comprobar si hay conexion
  ok=picp.check_connection();
  if (not ok):
    return None
  
  #-- Comprobar si el servidor al que se accede es el correcto (GENERIC)
  ok=picp.check_server_type();
  if (not ok):
    return None
    
  return picp  

#---------------------------------------------------------------------------#
#-- Funcion de estado por defecto
#---------------------------------------------------------------------------#
def default_stateCallback(op,inc,total):

  #--------------------------------------------------
  #- Comienzo de la grabacion
  #--------------------------------------------------
  if op==libStargate.Picp.WRITING_START:
  
    print "Tamano: %d palabras" % (total)
    sys.stdout.write("\nGRABANDO:\n")
    cad="".join(["." for i in range(50)])
    back="".join(["\b" for i in range(50)])
    
    #-- Imprimir la "barra de estado" con puntos
    sys.stdout.write(cad)
    
    #-- Llevar el cursos a la izquierda
    sys.stdout.write(back)
    sys.stdout.flush()
    
    global contador
    contador=0
   
    return True
   
  
  #------------------------------------------------
  #--  Grabacion de una palabra
  #------------------------------------------------    
  elif op==libStargate.Picp.WRITING_INC:
  
    #-- Contador de asteriscos
    contador+=float(50)/float(total);
    
    #-- Quedarse solo con la parte entera
    na = int(contador)
    contador = contador - na;
    
    #-- Imprimir
    ast = "".join(["*" for i in range(na)])
    sys.stdout.write(ast)
    sys.stdout.flush()
      
    return True
    
  #------------------------------------
  #-- Fin de la grabacion
  #------------------------------------    
  elif op==libStargate.Picp.WRITING_END:
  
    print "OK"
    contador=0
   
    return True
    
  #----------------------------------
  #-- Comienzo de la verificacion
  #----------------------------------    
  elif op==libStargate.Picp.VERIFYING_START:
    
    #-- Actualizar barra de status
    print "VERIFICANDO"
    cad="".join(["." for i in range(50)])
    back="".join(["\b" for i in range(50)])
    
    #-- Imprimir la "barra de estado" con puntos
    sys.stdout.write(cad)
    
    #-- Llevar el cursos a la izquierda
    sys.stdout.write(back)
    sys.stdout.flush()
    
    return True
    
  #-------------------------------------
  #-- Error de verificacion
  #-------------------------------------    
  elif op==libStargate.Picp.VERIFYING_ERROR:
  
    sys.stdout.write("X")
    sys.stdout.flush()
   
    return False
    
  #-----------------------------------
  #--  Verificacion de una palabra    
  #------------------------------------
  elif op==libStargate.Picp.VERIFYING_INC:
    
    #-- Contador de asteriscos
    contador+=float(50)/float(total);
    
    #-- Quedarse solo con la parte entera
    na = int(contador)
    contador = contador - na;
    
    #-- Imprimir
    ast = "".join(["o" for i in range(na)])
    sys.stdout.write(ast)
    sys.stdout.flush()
    
    return True
    
  #--------------------------------------------
  #-- Fin de la verificacion
  #--------------------------------------------    
  elif op==libStargate.Picp.VERIFYING_END:
  
    print "OK"
    
    return True
 
  #-----------------------
  #-- Evento desconocido  
  #-----------------------    
  else:
    return False

#----------------------------------------
#- Clase para la gestion de los errores
#----------------------------------------
class Error (Exception):
  """
  Excepciones producidas en el modulo PICP
  """
  pass


#---------------------------------------------------------------------
#  CLASS STARGATE PICP
#---------------------------------------------------------------------
class Open (libStargate.Main.Stargate):

  #-- Inicializar la clase
  def __init__(self, port, logCallback=libStargate.Main.default_log):
    libStargate.Main.Stargate.__init__(self,port,logCallback)
      
    self.__pc = 0 # Program counter

  #------------------------------------------------
  #-- Obtener si el servidor es el correcto (PICP)
  #------------------------------------------------
  def check_server_type(self):
  
    #-- servidor debe ser del tipo PICP
    ok=libStargate.Main.Stargate.check_server_type(self,PICP_IDENT);
    
    if ok:
      self.log("Servidor correcto");
      self.log("\n");
      return True;
    else:
      self.log("Servidor esperado: %s. \n" % PICP_IDENT);
      self.log("---> SERVIDOR INCORRECTO\n");
      return False;
    

  #-------------------
  #-- Service RESET
  #-------------------
  def Reset_target(self):
  
    #-- If not open, exception
    if not self.serial:
      raise StargateError, 'Serial port not open'
    
    #-- Enviar trama 
    self.serial.write (TRAMA_RESET_CAB)
    
    #-- Esperar respuesta
    reset = self.serial.read(1);
    
    #-- Timeout o dato incorrecto
    if not len (reset) or reset != TRAMA_RRESET_CAB:
    
      #-- Vaciar los buffers de entrada y salida
      self.serial.flushInput ()
      self.serial.flushOutput ()
      raise Error, ERROR_COMUNICACION+' (Reset_target)'
      

    self.__pc = 0x0


  #----------------------------------------
  #-- Service READ DATA (PROGRAM MEMORY)
  #----------------------------------------
  def readDataPM (self):
  
    #-- If not open, exception
    if not self.serial:
      raise StargateError, 'Serial port not open'
    
    #-- Enviar trama 
    self.serial.write (TRAMA_RD_CAB)
    
    #-- Esperar respuesta
    data = self.serial.read (3);
    
    #-- Timeout o dato incorrecto
    if not len (data) or data [0] != TRAMA_RRD_CAB:
    
      #-- Vaciar los buffers de entrada y salida
      self.serial.flushInput ()
      self.serial.flushOutput ()
      raise Error,ERROR_COMUNICACION+' (ReadDataPM)'
    
    return ((ord(data [2]) & 0x3F) << 8) + ord(data [1])


  #---------------------
  #-- Service INC ADDR
  #---------------------
  def incAddr (self):
  
    #-- If not open, exception
    if not self.serial:
      raise StargateError, 'Serial port not open'
    
    #-- Enviar trama 
    self.serial.write (TRAMA_INC_CAB)
    
    #-- Esperar respuesta
    data = self.serial.read (1);
    
    #-- Timeout o dato incorrecto
    if not len (data) or data != TRAMA_RINC_CAB:
    
      #-- Vaciar los buffers de entrada y salida
      self.serial.flushInput ()
      self.serial.flushOutput ()
      raise Error,ERROR_COMUNICACION+' (incAddr)'
      
    #-- Incrementar el contador de programa
    self.__pc += 1


  #-------------------------------------------
  #-- Service LOAD DATA (PROGRAM MEMORY)
  #-------------------------------------------
  def loadDataPM (self, data):
  
    #-- If not open, exception
    if not self.serial:
      raise StargateError, 'Serial port not open'
    
    #-- Enviar trama 
    self.serial.write (TRAMA_DATA_CAB)
    self.serial.write (chr(data & 0xFF))
    self.serial.write (chr(data >> 8 & 0xFF))
    
    #-- Esperar respuesta
    data = self.serial.read (1);
    
    #-- Timeout o dato incorrecto
    if not len (data) or data != TRAMA_RDATA_CAB:
    
      #-- Vaciar los buffers de entrada y salida
      self.serial.flushInput ()
      self.serial.flushOutput ()
      raise Error,ERROR_COMUNICACION+' (LoadDataPM)'

      
  #----------------------
  #-- Service BEGIN CYCLE
  #----------------------
  def beginCycle (self):
  
    #-- If not open, exception
    if not self.serial:
      raise StargateError, 'Serial port not open'
    
    #-- Enviar trama 
    self.serial.write (TRAMA_BEG_CAB)
    
    #-- Esperar respuesta
    data = self.serial.read (1);
    
    #-- Timeout o dato incorrecto
    if not len (data) or data != TRAMA_RBEG_CAB:
    
      #-- Vaciar los buffers de entrada y salida
      self.serial.flushInput ()
      self.serial.flushOutput ()
      raise Error,ERROR_COMUNICACION+' (beginCycle)'    
      

  #----------------------
  #-- Service LOAD CONFIG
  #----------------------
  def loadConfig (self):
  
    #-- If not open, exception
    if not self.serial:
      raise StargateError, 'Serial port not open'
    
    #-- Enviar trama 
    self.serial.write (TRAMA_CONF_CAB)
    
    #-- Esperar respuesta
    reset = self.serial.read (1);
    
    #-- Timeout o dato incorrecto
    if not len (reset) or reset != TRAMA_RCONF_CAB:
    
      #-- Vaciar los buffers de entrada y salida
      self.serial.flushInput ()
      self.serial.flushOutput ()
      raise Error, ERROR_COMUNICACION+' (loadConfig)'

    #-- Actualizar valor del PC
    self.__pc = 0x2000

 
  #------------------------------------------
  #-- Servicio JUMP
  #------------------------------------------  
  def jump(self,salto):
    #-- If not open, exception
    if not self.serial:
      raise StargateError, 'Serial port not open'
    
    #-- Si el salto es '0' no se hace nada
    if salto==0: return
    
    #-- Enviar trama 
    self.serial.write (TRAMA_JUMP_CAB)
    self.serial.write (chr(salto & 0xFF))
    self.serial.write (chr(salto >> 8 & 0xFF))
    
    #-- Actualizar contador de programa
    self.__pc += salto

    #-- Esperar respuesta
    data = self.serial.read (1);

    #-- Timeout o dato incorrecto
    if not len (data) or data != TRAMA_RJUMP_CAB:
    
      #-- Vaciar los buffers de entrada y salida
      self.serial.flushInput ()
      self.serial.flushOutput ()
      raise Error, ERROR_COMUNICACION+' (jump)'

  #-------------------------------------------------------------------
  #-- Lectura de la palabra de configuracion y de la identificacion
  #-- del pic
  #-------------------------------------------------------------------
  def readConfig (self):
    self.Reset_target()
    self.loadConfig()
    self.jump(6)
    id = self.readDataPM ()
    self.incAddr ()
    config = self.readDataPM ()
    return id, config
    
  #----------------------
  #-- Method WRITE CONFIG
  #----------------------
  def writeConfig (self, config):
    self.Reset_target()
    self.loadConfig()
    self.jump(7)
    self.loadDataPM (config)
    self.beginCycle()  

  #--------------------
  #-- Method WRITE DATA
  #--------------------
  #-- Escribir un bloque contiguo de datos
  #-- Devuelve: True si todo ok.
  #--           False si se ha abortado.
  def writeData (self, data, prog_len, stateCallback): # PIC16F87XA version
  
    #-- El bloque debe estar compuesto de sub-bloques de 8 palabras
    #-- Si no es asi se rellenan con ceros hasta completar
    rem = len (data) % 8
    if rem!=0:
      data.extend ((8 - rem) * [0])
    
    #--- Escribir los diferentes sub-bloques de 8 palabras
    i=0;
    for palabra in data:
    
      #-- Primero enviar el dato
      self.loadDataPM (palabra)
      i=i+1; 
      
      #-- Si es el ultimo byte del bloque realizar la grabacion
      if (i % 8 == 0):
        self.beginCycle()
        
      #-- Esta pausa NO es necesaria para el protocolo de grabacion
      #-- Sin embargo con algunos conversores USB/serie la grabacion
      #-- del bootloader no funcionaba bien (pero si con un puerto
      #-- serie nativo). Al introducir esta pausa los problemas se
      #-- solucionaron.
      time.sleep(0.01);  
        
      #-- Incrementar el PC  
      self.incAddr()
      
      #-- Llamar a la funcion de callback
      ok=stateCallback (WRITING_INC, 1, prog_len)
      if not ok:
        return False
    
    #-- Bloque grabado correctamente
    return True;    


  #---------------------------
  #-- Method VERIFY DATA
  #---------------------------
  #-- Verificar un bloque contiguo de datos. Se comprueba que los datos
  #-- situados a partir del PC son iguales que los pasados en data
  def verifyData(self,data, prog_len, stateCallback):
  
    #-- Repetir cada valor del bloque 
    for palabraok in data:
    
      #-- Leer dato de la posicion actual
      palabra=self.readDataPM()
      
      #-- Si lo leido es distinto de lo que hay en el bloque la verificacion
      #-- falla
      if palabra!=palabraok:
        stateCallback(VERIFYING_ERROR, 1, prog_len);
        raise Error, ERROR_VERIFICACION;
      else:  
        ok=stateCallback (VERIFYING_INC, 1, prog_len)  
        if not ok:
          return False;
      
      #-- Incrementar contador de programa
      self.incAddr()
      
    return True
 
  #-----------------------
  #-- Method WRITE PROGRAM
  #-----------------------
  def writeProgram (self, program, stateCallback=default_stateCallback):
    
    #-- Obtener la longitud del programa
    #-- Longitud, una vez en la escritura, otra en la verificación
    programLen = sum (len (block) - 1 for block in program)

    #------------------------------------------------------------------------
    #-- PARTE I: Grabacion de los bloques de codigo en el PIC
    #------------------------------------------------------------------------

    #-- Inicializar la visualización
    ok=stateCallback(WRITING_START, 0, programLen)
    if not ok:
      return False

    #-- Hacer un reset. La direccion actual es la 0
    self.Reset_target()
    
    #-- Repetir para cada bloque de codigo
    for i, block in enumerate (program):
    
      #-- Saltar a la direccion de comienzo del bloque
      self.jump (block [0] - self.__pc) 
      
      #-- Escribir el codigo en el PIC. Si son datos de configuracion no
      if block[0]<0x2000:
        ok=self.writeData (block [1:], programLen, stateCallback)
        if not ok:
          return False
    
    #-- Senalizar fin de grabacion    
    ok=stateCallback(WRITING_END,0,programLen)
    if not ok:
      return False
      
    #------------------------------------------------------------------------
    #-- PARTE II: Verificacion de que se ha grabado bien
    #------------------------------------------------------------------------  
    
    #-- Inicializar la visualización
    ok=stateCallback (VERIFYING_START, 0, programLen)
    if not ok:
      return False
    
    #-- Hacer un reset. La direccion actual es la 0
    self.Reset_target()
    a = 0x00

    status = True

    #-- Repetir para cada bloque de codigo
    for i, block in enumerate (program):
      #-- Saltar a la direccion de comienzo del bloque
      self.jump (block [0] - self.__pc)
      
      #-- Verificar el bloque
      if block[0]<0x2000:
        ok=self.verifyData (block [1:], programLen, stateCallback)
        if not ok:
          return False;
          
        a = block [0] + len (block [1:])

    #-- Fin de la verificacion  
    ok=stateCallback(VERIFYING_END,0,programLen) 
    if not ok:
      return False

    return True
