#! /usr/bin/python
# -*- coding: iso-8859-15 -*-
#---------------------------------------------------------------------------
#--  Modulo Stargate para el movimiento de hasta 8 servos
#--
#--  LICENCIA GPL
#---------------------------------------------------------------------------

# Description: libStargate is a library for comunication to stargates
# Copyright: (C) 2007 by Juan González Gómez
#            (C) 2007 by Rafael Treviño Menéndez
# Authors: Juan González Gómez     <juan@iearobotics.com>
#          Rafael Treviño Menéndez <skasi.7@gmail.com>

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#------------------------------------------------------------------------------


import libStargate.Main


#--- Cadena identificativa del servidor
SERVOS8_IDENT = "SERVOS8"

#-----------------------------------
#------ Cabeceras de las tramas 
#----------------------------------- 
TRAMA_ENA_CAB    = 'E'    #-- Servicio Enable
TRAMA_POS_CAB    = 'W'    #-- Servicio de Posicionamiento

#----------------------------------------------------------------------
#-- Valores de los extremos de los servos para diferentes servidores  
#----------------------------------------------------------------------
EXTREMOS = {0x000: (0,0x89),  # Valores por defecto si no se indica nada

            0x300: (0,0xE6),  # Skypic a 4Mhz
            0x301: (0,0x89),  # Skypic a 20Mhz
            0x101: (0,0xE6),  # CT6811
           }


#-----------------------------------------------
#- Abrir la conexion con el servidor 
#-----------------------------------------------
def Open_session(serialName,logCallback=libStargate.Main.default_log):
  
  #-- Abrir puerto serie 
  try:
    s = libStargate.Servos8.Open (serialName,logCallback)
  except:
    logCallback("Error al abrir puerto serie")
    return None
    
  #---------------------------------------------
  #-- Comprobar la conexion con el servidor   
  #---------------------------------------------
  
  #--- Comprobar si hay conexion
  ok=s.check_connection();
  if (not ok):
    return None
  
  #-- Comprobar si el servidor al que se accede es el correcto (GENERIC)
  ok=s.check_server_type();
  if (not ok):
    return None
    
  return s


#----------------------------------------------------------------------------
#-              Clase para la gestion de los errores
#----------------------------------------------------------------------------
class Error (Exception):
  """
  Excepciones producidas en el modulo Stargate
  """
  pass
  
#---------------------------------------------------------------------
#  CLASE STARGATE SERVOS8
#---------------------------------------------------------------------
class Open(libStargate.Main.Stargate):

  #-- Inicializar la clase
  def __init__(self, port, logCallback=libStargate.Main.default_log):
    libStargate.Main.Stargate.__init__(self,port,logCallback)
    
    #-- Inicializar los limites superiores e inferiorese por defecto
    (self.__li, self.__ls) = EXTREMOS[0x00]
    
    
  #----------------------------------------------------
  #-- Obtener si el servidor es el correcto (SERVOS8)
  #----------------------------------------------------
  def check_server_type(self):
  
    #-- servidor debe ser del tipo correcto
    ok=libStargate.Main.Stargate.check_server_type(self,SERVOS8_IDENT);
    
    if ok:
      self.log("Servidor correcto");
      self.log("\n");
    else:
      self.log("Servidor esperado: %s. \n" % SERVOS8_IDENT);
      self.log("---> SERVIDOR INCORRECTO\n");
      return False;
      
    #--------------------------------------------------
    #-- Adaptar los extremos segun el tipo de servidor
    #--------------------------------------------------
    #-- Leer la identificcion
    (iserv, im, id) = libStargate.Main.Stargate.id_bytes(self)   
    
    #-- Segun el tipo de servidor detectado, modificar los limites 
    #-- supeior e inferior de los valores que se pueden enviar
    placa = im<<4 | id>>4
    (self.__li, self.__ls) = EXTREMOS[placa] 
    
    return True

  #-------------------
  #-- Servicio ENABLE
  #-------------------
  def enable(self, mask):
  
    #-- Si puerto serie no abierto retornar
    if self.serial==None: return 0
    
    #-- Enviar trama 
    self.serial.write(TRAMA_ENA_CAB)
    self.serial.write(chr(mask&0xFF))
    
    return 1
    
  #------------------------------------------------------------------
  #-- Servicio POS
  #-- Se especifica el angulo del servo en grados, entre -90 y 90
  #------------------------------------------------------------------
  def pos(self, servo, pos):
  
    #-- Si puerto serie no abierto retornar
    if self.serial==None: return 0
    
    #-- Convertir de grados al rango del servo
    #-- Esta conversion depende de los valores limites (superior e inferior)
    #-- implementados en el servidor
    pos = int((self.__ls - self.__li + 1)/2  + (self.__ls - self.__li)*pos/180)
    
    #-- Invocar el servicio de posicionamiento de bajo nivel
    self.pos_raw(servo,pos)
    
    return 1
  
  #---------------------------------------
  #- Metodo GET. Equivalente a un POS
  #---------------------------------------
  def __setitem__(self, servo, pos): 
    return self.pos(servo,pos)  
  
  #--------------------------------------------------------------------
  #-- Servicio POS de bajo nivel. Se especifica la posicion como un 
  #-- byte entre 0 y 255. Dependiendo del servidor, el servo se 
  #-- situara en un angulo u otro
  #--------------------------------------------------------------------
  def pos_raw(self, servo, pos):
    #-- Si puerto serie no abierto retornar
    if self.serial==None: return 0
    
    #-- Enviar trama 
    self.serial.write(TRAMA_POS_CAB)
    self.serial.write(chr(servo&0xFF))
    self.serial.write(chr(pos&0xFF))
    
    return 1
