/*******************************************************/
/* Libreria SkyRGB                            Oct-2008 */
/*-----------------------------------------------------*/
/* Libreria para el control de la tarjeta SkyRGB       */
/*-----------------------------------------------------*/
/* Andres Prieto-Moreno                                */
/*******************************************************/

//-- Especificar el pic a emplear
#include <pic16f876a.h>
#include "libRGB.h"

//-- variables globales

unsigned char flag;
unsigned char vel;	


/****************************
	FUNCIONES AUXILIARES
****************************/

void pausa_1ms() {
	unsigned int tmp;
	
	tmp=500;
	while (tmp>0) {
		tmp--;
	}
}

/****************************
      LIBRERIA LIBI2C
*****************************/

// Leer Byte por el I2C
char i2c_ReadByte() {
	SSPCON2=SSPCON2 | 0x08;           // Activar el RCEN
	while  ((SSPCON2 & 0x08)!=0 ) ;   // esperar hasta que RCEN = 0  
	if ( (SSPSTAT & 0x01) == 0x01 ) {
		return SSPBUF;                // Leo el registro de datos
	}
	else {
		return 0;
	}
}

// verifica ACK
// devuelve 0 si no hemos recibido nada del Slave
// devuelve 1 si ACK
unsigned char i2c_CheckACK() {
	if ( (SSPCON2 & 0x40) ==0 ) 
		return 0;   // correcto (lo he recibido )
	else 
		return 1;	// incorrecto (no lo he recibido)
}

// Envia Ack
void i2c_SendAck() {
	SSPCON2= SSPCON2 & 0xDF;  // Establece un ACK
	SSPCON2= SSPCON2 | 0x10;  // Lo envia
}

// Envia Nack para finalizar la recepecion
void i2c_SendNack() {
	SSPCON2= SSPCON2 | 0x20;  // Establece un NACK
	SSPCON2= SSPCON2 | 0x10;  // Lo envia
}

// Envia un Dato por el Bus I2C
// Entrada el dato a enviar
unsigned char i2c_SendByte(char dato) {
	SSPBUF=dato;
	
	if ( (SSPCON & 0x80)==0x80 ) {  // WCOL activado (colision) ? 
		return -1;
	} else {	
		while (SSPSTAT&0x01) ;// mientras que el buffer esta lleno BF=1 espero  
		return 0;
	}
}

// Envia la condicion de STOP
void i2c_SendStop() {
	SSPCON2=SSPCON2 | 0x04; // send stop bit (PEN = 1)
	while(PEN);			
}

// Envia la condicion de START
void i2c_SendStart() {
	SSPCON2=SSPCON2 |0x01;   // send start bit  SEN=1
	while(SEN);	// and wait for it to clear 
}

void i2c_CheckIdle() {
	while ( (SSPCON2 & 0x1F ) != 0 );   // esperamos a que termine cualquier operacion
	PIR1=PIR1 & 0xF7;                   // Limpio el Flag de eventos
}

// Repeated start from master
void i2c_RStart() {
	SSPCON2=SSPCON2 |0x02;  // RSEN=1;
	while ( RSEN );
}

/**********************************
 FUNCIONES DE ALTO NIVEL  I2C
***********************************/

/* Configurar I2C
	Configuramos el I2C como Master
	y a una velocidad de 1Mhz
*/
void i2c_configure() {
	// configuramos SCL y SDA como pines de entrada
	TRISC=TRISC | 0x18;  // 1 entrada / 0 salida  
	
	SSPSTAT=0x0;  
	SSPSTAT=SSPSTAT | 0x80;  // SMP=1 Slew rate disable for 100Kh
							 // CKE=0 (mono maestro I2C) y UA=0 (7 bits)	
	
	SSPCON=0x08;	 		 // I2C master mode (uso la formula del reloj con SSPAD)  
	SSPCON=SSPCON | 0x20;	 // enable I2C  SSPEN=1
	
	SSPCON2=0x00;   // no se usa por ahora
	// velocidad = FOSC / ( 4 * (sspad + 1) )
	// Fosc=20Mhz
	SSPADD=49;  // 49->100Khz 
	
	// Limpia Flags de eventos
	PIR1=0;  //Limpia flag de eventos SSP 
}


// writes data to I2c
void Write_RGB(unsigned char reg, unsigned char rgb )
{
	// El esclavo no enva ACK, por eso ponemos una pausa.
	
	SSPIF=0;
	
	// 1 Envia Bit de Start
	i2c_CheckIdle();
	i2c_SendStart();	

	// 2 Envio la direccion del modulo
	i2c_SendByte(RGB_W);
	pausa_1ms();  // espero a que llegue el ACK
	SSPCON2 = SSPCON2 & 0xBF;  // borro ACK	
	
	// 3 mando el registro sobre el que voy a actuar
	i2c_SendByte(reg); 
	pausa_1ms();  // espero a que llegue el ACK
	SSPCON2 = SSPCON2 & 0xBF;  // borro ACK
			
	// 4 mando el dato (byte mas significativo)
	i2c_SendByte(rgb); 
	pausa_1ms();  // espero a que llegue el ACK
	SSPCON2 = SSPCON2 & 0xBF;  // borro ACK
		
	
	// 5 termino el envio
	i2c_SendStop();
	i2c_CheckIdle();  
}
	
//******************************************
//* Funciones alto nivel SKYRGB            *
//******************************************

// Inicializa la placa
void RGB_init() {
	i2c_configure();
}


// Pone un color RGB en la placa
// red   - intensidad de rojo
// green - intensidad de verde
// blue  - intensidad de azul
// velocidad - velocidad de la transicion
void RGB(unsigned char red, unsigned char green, unsigned char blue, unsigned char velocidad) {
	Write_RGB(11,velocidad);
	Write_RGB(6,velocidad);	
	Write_RGB(1,velocidad);
	
	Write_RGB(10,blue);
	Write_RGB(5,green);
	Write_RGB(0,red);
	
}	

// Enciende el cubo con color Azul
void Blue(unsigned char intensidad, unsigned char velocidad) {
	RGB(0,0,intensidad, velocidad);
} 

// Enciende el cubo en color rojo
void Red(unsigned char intensidad, unsigned char velocidad) {
	RGB(intensidad,0,0, velocidad);
}

// Enciende el cubo en color verde
void Green(unsigned char intensidad, unsigned char velocidad) {
	RGB(0,intensidad, 0, velocidad);
}

// Hace un efecto de parpadeo
// Tipo = 0 Lo desactiva
// Tipo = 1 Activa latido Rojo
// Tipo = 2 Activa latido Verde
// Tipo = 3 Activa latido Azul
void Latido(unsigned char tipo, unsigned char min, unsigned char max, unsigned char velocidad) {
	
	if (tipo==0) {
		Write_RGB(2 , 0);  // patron
		Write_RGB(7 , 0);  // patron 
		Write_RGB(12, 0);  // patron
		RGB(0,0,0,0);
	}
	else {
		Write_RGB(3 ,max);
		Write_RGB(4 ,min);
		Write_RGB(8 ,max);
		Write_RGB(9 ,min);
		Write_RGB(13,max);
		Write_RGB(14,min);
		
		if (tipo==1) {
			Red(250,velocidad);
			Write_RGB(2 , 1);
		} 

		else if (tipo==2) {
			Green(250,velocidad);
			Write_RGB(7 , 1);
		} 

		else if (tipo==3) {
			Blue(250,velocidad);
			Write_RGB(12, 1);
		}
	}
	
}


