/***********************************************************************/
/* SIN.C       (C) Microbotica, S.L         Diciembre 1998.            */
/*---------------------------------------------------------------------*/
/* PROYECTO: PC-BOT                                                    */
/* SUBPROYECTO: Interprete para PC-BOT                                 */
/* DESCRIPCION: Analizador sintactico.                                 */
/***********************************************************************/

#define CINDENT  2   /* Numero de caracteres de indentado al imprimir */

#include "stdio.h"
#include "dos.h"
#include "conio.h"
#include "ctype.h"
#include "io.h"
#include "stdlib.h"
#include <alloc.h>


#include "lex.h"
#include "sin.h"

static struct tipo_token sig_token;
static int nsinerror=0;
static int nlinea=1;

/* Errores que se pueden producir en el analizador sintactico */
static char sin_error[26][80]={
  {"Error lexico"},                           /* 1 */
  {"No hay memoria suficiente"},              /* 2 */
  {"Instruccion desconocida"},                /* 3 */
  {"Se esperaba '('"},                        /* 4 */ 
  {"Sentencia SALIDA: Argumento erroneo"},    /* 5 */ 
  {"Sentencia SALIDA: Se esperaba ','"},      /* 6 */ 
  {"Sentencia SALIDA: Se esperaba ')'"},      /* 7 */
  {"Sentencia SALIDA: Se esperaba ';'"},      /* 8 */
  {"Sentencia ESPERA: Se esperaba '('"},      /* 9 */ 
  {"Sentencia ESPERA: Argumento erroneo"},    /* 10 */
  {"Sentencia ESPERA: Se esperaba ')'"},      /* 11 */
  {"Sentencia ESPERA: Se esperaba ';'"},      /* 12 */
  {"Sentencia MOTOR: Se esperaba '('"},      /* 13 */
  {"Sentencia MOTOR: Argumento erroneo"},    /* 14 */
  {"Sentencia MOTOR: Se esperaba ','"},      /* 15 */
  {"Sentencia MOTOR: Se esperaba ')'"},      /* 16 */
  {"Sentencia MOTOR: Se esperaba ';'"},      /* 17 */ 
  {"Sentencia IF: No hay condicion"},        /* 18 */
  {"Sentencia ENTRADA: Se esperaba '('"},    /* 19 */
 
  {"Sentencia ENTRADA: Argumento erroneo"},  /* 20 */
  {"Sentencia ENTRADA: Se esperaba ','"},    /* 21 */
  {"Sentencia ENTRADA: Se esperaba ')'"},    /* 22 */
  {"Blocke no finalizado, se esperaba '{'"}, /* 23 */ 
  {"Sentencia REPITE: Se esperaba '('"},     /* 24 */
  {"Sentencia REPITE: Argumento erroneo"},   /* 25 */
  {"Sentencia REPITE: Se esperaba ')'"},     /* 26 */
};


lista_sentencias *crear_lista_sentencias(int,int *);
void print_lista_sentencias(lista_sentencias *,int);
void *crear_sentencia(int,enum tsentencia *,int *);

/*******************************************************************/
/*             C   O   D   I   G   O                               */
/*******************************************************************/

char *getsin_error()
/*****************************************************************/
/* Devolver la cadena de error del ultimo error producido.       */
/*****************************************************************/
{
  if (nsinerror==0) return 0;  /* No hay errores */
  return (char *)&sin_error[nsinerror-1];
}

int get_num_sin_error()
/***********************************/
/*  Devolver el numero de error    */
/***********************************/
{
  return nsinerror;
}

int get_linea_error()
/*************************************************************/
/* Obtener el numero de linea donde se ha producido el error */
/*************************************************************/
{
  return nlinea;
}

int obtener_token(struct tipo_token *t,int fd)
/*********************************/
/* Leer el siguiente token.      */
/* Si hay error se devuelve un 0 */
/*********************************/
{
  struct tipo_token token;
  
  token.tipo=sig_token.tipo;
  token.valor=sig_token.valor;
  
  if (!leer_token(&sig_token,fd)) {
    nsinerror=1;                     /* Error lexico */
    return 0;
  }
  t->tipo=token.tipo;
  t->valor=token.valor;
  return 1;
}

int es_sentencia(struct tipo_token *t)
/***************************************************************/
/* Se devuelve 1 si el token se corresponde con el de una      */
/* sentencia.                                                  */
/***************************************************************/
{
  if (t->tipo==t_repite || t->tipo==t_if || t->tipo==t_espera ||
      t->tipo==t_motor  || t->tipo==t_salida) return 1;
  else return 0;
}

void eliminar_lf(int fd,int *error)
/******************************/
/* Eliminar los tokens de LF. */
/******************************/
{
  struct tipo_token t;
  
  *error=0;
  while (sig_token.tipo==t_fin_linea) {
    if (!obtener_token(&t,fd)) {
      nsinerror=1;    /* Error lexico */
      *error=1;
      return;
    }
    nlinea++;
  }
  return;
}

void eliminar_linea(int fd,int *error)
/*******************************************************************/
/* Eliminar todos los tokens hasta llegar a uno LF o FIN fichero   */
/*******************************************************************/
{
  struct tipo_token t;
  
  *error=0;
  while (sig_token.tipo!=t_fin_linea && sig_token.tipo!=t_fin_fichero) {
    if (!obtener_token(&t,fd)) {
      nsinerror=1;    /* Error lexico */
      *error=1;
      return;
    }
  }
}

enum tsentencia obtener_sentencia(struct tipo_token *t)
/********************************************************/
/* Devolver el tipo de sentencia                        */
/********************************************************/
{
  switch(t->tipo) {
    case t_repite : return s_repite;
    case t_if     : return s_if;
    case t_espera : return s_espera;
    case t_motor  : return s_motor;
    case t_salida : return s_salida;
  }
  
  return s_nula;  /* Esto no deberia ocurrir */
}

int token_esperado(int fd,enum ttoken tes,struct tipo_token *t,
                   int nerror, int *error)
/**************************************************************/
/*  La funcion devuelve 1 si se ha recibido el token esperado */
/*  En caso contrario se devuelve 0, se actualiza la variable */
/*  global nsinerror con nerror y se activa error.            */
/*  Si no hay error se devuelve el token leido.               */
/**************************************************************/
{
  
  if (sig_token.tipo!=tes) { 
    *error=1;                /* Se ha recibido token distinto del */
    nsinerror=nerror;        /* que se esperaba                   */
    *error=1;
    return 0;
  }
  /* Leer el token  */
  if (!obtener_token(t,fd)) {
      nsinerror=1;    /* Error lexico */
      *error=1;
      return 0;
  }
  return 1;
}

void filtrar_tokens_sentencia(int fd,int *error)
/*******************************************************/
/* Leer todos los tokens innecesarios hasta llegar a   */
/* un token correspondiente a una sentencia o a '}'    */
/* Los tokens innecesarios son LF y sentencias NOTA    */
/*******************************************************/
{

  while (!es_sentencia(&sig_token) && sig_token.tipo!=t_cierra_block) {
    switch(sig_token.tipo) {
      case t_fin_linea : eliminar_lf(fd,error); 
                         if (*error) return;
                         break;
      case t_nota : eliminar_linea(fd,error);
                    if (*error) return;
                    break;
      case t_fin_fichero: return;
      default: nsinerror=3;
               *error=1;
               return;
    }
  }
}


tipo_sespera *crear_sespera(int fd,int *error)
/*********************************************************/
/* Crear nodo del tipo espera y devolver puntero         */
/*********************************************************/
{
  tipo_sespera *espera;
  struct tipo_token t;
  int seg;   /* Segundos de espera */
  
  *error=0;
  
  if (!token_esperado(fd,t_abre_par,&t,9,error)) return NULL;
  if (!token_esperado(fd,t_numero,&t,10,error))  return NULL;
  seg=(int)t.valor; 
  if (!token_esperado(fd,t_cierra_par,&t,11,error)) return NULL;
  if (!token_esperado(fd,t_puntycoma,&t,12,error)) return NULL;
 
  /* Sentencia ESPERA es sintacticamente correcta */
  
  /* Crear espacio para el nodo */
  espera=(tipo_sespera *) malloc (sizeof(tipo_sespera));
  if (espera==NULL) {   
    nsinerror=2;    /* Error: no hay espacio suficiente  */
    *error=1;
    return NULL;
  }
  espera->seg=seg;       /* Segundos    */
  espera->nlinea=nlinea;

  /* Devolver puntero */
  return espera; 
}

tipo_ssalida *crear_ssalida(int fd,int *error)
/*********************************************************/
/* Crear nodo del tipo salida y devolver puntero         */
/*********************************************************/
{
  tipo_ssalida *salida;
  struct tipo_token t;
  int nsal;                 /* Numero de salida    */
  enum tident_io estado;    /* Estado de la salida */
  
  *error=0;

  if (!token_esperado(fd,t_abre_par,&t,4,error)) return NULL;
  if (!token_esperado(fd,t_numero,&t,5,error)) return NULL;
  nsal=(int)t.valor;  /* Argumento 1: numero de salida */
  if (!token_esperado(fd,t_coma,&t,6,error)) return NULL;
  
  /* El siguiente token solo puede ser t_on o t_off */
  if (sig_token.tipo!=t_on && sig_token.tipo!=t_off) {
    *error=1;
    nsinerror=5;   /* Argumento erroneo */
    *error=1;
    return NULL;
  }
  
  /* Leer el token t_on o t_off */
  if (!obtener_token(&t,fd)) {
      nsinerror=1;    /* Error lexico */
      *error=1;
      return NULL;
  }
  
  /* Argumento 2: Estado de la salida */
  if (t.tipo==t_on) estado=i_on;
  else if (t.tipo==t_off) estado=i_off;
  
  if (!token_esperado(fd,t_cierra_par,&t,7,error)) return NULL;
  if (!token_esperado(fd,t_puntycoma,&t,8,error)) return NULL;
  
  /* Sentencia SALIDA es sintacticamente correcta */
  
  /* Crear espacio para el nodo */
  salida=(tipo_ssalida *) malloc (sizeof(tipo_ssalida));
  if (salida==NULL) {   
    nsinerror=2;    /* Error: no hay espacio suficiente  */
    *error=1;
    return NULL;
  }
  salida->nsal=nsal;     /* Numero de salida    */
  salida->estado=estado; /* Estado de la salida */
  salida->nlinea=nlinea;

  /* Devolver puntero */
  return salida;
}

tipo_smotor *crear_smotor(int fd,int *error)
/****************************************************/
/* Crear nodo del tipo motor y devolver puntero     */
/****************************************************/
{
  tipo_smotor *motor;
  struct tipo_token t;
  int nmotor;               /* Numero de motor     */
  enum tident_mot estado;   /* Estado del motor    */
  
  *error=0;
  
  if (!token_esperado(fd,t_abre_par,&t,13,error)) return NULL;
  if (!token_esperado(fd,t_numero,&t,14,error)) return NULL;
  nmotor=(int)t.valor;  /* Argumento 1: numero de motor */
  if (!token_esperado(fd,t_coma,&t,15,error)) return NULL;
  
  /* El siguiente token solo puede ser t_off,t_izquierda, t_derecha */
  if (sig_token.tipo!=t_off && sig_token.tipo!=t_izquierda &&
      sig_token.tipo!=t_derecha) {
    *error=1;
    nsinerror=14;   /* Argumento erroneo */
    return NULL;
  }
  
  /* Leer el segundo argumento */
  if (!obtener_token(&t,fd)) {
      nsinerror=1;    /* Error lexico */
      *error=1;
      return NULL;
  }
  
  /* Argumento 2: Estado de los motores */
  if (t.tipo==t_off) estado=m_off;
  else if (t.tipo==t_derecha) estado=m_derecha;
  else if (t.tipo==t_izquierda) estado=m_izquierda;
  
  if (!token_esperado(fd,t_cierra_par,&t,16,error)) return NULL;
  if (!token_esperado(fd,t_puntycoma,&t,17,error)) return NULL;
  
  /* Sentencia MOTOR es sintacticamente correcta */
  
  /* Crear espacio para el nodo */
  motor=(tipo_smotor *) malloc (sizeof(tipo_smotor));
  if (motor==NULL) {   
    nsinerror=2;    /* Error: no hay espacio suficiente  */
    *error=1;
    return NULL;
  }
  motor->nmotor=nmotor;  /* Numero de motor   */
  motor->estado=estado;  /* Estado del motor  */
  motor->nlinea=nlinea;

  /* Devolver puntero */
  return motor;
}

tipo_centrada *crear_centrada(int fd,int *error)
/**************************************************/
/* Crear nodo del tipo entrada y devolver puntero */
/**************************************************/
{
  tipo_centrada *ent;
  struct tipo_token t;
  int  nent;                /* Numero de la Entrada */
  enum tident_io estado;    /* Estado de la entrada */
  
  *error=0;
  
  if (!token_esperado(fd,t_entrada,&t,18,error)) return NULL;
  if (!token_esperado(fd,t_abre_par,&t,19,error)) return NULL;
  if (!token_esperado(fd,t_numero,&t,20,error)) return NULL;
  nent=(int)t.valor;  /* Argumento 1: numero de salida */
  if (!token_esperado(fd,t_coma,&t,21,error)) return NULL;
  
  /* El siguiente token solo puede ser t_on o t_off */
  if (sig_token.tipo!=t_on && sig_token.tipo!=t_off) {
    *error=1;
    nsinerror=20;   /* Argumento erroneo */
    return NULL;
  }
  
  /* Leer el token t_on o t_off */
  if (!obtener_token(&t,fd)) {
      nsinerror=1;    /* Error lexico */
      *error=1;
      return NULL;
  }
  
  /* Argumento 2: Estado de la salida */
  if (t.tipo==t_on) estado=i_on;
  else if (t.tipo==t_off) estado=i_off;
  
  if (!token_esperado(fd,t_cierra_par,&t,22,error)) return NULL;
  
  /* Condicion ENTRADA sintacticamente correcta */
  
  /* Crear espacio para el nodo */
  ent=(tipo_centrada *) malloc (sizeof(tipo_centrada));
  if (ent==NULL) {   
    nsinerror=2;    /* Error: no hay espacio suficiente  */
    *error=1;
    return NULL;
  }
  ent->nent=nent;        /* Numero de entrada    */
  ent->estado=estado;    /* Estado de la entrada */
  ent->nlinea=nlinea;

  /* Devolver puntero */
  return ent;
}

tipo_sif *crear_sif(int fd, int *error)
/*************************************************/
/* Crear nodo del tipo if y devolver puntero     */
/*************************************************/
{
  tipo_sif *pif;
  struct tipo_token t;
  int n;
  tipo_centrada *condicion;  /* Condicion del if */
  lista_sentencias *ss;      /* Puntero a lista de sentencias */
  void *s;
  enum tsentencia ts;        /* Tipo de sentencia      */
  
  *error=0;

  /* Leer la condicion */
  condicion=(tipo_centrada *)crear_centrada(fd,error);
  if (*error) return NULL;

  /* Crear bloque */
  if (sig_token.tipo==t_abre_block) {
    if (!token_esperado(fd,t_abre_block,&t,0,error)) return NULL;
    ss=(lista_sentencias *)crear_lista_sentencias(fd,error);
    if (*error) return NULL;
    if (!token_esperado(fd,t_cierra_block,&t,23,error)) return NULL;
  }
  else { /* Crear lista con 1 sentencia */
    s=crear_sentencia(fd,&ts,error);
    if (*error) return NULL;
    ss=(lista_sentencias *) malloc (sizeof(lista_sentencias));
    if (ss==NULL) {   
      nsinerror=2;    /* Error: no hay espacio suficiente  */
      *error=1;
      return NULL;
    }
    ss->s=s;
    ss->ts=ts;
    ss->ss=NULL;
  }

  /* Crear espacio para el nodo */
  pif=(tipo_sif *) malloc (sizeof(tipo_sif));
  if (pif==NULL) {   
      nsinerror=2;    /* Error: no hay espacio suficiente  */
      *error=1;
      return NULL;
  }
  pif->condicion=condicion;
  pif->nlinea=nlinea;
  pif->ss=ss;
  
  return pif;
}

tipo_srepite *crear_srepite(int fd,int *error)
/*******************************************************/
/* Crear nodo del tipo repite y devolver puntero       */
/*******************************************************/
{
  tipo_srepite *rep;
  struct tipo_token t;
  lista_sentencias *ss;      /* Puntero a lista de sentencias */
  int repeticion;            /* Valor de repeticion           */
  void *s;
  enum tsentencia ts;        /* Tipo de sentencia      */
    
  *error=0;

  if (!token_esperado(fd,t_abre_par,&t,24,error)) return NULL;
  if (!token_esperado(fd,t_numero,&t,25,error))  return NULL;
  repeticion=(int)t.valor; 
  if (!token_esperado(fd,t_cierra_par,&t,26,error)) return NULL;
 
  /* Crear bloque */
  if (sig_token.tipo==t_abre_block) {
    if (!token_esperado(fd,t_abre_block,&t,0,error)) return NULL;
    ss=(lista_sentencias *)crear_lista_sentencias(fd,error);
    if (*error) return NULL;
    if (!token_esperado(fd,t_cierra_block,&t,23,error)) return NULL;
  }
  else { /* Crear lista con 1 sentencia */
    s=crear_sentencia(fd,&ts,error);
    if (*error) return NULL;
    ss=(lista_sentencias *) malloc (sizeof(lista_sentencias));
    if (ss==NULL) {   
      nsinerror=2;    /* Error: no hay espacio suficiente  */
      *error=1;
      return NULL;
    }
    ss->s=s;
    ss->ts=ts;
    ss->ss=NULL;
  }
 
  /* Sentencia REPITE es sintacticamente correcta */
  /* Crear espacio para el nodo */
  rep=(tipo_srepite *) malloc (sizeof(tipo_srepite));
  if (rep==NULL) {   
    nsinerror=2;    /* Error: no hay espacio suficiente  */
    *error=1;
    return NULL;
  }
  rep->repeticion=repeticion; 
  rep->quedan=repeticion;
  rep->nlinea=nlinea;
  rep->ss=ss;

  /* Devolver puntero */
  return rep;   
}

void *crear_sentencia(int fd,enum tsentencia *ts,int *error)
/********************************************************/
/*  Crear un nodo de sentencia y devolver el puntero    */
/* Se devuelve en ts el tipo de sentencia creada        */
/* Si hay algun error se devuelve NULL                  */
/* Si se detecta '}' se retorna y no se informa de error*/
/********************************************************/
{
  void *s;
  struct tipo_token t;
    
 /* Inicialmente no hay error */
  *error=0;
 
  /* Eliminar LF's e intrucciones NOTA */
  filtrar_tokens_sentencia(fd,error);
  if (*error) return NULL;
  
  if (sig_token.tipo==t_cierra_block) return NULL;
  if (sig_token.tipo==t_fin_fichero)  return NULL;
  
  /* Obtener el token de la sentencia */
  if (!obtener_token(&t,fd)) {
      nsinerror=1;    /* Error lexico */
      *error=1;
      return NULL;
  }  
    
  /* Obtener el tipo de sentencia */
  *ts=obtener_sentencia(&t);
  
  /* Puntero a la instruccion */
  switch(*ts) {
    case s_repite: s=(tipo_srepite *)crear_srepite(fd,error); break;
    case s_salida: s=(tipo_ssalida *)crear_ssalida(fd,error); break;
    case s_espera: s=(tipo_sespera *)crear_sespera(fd,error); break;
    case s_motor : s=(tipo_smotor  *)crear_smotor(fd,error);  break;
    case s_if    : s=(tipo_sif     *)crear_sif(fd,error);     break;
    default      : s=NULL;
  }
  
  /* Devolver puntero de la instruccion creada */
  return s;
}

lista_sentencias *crear_lista_sentencias(int fd,int *error)
/*****************************************************************/
/* Crear la lista de sentencias y devolver el puntero.           */
/* Si hay algun error, entonces error=1, sino 0.                 */
/*****************************************************************/
{
  enum tsentencia ts;    /* Tipo de sentencia      */
  void *s;               /* Puntero a la sentencia */
  lista_sentencias *ss;  /* Puntero a las siguientes sentencias */
  lista_sentencias *ls;
  struct tipo_token t;

  /* Inicialmente no hay error */
  *error=0;

  s=crear_sentencia(fd,&ts,error);
  if (*error) return NULL;

  if (s==NULL) return NULL;  /* No hay mas sentencias */
  
  ss=crear_lista_sentencias(fd,error);
  if (*error) return NULL;
  
  /* Asignar espacio al nuevo elemento de la lista */
  ls=(lista_sentencias *) malloc (sizeof(lista_sentencias));
  if (ls==NULL) {   
    nsinerror=2;    /* Error: no hay espacio suficiente  */
    *error=1;
    return NULL;
  }
  
  ls->ts=ts;
  ls->s=s;
  ls->ss=ss;
  return ls;
}

lista_sentencias *crear_ast(int fd,int *error)
/******************************************************************/
/* Crear el arbol abstracto de datos. Se devuelve un puntero al   */
/* comienzo. En caso de que se produzca un error se devuelve      */
/* el valor 1 en el parametro error.                              */
/******************************************************************/
{
  nsinerror=0;
  nlinea=1;

  reset_lex();  /* Inicializar variables del lexico */
  
  /* Leer el primer token */
  if (!leer_token(&sig_token,fd)) {
    *error=1;    /* Error */
    nsinerror=1; /* Error del analizador lexico */
    return NULL;
  }
  return crear_lista_sentencias(fd,error);
}


/***************************************************************/
/*  IMPRESION DEL AST                                          */
/***************************************************************/

void espacios(int num)
/*****************************/
/* Imprimir num espacios     */
/*****************************/
{
  int i;
  
  for (i=0; i<num; i++) printf (" ");
}


void print_estado_io(enum tident_io estado)
{
  switch(estado) {
    case i_on:  printf ("ON");  break;
    case i_off: printf ("OFF"); break;
  }
}

void print_estado_mot(enum tident_mot estado)
{
  switch(estado) {
    case m_off      : printf ("OFF");       break;
    case m_derecha  : printf ("DERECHA");   break;
    case m_izquierda: printf ("IZQUIERDA"); break;
  }
}

void print_ssalida(tipo_ssalida *s,int indent)
/**************************************/
/* Imprimir una sentencia de salida   */
/**************************************/
{
  espacios(indent);
  printf ("SALIDA(");
  if (s==NULL) {
    printf ("null\n");
  }
  else {
    printf ("%u,",s->nsal);
    print_estado_io(s->estado); 
    printf (");\n");
  }
}

void print_sespera(tipo_sespera *esp,int indent)
/********************************/
/* Imprimir sentencia espera    */
/********************************/
{
  espacios(indent);
  printf ("ESPERA(");
  if (esp==NULL) {
    printf ("null\n");
  }
  else {
    printf ("%u",esp->seg);
    printf (");\n");
  }
}

void print_smotor(tipo_smotor *mot,int indent)
/********************************/
/* Imprimir sentencia motor     */
/********************************/
{
  
  espacios(indent);
  printf ("MOTOR(");
  if (mot==NULL) {
    printf ("null\n");
  }
  else {
    printf ("%u,",mot->nmotor);
    print_estado_mot(mot->estado);
    printf (");\n");
  }
}

void print_centrada(tipo_centrada *ent)
/************************************/
/* Imprimir condicion de entrada    */
/************************************/
{
  if (ent==NULL) {
    printf ("NULL ");
    return;
  }
  printf ("ENTRADA(");
  printf ("%u,",ent->nent);
  print_estado_io(ent->estado);
  printf (") ");
}

void print_sif(tipo_sif *sif, int indent)
/*********************************/
/* Imprimir sentencia if         */
/*********************************/
{
  lista_sentencias *ss;

  espacios(indent);

  printf ("IF ");
  if (sif==NULL) {
    printf ("(null)\n");
    return;
  }
  print_centrada(sif->condicion);
  
  /* Acceder a las sentencias de la condicion if */
  ss=sif->ss;
  
  if (ss==NULL) {
    printf ("(null)\n");
    return;
  }
  
  if (ss->ss==NULL) {  /* Solo hay una instruccion */
    print_lista_sentencias(ss,indent);
    return;
  }
  
  printf ("{\n");
  print_lista_sentencias(ss,indent+CINDENT);
  espacios(indent);
  printf ("}\n");
}

void print_srepite(tipo_srepite *srep,int indent)
/********************************/
/* Imprimir sentencia repite    */
/********************************/
{
  lista_sentencias *ss;
  int nrep;

  espacios(indent);

  printf ("REPITE(");
  if (srep==NULL) {
    printf ("null)\n");
    return;
  }
  printf ("%u) ",srep->repeticion);
  
  /* Acceder a las sentencias de repite */
  ss=srep->ss;
  if (ss==NULL) {
    printf ("(null)\n");
    return;
  }
  if (ss->ss==NULL) {  /* Solo hay una instruccion */
    print_lista_sentencias(ss,indent);
    return;
  }
  printf ("{\n");
  print_lista_sentencias(ss,indent+CINDENT);
  espacios(indent);
  printf ("}\n");
}

void print_sentencia(lista_sentencias *ls,int indent)
/*****************************************************************/
/* Segun el tipo de sentencia llamar a la rutina correspondiente */
/*****************************************************************/
{
  switch(ls->ts) {
    case s_repite : print_srepite((tipo_srepite *)ls->s,indent);  break;
    case s_if     : print_sif((tipo_sif *)ls->s,indent);      break;
    case s_espera : print_sespera((tipo_sespera *)ls->s,indent);  break;
    case s_motor  : print_smotor((tipo_smotor *)ls->s,indent);   break;
    case s_salida : print_ssalida((tipo_ssalida *)ls->s,indent);  break;
  }
}

void print_lista_sentencias(lista_sentencias *ls, int indent)
/******************************************/
/* Imprimir una lista de sentencias       */
/******************************************/
{
  if (ls==NULL) return;
  print_sentencia(ls,indent);
  print_lista_sentencias(ls->ss,indent);
}

void print_ast(lista_sentencias *ls)
/******************************************/
/* Imprimir en pantalla el ast pasado     */
/******************************************/
{
  
  print_lista_sentencias(ls,0);
  printf ("FIN PROGRAMA\n\n");
}
