/**********************************************************************/
/* LEX.C  (C) Microbotica.     Diciembre 1998.                        */
/*--------------------------------------------------------------------*/
/* PROYECTO: PC-BOT                                                   */
/* SUBPROYECTO: Interprete para PC-BOT                                */
/* DESCRIPCION: Analizador lexico.                                    */
/**********************************************************************/

#include "stdio.h"
#include "dos.h"
#include "conio.h"
#include "ctype.h"
#include "io.h"
#include "stdlib.h"
#include "string.h"

#include "lex.h"

#define I_REPITE  0
#define I_SI      1
#define I_ESPERA  2
#define I_ENTRADA 3
#define I_SALIDA  4
#define I_MOTOR   5
#define I_NOTA    6

/***********************************************************************/
/*     V  A  R  I  A  B  L  E  S        G  L  O  B  A  L  E  S         */
/***********************************************************************/

/********************************/
/* Instrucciones disponibles    */
/********************************/
static char instruccion[MAXINST][MAXLONGINST]={ 
  {"REPITE"},  /* Instruccion para bucles */
  {"SI"},      /* Instruccion condicional */
  {"ESPERA"},  /* Instruccion de espera   */
  {"ENTRADA"}, /* Instruccion de entrada  */
  {"SALIDA"},  /* Instruccion de salida   */
  {"MOTOR"},   /* Instruccion de manejo motor */
  {"NOTA"},    /* Instruccion de comentario   */    
};

static char carsig=' ';   /*  Siguiente caracter al que se ha leido      */
static int nlexerror=0;   /*  Numero de Error producido   */
extern int errno;

/* Errores que se pueden producir en el analizador lexico */
static char lex_error[1][80]={
  {"Error de lectura del fichero"},    /* 1 */
};

/*************************************************************************/
/*            C     O      D      I     G       O                        */
/*************************************************************************/

char *getlex_error()
/***************************************************************/
/*  Devolver la cadena de error del ultimo error producido     */
/***************************************************************/
{
  if (nlexerror==0) return 0;            /* No hay error */
  return (char *)&lex_error[nlexerror-1];
}

int get_num_lex_error()
/*******************************/
/* Devolver el numero de error */
/*******************************/
{
  return nlexerror;
}

void reset_lex()
/**********************************************/
/* Inicializar todas las variables del lexico */
/**********************************************/
{
  carsig=' ';
  nlexerror=0;
}

char upper(char car)
/************************************************/
/* Pasar el caracter especificado a mayusculas  */
/************************************************/
{
  return (car>='a' && car<='z') ? car-'a'+'A' : car;
}

int leer_caracter(char *car, int fd)
/***********************************************************************/
/* Leer el siguiente caracter del fichero. En car se devuelve 0 si no  */
/* hay mas caracteres por leer en el fichero.                          */
/* La funcion devuelve 1 si no ha habido errores en la lectura. De lo  */
/* contrario devuelve 1.                                               */
/***********************************************************************/
{
  char c;
  int n;

  c=carsig;
  n=read(fd,&carsig,1);
  if (n==0) carsig=0;    /* No quedan mas caracteres por leer */
  if (n==-1) {           /* Error al leer del fichero         */
    carsig=0;
    *car=0;
    nlexerror=1;
    return 0;
  }
  carsig=upper(carsig);  /* Leer siempre en mayusculas */
  
  *car=c;
  return 1;
}

int es_numero(char car)
/*****************************************************************/
/* Se devuelve 1 si el caracter introducido es un digito decimal  */
/******************************************************************/
{
  return (car<='9' && car>='0') ? 1 : 0;
}

int es_letra(char car)
/**********************************************************************/
/* Se devuelve 1 si el caracter es una letra                          */
/* Solo se comprueba con las mayusculas porque todo lo que se lee se  */
/* pasa a mayusculas.                                                 */
/**********************************************************************/
{
  return (car>='A' && car<='Z') ? 1 : 0;
}

int es_instruccion(char *identif, struct tipo_token *t)
/********************************************************************/
/* Se devuelve 1 si el identificador pasado como argumento es una   */
/* instruccion. En ese caso se devuelve en t el tipo de token.      */
/********************************************************************/
{
    if (strcmp(identif,instruccion[I_REPITE])==0) {
      t->tipo=t_repite;
      return 1;
    }
    if (strcmp(identif,instruccion[I_SI])==0) {
      t->tipo=t_if;
      return 1;
    }
    if (strcmp(identif,instruccion[I_ESPERA])==0) {
      t->tipo=t_espera;
      return 1;
    }
    if (strcmp(identif,instruccion[I_ENTRADA])==0) {
      t->tipo=t_entrada;
      return 1;
    }
    if (strcmp(identif,instruccion[I_SALIDA])==0) {
      t->tipo=t_salida;
      return 1;
    }
    if (strcmp(identif,instruccion[I_MOTOR])==0) {
      t->tipo=t_motor;
      return 1;
    }
    if (strcmp(identif,instruccion[I_NOTA])==0) {
      t->tipo=t_nota;
      return 1;
    }
    
    return 0;
}

int es_idpred(char *identif, struct tipo_token *t)
/********************************************************************/
/* Se devuelve 1 si el identificador pasado como argumento es un    */
/* identificador predefinido. En ese caso se devuelve en t el tipo  */
/* de token.                                                        */
/********************************************************************/
{
    if (strcmp(identif,"ON")==0) {
      t->tipo=t_on;
      return 1;
    }
    if (strcmp(identif,"OFF")==0) {
      t->tipo=t_off;
      return 1;
    }
    if (strcmp(identif,"DERECHA")==0) {
      t->tipo=t_derecha;
      return 1;
    }
    if (strcmp(identif,"IZQUIERDA")==0) {
      t->tipo=t_izquierda;
      return 1;
    }
    
    return 0;
}

int leer_numero(byte *num, int fd)
/***********************************************************/
/* Leer un numero. Si se ha producido algun error se       */
/* devuelve 0.                                             */
/***********************************************************/
{
  char numcad[4];  /* Solo se leen numeros de hasta 3 cifras */
  int i=0;
  char c;
  
  numcad[0]=0;
  
  /* Eliminar los '0' iniciales del numero */
  while(carsig=='0') {
    if (!leer_caracter(&c,fd)) return 0; /* Error */
  }
  
  /* Meter en numero ASCII en la cadena numcad */
  while (es_numero(carsig) && i<3) {
    if (!leer_caracter(&numcad[i],fd)) return 0;  /* Error */
    i++;
  }
  numcad[i]=0;
  *num=(byte)atoi(numcad);
  
  return 1;
}

int leer_identificador(char *cad,int fd)
/***************************************************************/
/* Leer un identificador. Se define como un conjunto letras y  */
/* numeros que deben empezar siempre por una letra             */
/* Los identificadores de tamano mayor que MAXIDENTIF seran    */
/* truncados. La funcion devuelve 0 si se ha producido un error*/
/***************************************************************/
{
  int i=0;
  
  cad[0]=0;
  while ((es_letra(carsig) || es_numero(carsig)) && i<(MAXIDENTIF)) {
    if (!leer_caracter(&cad[i],fd)) return 0; /* Error */
    i++;  
  }
  cad[i]=0;
  
  return 1;
}

int leer_token(struct tipo_token *t, int fd)
/*********************************************/
/*  Leer un token del fichero especificado   */
/* Se devuelve 0 si ha habido error          */
/*********************************************/
{
  char c;
  char identif[MAXIDENTIF+1];
  byte num;

  nlexerror=0;   /* Inicialmente no hay errores */
  
  /* Eliminar los espacios que haya */
  while (carsig==' ') {
    if (!leer_caracter(&c,fd)) return 0;  /* Error */ 
  }  
  
  /* Leer un numero */
  if (es_numero(carsig)) {
    if (!leer_numero(&num,fd)) return 0;  /* Error */
    t->tipo=t_numero;
    t->valor=num;
    return 1;
  }
  
  /* Leer un identificador */
  if (es_letra(carsig)) {
    if (!leer_identificador(identif,fd)) return 0; /* Error */
    if (es_instruccion(identif,t)) return 1;
    if (es_idpred(identif,t)) return 1;
    t->tipo=t_unknow;
    return 1;
  }
  
  /* Leer caracteres especiales */
  switch(carsig) {
    case 0   : t->tipo=t_fin_fichero;
               break;
    case LF  : t->tipo=t_fin_linea;
               break;
    case '{' : t->tipo=t_abre_block;
               break;
    case '}' : t->tipo=t_cierra_block; 
               break;
    case '(' : t->tipo=t_abre_par;
               break;
    case ')' : t->tipo=t_cierra_par;
               break;
    case ',' : t->tipo=t_coma;
               break;
    case ';' : t->tipo=t_puntycoma;
               break;
    default  : t->tipo=t_unknow;           
  }
  
  leer_caracter(&c,fd);
  return 1;
}

void print_token(struct tipo_token *t)
/*************************************************************************/
/* Imprimir el tipo de token. Esta funcion solo se utiliza a efectos de  */
/* depuracion.                                                           */
/*************************************************************************/
{
  switch(t->tipo) {
    /* Instrucciones */
    case t_repite : printf ("REPITE ");  break;
    case t_if     : printf ("IF ");      break;
    case t_espera : printf ("ESPERA ");  break;
    case t_entrada: printf ("ENTRADA "); break;
    case t_salida : printf ("SALIDA ");  break;
    case t_motor  : printf ("MOTOR ");   break;
    case t_nota   : printf ("NOTA ");    break;
    
    /* Tokens de delimitacion */
    case t_abre_block  : printf ("{ ");  break;
    case t_cierra_block: printf ("} ");  break;
    case t_abre_par    : printf ("( ");  break;
    case t_cierra_par  : printf (") ");  break;
    case t_coma        : printf (", ");  break;
    case t_puntycoma   : printf ("; ");  break;
    case t_fin_linea   : printf ("LF\n"); break;
    case t_fin_fichero : printf ("FIN\n"); break;
    
    /* Otros tokens */
    case t_on          : printf ("ON ");        break;
    case t_off         : printf ("OFF ");       break;
    case t_derecha     : printf ("DERECHA ");   break;
    case t_izquierda   : printf ("IZQUIERDA "); break;
    case t_numero : printf ("%u ",t->valor); break;
    case t_unknow : printf ("UNKNOW "); break;
  }
}

