/***************************************************************************/
/* INT.C     (C) Microbotica, S.L.     Diciembre 1998.                     */
/*-------------------------------------------------------------------------*/
/* PROYECTO: PC-BOT                                                        */
/* SUBPROYECTO: Interprete para PC-BOT                                     */
/* DESCRIPCION: Modulo interprete.                                         */
/***************************************************************************/

/**************************************************************************/
/* Correspondencia entre PC-BOT y los bits de los puertos del 68HC11      */
/*                                                                        */
/* SALIDAS DIGITALES     ENTRADAS DIGITALES        MOTORES   ON  SENTIDO  */
/*                                                                        */
/*   SALIDA 1 ----> PD4    ENTRADA1 ---> PC0     MOTOR 1 --> PB0  PB1     */
/*   SALIDA 2 ----> PD3    ENTRADA2 ---> PC1     MOTOR 2 --> PB2  PB3     */
/*   SALIDA 3 ----> PD2    ENTRADA3 ---> PC2     MOTOR 4 --> PB4  PB5     */
/*   SALIDA 4 ----> PA3    ENTRADA4 ---> PC3     MOTOR 5 --> PB6  PB7     */
/*   SALIDA 5 ----> PA4    ENTRADA5 ---> PC4                              */
/*   SALIDA 6 ----> PA5    ENTRADA6 ---> PC5                              */
/*   SALIDA 7 ----> PA6    ENTRADA7 ---> PC6                              */
/*   SALIDA 8 ----> PA7    ENTRADA8 ---> PC7                              */
/*                                                                        */
/**************************************************************************/

#include "stdio.h"
#include "dos.h"
#include "conio.h"
#include "ctype.h"
#include "io.h"
#include "stdlib.h"
#include <alloc.h>

/* Modulos del interprete */
#include "sin.h"
#include "int.h"

/* Modulos de la CT */
#include "r6811pc.h"
#include "ctclient.h"

#define KEYABORT ' '   /* Tecla para abortar la interpretacion */
#define KEYPAUSE 'P'   /* Tecla para realizar una pausa en la interpretacion */

/* Localizacion de cada bit de entrada/salida */
struct tipo_bit_io {
  int npuerto;  /* Numero de puerto:0-A, 1-B, 2-C, 3-D */
  int mascara;  /* Mascara de activacion del bit       */
};

/* Localizacion de los bits de los motores */
struct tipo_bit_mot {
  int npuerto;
  int mascara_on;     /* Bit de on/off */
  int mascara_sent;   /* Bit de sentido de giro */
};

static int ninterror=0;
static int intlinea=1;

/* Errores que se pueden producir en el analizador sintactico */
static char int_error[3][80]={
  {"ERROR INTERNO!! Puntero a Null"},         /* 1 */
  {"RESET de la placa PC-BOT"},               /* 2 */
  {"Programa interrumpido por usuario"},      /* 3 */ 
};


byte puerto[4]={0,0,0,0};
struct tipo_bit_io bit_salida[8]= {
  {3,0x10},  /* SALIDA 1 */
  {3,0x08},  /* SALIDA 2 */
  {3,0x04},  /* SALIDA 3 */
  {0,0x08},  /* SALIDA 4 */
  {0,0x10},  /* SALIDA 5 */    
  {0,0x20},  /* SALIDA 6 */
  {0,0x40},  /* SALIDA 7 */
  {0,0x80},  /* SALIDA 8 */   
};

struct tipo_bit_io bit_entrada[8]={
  {2,0x01},  /* ENTRADA 1 */
  {2,0x02},  /* ENTRADA 2 */
  {2,0x04},  /* ENTRADA 3 */
  {2,0x08},  /* ENTRADA 4 */
  {2,0x10},  /* ENTRADA 5 */
  {2,0x20},  /* ENTRADA 6 */
  {2,0x40},  /* ENTRADA 7 */
  {2,0x80},  /* ENTRADA 8 */       
};

struct tipo_bit_mot bit_motor[4]={
  {1,0x01,0x02},  /* MOTOR 1 */
  {1,0x04,0x08},  /* MOTOR 2 */
  {1,0x10,0x20},  /* MOTOR 3 */
  {1,0x40,0x80},  /* MOTOR 4 */   
}; 

extern char upper(char car);
int interpreta_sentencias(lista_sentencias *);


/* ---------------------------------------------------------------------- */
/* -   FUNCIONES ASOCIADAS A LOS ERRORES DEL INTERPRETE                 - */
/* ---------------------------------------------------------------------- */

char *getint_error()
/*****************************************************************/
/* Devolver la cadena de error del ultimo error producido.       */
/*****************************************************************/
{
  if (ninterror==0) return 0;  /* No hay errores */
  return (char *)&int_error[ninterror-1];
}

int get_num_int_error()
/***********************************/
/*  Devolver el numero de error    */
/***********************************/
{
  return ninterror;
}

int get_int_linea_error()
/*************************************************************/
/* Obtener el numero de linea donde se ha producido el error */
/*************************************************************/
{
  return intlinea;
}

/* ------------------------------------------------------------------- */
/* -     FUNCIONES DE BAJO NIVEL                                     - */
/*-------------------------------------------------------------------- */

int dir_puerto(int npuerto)
/******************************************/
/* Devolver la direccion de cada puerto   */
/******************************************/
{
  switch(npuerto) {
    case 0 : return PORTA;
    case 1 : return PORTB;
    case 2 : return PORTC;
    case 3 : return PORTD;
  }
  return PORTA;
}

void salida(int nsal, int estado)
/*****************************************************/
/* Poner la salida especificada al estado indicado   */
/*****************************************************/
{
  int npuerto;
  int mascara;
  int dir;

  nsal--;

  /* Obtener el puerto donde esta la salida especificada */
  npuerto=bit_salida[nsal].npuerto; 
  /* Obtener la mascara del bits */
  mascara=bit_salida[nsal].mascara;
  
  switch(estado) {
    case 0 : puerto[npuerto]&=(~mascara);
             break;  
    case 1 : puerto[npuerto]|=mascara;
             break;
  }
  
  dir=dir_puerto(npuerto);  /* Obtener la direccion del puerto */
  
  /* Actuar sobre el puerto */
  store(puerto[npuerto],dir);
}

void motor(int nmot, int estado)
/*****************************************************/
/*  Poner el motor especificado al estado indicado   */
/*****************************************************/
{
  int npuerto;
  int mascara_on;
  int mascara_sent;
  int dir;
  
  nmot--;
  
  /* Obtener el puerto donde esta la salida especificada */
  npuerto=bit_motor[nmot].npuerto; 
  /* Obtener las mascaras de los bits */
  mascara_on=bit_motor[nmot].mascara_on;
  mascara_sent=bit_motor[nmot].mascara_sent;
  
  switch(estado) {
    case 0 : puerto[npuerto]&=(~mascara_on);
             break;  /* Apagar motor */
    case 1 : puerto[npuerto]|=mascara_sent; /* Giro derecha */
             puerto[npuerto]|=mascara_on;
             break;
    case 2 : puerto[npuerto]&=(~mascara_sent); /* Giro izquierda */
             puerto[npuerto]|=mascara_on;
             break;
  }
  
  dir=dir_puerto(npuerto);  /* Obtener la direccion del puerto */
  
  /* Actuar sobre el puerto */
  store(puerto[npuerto],dir);
}



int check_kbd()
/******************************************************************/
/* Comprobar el teclado. Se devuelve 0 si el usuario ha abortado  */
/******************************************************************/
{
  char c;

  if (kbhit()) {
    c=upper(getch());
    if (c==KEYABORT) {
      ninterror=3;
      return 0;
    }
    if (c==KEYPAUSE) {
      printf ("PROGRAMA PARADO. Pulse una tecla para continuar\n");
      c=getch();
      if (c==KEYABORT) {
        ninterror=3;
        return 0;
      }
    }
  }
  return 1;
}

/* -------------------------------------------------- */
/*       FUNCIONES DE NAVEGACION                      */
/* -------------------------------------------------- */

int interpreta_repite(tipo_srepite *rep)
/**************************************/
/* Interpretar una sentencia REPITE   */
/**************************************/
{
  if (rep==NULL) {
    ninterror=1;
    return 0;
  }  
  
  intlinea=rep->nlinea;  /* Leer numero de linea */

  /* Bucle infinito */
  if (rep->repeticion==0) rep->quedan=1;

  while (rep->quedan>0) {
/*    printf ("Linea %u: REPITE(%u)\n",rep->nlinea,rep->quedan);*/
    if (rep->repeticion!=0) rep->quedan--;
    if (!interpreta_sentencias(rep->ss)) return 0;
  }
  return 1;
}

int interpreta_if(tipo_sif *sif)
/******************************/
/* Interpretar sentencia IF   */
/******************************/
{
  int npuerto;
  int nent;
  int dir;
  int mascara;
  int valor;
  int estado;
  int nlinea;

  if (sif==NULL) {
    ninterror=1;
    return 0;
  }  
  if (sif->condicion==NULL) {
    ninterror=1;
    return 0;
  }  
  
  intlinea=sif->condicion->nlinea;  /* Leer numero de linea */
  
  nent=sif->condicion->nent;  /* Numero de entrada a comprobar */
  npuerto=bit_entrada[nent-1].npuerto;
  mascara=bit_entrada[nent-1].mascara;
  dir=dir_puerto(npuerto);  /* Obtener la direccion del puerto */
  switch(sif->condicion->estado) {
    case i_on: estado=1; break;
    case i_off: estado=0; break;
  }
  
  /* Leer el puerto */
  load(&puerto[npuerto],dir);
  
  /* Verificar condicion */
  valor=(puerto[npuerto]&mascara);
  
  /* La condicion ON se cumple */
  if (valor==0 && estado==1) { 
    if (!interpreta_sentencias(sif->ss)) return 0;
    return 1;
  }
  
  /* La condicion OFF se cumple */
  if (valor>0 && estado==0) {
    if (!interpreta_sentencias(sif->ss)) return 0;
  }
  return 1;
}

int interpreta_espera(tipo_sespera *esp)
/**********************************/
/* Interpretar sentencia espera   */
/**********************************/
{
  int t;
  char c;
  
  if (esp==NULL) {
    ninterror=1;
    return 0;
  }
  
  intlinea=esp->nlinea;  /* Leer numero de linea */
  
  /* Si el argumento de espera es 0 se espera tecla de usuario */
  if (esp->seg==0) {
    printf ("Linea %u: PULSE UNA TECLA PARA CONTINUAR. ESPACIO PARA ABORTAR\n",intlinea);
    c=getch();
    if (c==KEYABORT) {
      ninterror=3;
      return 0;
    }
    return 1;
  }

  textcolor(14);
  cprintf ("Linea %u: ",intlinea);
  textcolor(10);
  cprintf ("ESPERA(%u);\n\r",esp->seg);
  textcolor(7);

  /* Realizar la espera en decimas de segundo */
  /* Cada decima se comprueba el teclado      */
  for (t=0; t<esp->seg; t++) {
    if (!check_kbd()) return 0;
    delay(100);
  }
  return 1;
}

int interpreta_salida(tipo_ssalida *sal)
/************************************/
/* Interpretar sentencia salida     */
/************************************/
{
  int estado;
  char cad[2][4]={"OFF","ON"};

  if (sal==NULL) {
    ninterror=1;
    return 0;
  }  
  
  switch(sal->estado) {
    case i_on : estado=1; break;
    case i_off: estado=0; break;
  }
  
  salida(sal->nsal,estado);
  textcolor(14);
  cprintf ("Linea %u: ",sal->nlinea);
  textcolor(9);
  cprintf ("SALIDA(%u,%s);\n\r",sal->nsal,cad[estado]);
  textcolor(7);
  
  intlinea=sal->nlinea;  /* Leer numero de linea */
  return 1;
}


int interpreta_motor(tipo_smotor *mot)
/*********************************/
/* Interpretar sentencia MOTOR   */
/*********************************/
{
  int estado;
  char cad[3][12]={"OFF","DERECHA","IZQUIERDA"};

  if (mot==NULL) {
    ninterror=1;
    return 0;
  }  
  
  switch(mot->estado) {
    case m_off: estado=0; break;
    case m_derecha: estado=1; break;
    case m_izquierda: estado=2; break;
  }
  motor(mot->nmotor,estado);
  textcolor(14);
  cprintf ("Linea %u: ",mot->nlinea);
  textcolor(11);
  cprintf ("MOTOR(%u,%s)\n\r",mot->nmotor,cad[estado]);
  textcolor(7);
  
  intlinea=mot->nlinea;  /* Leer numero de linea */
  return 1;
}

int interpreta_sentencias(lista_sentencias *ls)
/**********************************************************************/
/* Interpretar una lista de sentencias                                */
/* Se devuelve 1 si se han interpretado las sentencias correctamente  */
/* Se devuelve 0 en caso de error                                     */
/**********************************************************************/
{
  if (ls==NULL) return 1;  /* No hay mas sentencias */


  if (!check_kbd()) return 0;

  if (!check_conexion()) {
    ninterror=2;
    return 0;
  }

  switch(ls->ts) {
    case s_repite : if (!interpreta_repite((tipo_srepite *)ls->s)) return 0;
		    break;
    case s_if     : if (!interpreta_if((tipo_sif *)ls->s)) return 0;
		    break;
    case s_espera : if (!interpreta_espera((tipo_sespera *)ls->s)) return 0;
		    break;
    case s_salida : if (!interpreta_salida((tipo_ssalida *)ls->s)) return 0;
		    break;
    case s_motor  : if (!interpreta_motor((tipo_smotor *)ls->s)) return 0;
                    break;
  }
  
  if (!interpreta_sentencias(ls->ss)) return 0;
  
  return 1;
}

int interpreta_ast(lista_sentencias *ls)
/***********************/
/* Interpretar un AST  */
/***********************/
{
  ninterror=0;   /* Inicialmente no hay error sintactico */
  intlinea=1;

  /* Poner los puertos a cero */
  puerto[0]=0;
  puerto[1]=0;
  puerto[2]=0;
  puerto[3]=0;

  check_conexion();
  if (!hay_conexion()) {
    ninterror=2;           /* No hay conexion */
    return 0;
  }  

  /* ----- Configurar tarjeta PC-BOT correctamente ----   */
  store(0x00,PORTD);   /* Poner puerto D a cero          */
  store(0x00,SPCR);    /* Poner Puerto D en salida CMOS   */
  store(0xFF,DDRD);    /* Configurar puerto D para salida */
  store(0x80,PACTL);   /* Configurar PA7 para salida      */

  store(0x00,PORTA);
  store(0x00,PORTB);
  store(0x00,PORTD);

  if (ls==NULL) return 1;  /* Programa nulo */
  
  /* ---- Interpretar las sentencias ---  */
  if (!interpreta_sentencias(ls)) {
  return 0;
  }
  
  return 1;
}
