/*
 ͻ
  SERIE.C     (c) GRUPO J&J. Julio 1997.                                   
 ͹
                                                                           
  Versin 2.0.  Rutinas de comunicaciones serie mediante interrupciones.   
  Estas rutinas no son de propsito general sino que estn adaptadas para  
  trabajar con la tarjeta CT6811.                                          
                                                                           
 ͼ
*/


/*

			 ͻ
			                         
ĺ   I N T E R F A Z      
			                         
			 ͼ
*/

/*
 ͻ
                              
  APERTURA DEL PUERTO SERIE   
                              
 ͼ
*/


void abrir_puerto_serie(int puerto);
/*
Ŀ
  Antes de comenzar a trabajar con el puerto serie hay que llamar a   
  esta funcin. En el argumento se le pasa el nmero de COM con el    
  que se quiere trabajar. Asi por ejemplo para abrir el COM1 habr    
  que llamar a la funcin:                                            
                                                                      
  abrir_puerto_serie(1);                                              
                                                                      
                                                                      
  Los puertos soportados son COM1, COM2, COM3 y COM4. Si se especifi- 
  ca como argumento un nmero distinto de 1,2,3  4 se abrir el      
  COM1                                                                
                                                                      
    Despus de llamar a esta rutina el puerto especificado queda con- 
 figurado para trabajar con 8 bits de datos, 1 bit de Stop y ninguno  
 de paridad. La velocidad es el 9600 baudios.                         
                                                                      
                                                                      
*/


/*
 ͻ
                              
  RUTINAS DE CONFIGURACION    
                              
 ͼ
*/


void baudios(int velocidad);
/*
Ŀ
  Establecer la velocidad en baudios especificada. Slo se permiten   
  tres velocidades: 1200, 7680 y 9600 baudios. Si se especifica otra  
  velocidad diferente se tomaran 9600 baudios.                        
  que llamar a la funcin:                                            
                                                                      
  Ejemplo de utilizacin:  baudios(1200);                             
                                                                      
*/


void vaciar_buffer(void);
/*
Ŀ
  Vaciar el buffer de recepcin. Si existan caracteres se desechan   
*/

/*
 ͻ
                                
  RUTINAS DE LECTURA/ESCRITURA  
                                
 ͼ
*/


void enviar_car(char car);
/*
Ŀ
  Enviar un carcter por el puerto serie abierto.    
*/


int car_waiting(void);
/*
Ŀ
  Esta funcin devuelve un valor 0 si no existe ningn carcter pen-  
  diente de ser ledo en el buffer de recepcin. Se devuelve un valor 
  distinto de 0 si existe algn carcter pendiente de ser ledo.      
*/

char leer_car(void);
/*
Ŀ
  Leer un carcter del buffer de recepcin. Si no hay ningn carcter 
  pendiente de ser ledo se espera indefinidamente hasta que llegue   
  algn carcter.                                                     
*/


char leer_car_plazo(int plazo,int *timeout);
/*
Ŀ
  Leer un carcter del buffer de recepcin. Si el carcter no se ha   
  recibido dentro del plazo establecido se devuelve un valor distin-  
  to de 0 en timeout. En caso de llegar un carcter dentro del plazo  
  indicado se devuelve el carcter y se devuelve 0 en timeout.        
                                                                      
    El valor del plazo se especifica en tics de reloj. Cada tic equi- 
  vale aproximadamente a 55ms.                                        
                                                                      
  Ejemplo:  c=leer_car_plazo(5,&timeout);  Leer un caracter. Si el    
            caracter no llega en un plazo de 5 tics de reloj se       
            devuelve un 1 en timeout.                                 
                                                                      
*/


/*
 ͻ
                        
    OTRAS RUTINAS       
                        
 ͼ
*/

void dtr_on(void);
/*
Ŀ
  Activar la seal DTR     
*/

void dtr_off(void);
/*
Ŀ
 Desactivar la seal DTR   
*/

int ok_break(void);
/*
Ŀ
  Se devuelve un valor 0 si no se ha recibido ninguna seal de BREAK  
  Se devuelve un valor distinto de 0 si se ha recibo un BREAK         
*/

int wait_break(int plazo);
/*
Ŀ
  Esperar a que venga una seal de BREAK en el plazo indicado.        
  Si es as se devuelve un valor distinto de 0. Si transcurre el plazo
  y no se ha recibido un BREAK se devuelve 0.                         
    El plazo se especifica en TICS de reloj. Cada Tic equivale aprox. 
  a 55ms.                                                             
*/


/*
 ͻ
                               
  TERMINAR CON EL PUERTO SERIE 
                               
 ͼ
*/

void cerrar_puerto_serie();
/*Ŀ
    Antes de terminar el programa es necesario llamar    
    a este procedimiento para desactivar las rutinas de  
    interrupcin.                                        
   */



/*

		   ͻ
		                                 
ĺ I M P L E M E N T A C I O N  
		                                 
		   ͼ
*/



/***************************************************************************/
/*                                                                         */
/*      Breve descripcion de los registros de la UART                      */
/*   Para mas informacion:                                                 */
/*                                                                         */
/*       - Pag 256, "COMUNICACIONES SERIE". Joe Campbell. Ed. Anaya        */
/*                                                                         */
/***************************************************************************/
/*                                                                         */
/*    1.- Registro de DATOS: Utilizado para Leer-mandar BYTES              */
/*         (Offset 0) (Lectura-escritura)                                  */
/*                                                                         */
/*    2.- Registro de ACTIVACION DE INTERRUPCIONES (Offset 1)              */
/*        * Se produce interrupcin cuando se activen los bits:            */
/*            -0 : Dato recibido.                                          */
/*            -1 : Buffer transmision vacio. (listo para enviar)           */
/*            -2 : Seal de BREAK detectada.                               */
/*            -3 : Entrada RS-232-> se genera una interrupcin cuando      */
/*                 cambia el estado de cualquiera de los estados del RS-232*/
/*            -4-7: Siempre a cero.                                        */
/*                                                                         */
/*    3.- IDENTIFICACION DE INTERRUPCION: Indican la interrupcin que est */
/*         pendiente: (Offset 2)                                           */
/*             Bit 2    Bit 1   Bit 0                                      */
/*               0        0       1    => No hay interrupcion pendiente    */
/*               0        1       0    => Buffer transmision vacio         */
/*               1        0       0    => Dato recibido                    */
/*               1        1       0    => BREAK                            */
/*               0        0       0    => Entrada RS-232                   */
/*                                                                         */
/*                                                                         */
/*    4.- Registro CONTROL DE LINEA: Formato de datos: (offset 3)          */
/*                                                                         */
/*          Numero de bits de datos:                                       */
/*           Bits 1  Bit 0                                                 */
/*             0       0     ==> 5 bits de datos                           */
/*             0       1     ==> 6  "       "                              */
/*             1       0     ==> 7  "       "                              */
/*             1       1     ==> 8  "       "                              */
/*                                                                         */
/*          Numero de Bits de STOP:                                        */
/*           Bit 2                                                         */
/*            0              ==> 1 Bit de STOP                             */
/*            1              ==> 2 Bits de STOP                            */
/*                                                                         */
/*          Paridad:                                                       */
/*            Bit 5   Bit 4   Bit 3                                        */
/*              0       0       0   ===> NINGUNA PARIDAD                   */
/*              0       0       1   ===> PARIDAD IMPAR                     */
/*              0       1       1   ===> PARIDAD PAR                       */
/*              1       0       1   ===> MARCA                             */
/*              1       1       1   ===> ESPACIO                           */
/*                                                                         */
/*          Control BREAK:                                                 */
/*            Bit 6                                                        */
/*              0        ==> BREAK Off                                     */
/*              1        ==> BREAK ON                                      */
/*                                                                         */
/*          DLAB  --> Es un truco para utilizar los registros 1 y 2 para   */
/*                    (Datos y activacion interrupciones) para seleccionar */
/*                    la velocidad, cuando este bit esta activo.           */
/*                                                                         */
/*                                                                         */
/*    5.- Registro CONTROL DEL MODEM: Control salida RS-232  (offset 4)    */
/*                                                                         */
/*          Bit 0: Al activarse se activa Data Terminal Ready (DTR)        */
/*          Bit 1: Se activa Request To Send (RTS)                         */
/*          Bit 2: Definible por el usuario                                */
/*          Bit 3: Definible por el usuario                                */
/*          Bit 4: Test local.                                             */
/*          Bits 5-7: Siempre a cero.                                      */
/*                                                                         */
/*    6.- Registro ESTADO DE LINEA: Control estado Serializacin (offset 5)*/
/*                                                                         */
/*          Bit 0: Dato listo. (esperando a ser leido)                     */
/*          Bit 1: Error de sobreescritura.                                */
/*          Bit 2: Error de Paridad                                        */
/*          Bit 3: Error de Trama                                          */
/*          Bit 4: Deteccin de BREAK                                      */
/*          Bit 5: Buffer transmisin vacio                                */
/*          Bit 6: Transmisor vacio                                        */
/*          Bit 7: Siempre a cero                                          */
/*                                                                         */
/*    7.- Registro ESTADO DEL MODEM: (entrada RS-232)                      */
/*                                                                         */
/*          Bit 0: Delta CTS: Clear To Send.                               */
/*          Bit 1: Delta DSR: Data Set Ready                               */
/*          Bit 2: Delta RI : Ring Indicator.                              */
/*          Bit 3: Delta DCD: Data Carrier Detect.                         */
/*                                                                         */
/*          Bit 4: CTS                                                     */
/*          Bit 5: DSR                                                     */
/*          Bit 6: RI                                                      */
/*          Bit 7: DCD                                                     */
/*                                                                         */
/*          Los bits donde pone delta indican que se ha producido un       */
/*        desde la ultima vez que se leyo el registro. (0-3). Los bits 4-7 */
/*        indican el estado "absoluto" de las lineas RS-232.               */
/*                                                                         */
/***************************************************************************/

#include "dos.h"
#include "conio.h"


/*
Ŀ
     DEFINICIONES      

*/

#define BUFFER_SERIE 512     /* 512 bytes de buffer */


#define COM1  0x3f8   /* Direccin base de la Uart para COM1  */
#define COM2  0x2f8   /* Direccin base de la Uart para COM2  */
#define COM3  0x3e8   /* Direccin base de la Uart para COM3  */
#define COM4  0x2e8   /* Direccin base de la Uart para COM4  */



/* ---- OFFSET de los registros de la UART: ----- */

#define datos            0   /* Registro de datos                        */
#define act_interrup     1   /* Registro activacion interrupciones       */
#define ident_interrup   2   /* Registro indentificacin interrupciones  */
#define control_linea    3   /* Registro Control de linea                */
#define control_modem    4   /* Registro Control de modem                */
#define estado_linea     5   /* Registro Estado de Linea                 */
#define estado_modem     6   /* Registro Estado del Modem                */

#define N81     0x03     /* Mscara para configurar la UART con el       */
			 /* formato: 8bits de datos, 1 STOP y no paridad */
#define ON  1
#define OFF 0


/*
Ŀ
   VARIABLES GLOBALES  

*/


int   puerto_actual=COM1;  /* Puerto Actual. Por defecto COM1 */
char  hwint;               /* Estado de las interrupciones HW */

/* -- Buffer circular donde almacenar los datos recibidos -- */
char buffer[BUFFER_SERIE];
int  cabeza=0;		 	/* Cabeza del buffer */
int  cola=0;                    /* Cola del buffer   */
int  ncar=0;   /* Indica el nmero de caracteres que hay en el buffer */

char sbreak;   /* Indica si se ha recibido o no un BREAK */

long int far *tempo=(long int far *)0x0000046C; /* Temporizador PC */

void interrupt serie();  /* Rutina de servicio interrupcin de las   */
			 /* interrupciones de la UART                */

void accion_rxcar();     /* Accion a realizar al recibir un carcter  */
void accion_break();     /* Accin a realizar al recibir un BREAK     */

void interrupt serie(...)
/*
Ŀ
  Rutina de servicio de interrupcin de la UART. La nica interrupcin que 
  se tiene en cuenta es la de dato recibido. Cada vez que se recibe un     
  dato, se activa esta interrupcin y se mete el dato en un buffer.        
                                                                           
   Si se recibe una seal de BREAK se activa la variable global sbreak     
  para sealizar que ha venido una seal de BREAK.                         
*/
{
  char c;

  /* Primero se comprueba si se ha recibido una seal de BREAK */

  c=inportb(puerto_actual+estado_linea);

  if ( (c & 0x10)==0x10 ) {             /* Si  BREAK */
    sbreak=ON;                           /* Sealizar BREAK activo */
    accion_break();                      /* Realizar una accin    */
    
    c=inportb(puerto_actual+datos);      /* Quitar flag interrupcin */
    outportb(0x20,0x20);   /* Mandar orden EOI al 8259 */
    return;
  }
  else {
    sbreak=OFF;
  }

  /* No se ha recibido seal de BREAK --> Ha venido un caracter */

  c=inportb(puerto_actual+datos);   /* Leer dato recibido       */
  if (ncar<BUFFER_SERIE) {          /* Si buffer no est lleno  */
    buffer[cola]=c;
    cola=(cola + 1) % BUFFER_SERIE;
    ncar++;
  }
  accion_rxcar();
  outportb(0x20,0x20);              /* Mandar orden EOI al 8259 */
}


/*
 ͻ
                              
  APERTURA DEL PUERTO SERIE   
                              
 ͼ
*/

void abrir_puerto_serie(int puerto)
/*
Ŀ
  Antes de comenzar a trabajar con el puerto serie hay que llamar a        
  esta funcin. En el argumento se le pasa el nmero de COM con el         
  que se quiere trabajar. Asi por ejemplo para abrir el COM1 habr         
  que llamar a la funcin:                                                 
                                                                           
  abrir_puerto_serie(1);                                                   
                                                                           
                                                                           
  Los puertos soportados son COM1, COM2, COM3 y COM4. Si se especifi-      
  ca como argumento un nmero distinto de 1,2,3  4 se abrir el           
  COM1                                                                     
                                                                           
    Despus de llamar a esta rutina el puerto especificado queda con-      
 figurado para trabajar con 8 bits de datos, 1 bit de Stop y ninguno       
 de paridad. La velocidad es el 9600 baudios.                              
                                                                           
                                                                           
   INFORMACION SOBRE LAS INTERRUPCIONES SERIE:                             
                                                                           
                                                                           
    - Las interrupciones del puerto serie son interrupciones hardware, por 
      tanto habr que habilitarlas en el registro de mscara de            
      interrupcin del controlador de interrupciones 8259.                 
                                                                           
      Este registro se encuentra en el puerto 0x21. Los 8 bits de este     
      puerto habilitan o deshabilitan distintas interrupciones.            
      Un 1 indica que la interrupcin est DESHABILITADA. Un 0 indica que  
      la interrupcin est HABILITADA. El contenido del registro es el     
      siguiente:                                                           
                                                                           
                                                                           
    N de bit:     7       6       5      4      3     2     1        0    
                                                                           
                 LPT1   Disketera LPT2  COM1   COM2  ----  TECLADO  TIMER  
                                                                           
      Por tanto, para permitir interrupciones del COM1 hay que poner a     
      cero el bit 4. Para el COM2 se pone a cero el bit 3.                 
                                                                           
    - En la UART inicialmente no se permiten interrupciones. Las interrup- 
      ciones se activan a travs del registro de activacin de interrup-   
      cin. Hay 4 causas de interrupcin, cada una asociada a un bit del   
      registro citado. Para permitir interrupciones cuando ocurra alguna   
      de las 4 posibles causas de interrupcin hay que poner a 1 el bit    
      correspondiente.                                                     
                                                                           
    - Finalmente, para que las interrupciones salgan de la UART, hay que   
      poner a 1 el bit 3 del registro de control del modem. En             
      encontrar este puto bit he tardado mucho tiempo, porque la informa-  
      cin no se encuentra en ningn lado. As que considerate agraciado   
      de saberlo.                                                          
                                                                           
*/
{
  char mask;
  char vect;
  char c1,c2;

  ncar=0;     /* Inicializar variables de la rutina de interrupcin */
  cola=0;
  cabeza=0;

  switch(puerto) {
    case 2 : puerto_actual=COM2;
	     mask=0xF7;
	     vect=0x0b;
	     break;
    case 3 : puerto_actual=COM3;
	     mask=0xEF;
	     vect=0x0c;
	     break;
    case 4 : puerto_actual=COM4;
	     mask=0xF7;
	     vect=0x0b;
	     break;
    default:
	puerto_actual=COM1;
	mask=0xEF;
	vect=0x0c;
  }


  /* Configuracin para N81 */
  outportb(puerto_actual+3,0x03);

  setvect(vect,serie);  /* Cambiar vector de interrupcin */

  hwint=inportb(0x21);   /* Obtener el estado de las interrupcines hw */
  c2=hwint & mask;
  outportb(0x21,c2);     /* Habilitar la interrupcin correspondiente */

  /* --- Permitir la interrupcin dato recibido y BREAK ---- */
  outportb(puerto_actual+act_interrup,0x05);

  /* --- Activar GPO2 para habilitar las interrupciones --- */
  outportb(puerto_actual+control_modem,0x08);
  inportb(puerto_actual+datos);
}


/*
 ͻ
                              
  RUTINAS DE CONFIGURACION    
                              
 ͼ
*/

void baudios(int baud)
/*
Ŀ
  Establecer la velocidad en baudios especificada. Slo se permiten   
  tres velocidades: 1200, 7680 y 9600 baudios. Si se especifica otra  
  velocidad diferente se tomaran 9600 baudios.                        
  que llamar a la funcin:                                            
                                                                      
  Ejemplo de utilizacin:  baudios(1200);                             
                                                                      
 */
{
  unsigned char ba=0,bb=0x60;  /*  Byte alto y byte bajo  */
  unsigned char control;

  switch(baud){
    case 1200 : ba=0x00; bb=0x60; break;
    case 7680 : ba=0x00; bb=0x0f; break;
    case 9600 : ba=0x00; bb=0x0c; break;

    default   : ba=0x00; bb=0x0c;
  }

  /* Activar bit 7 del registro de control de linea  */
  outportb(puerto_actual+control_linea,0x80);

  /*  Situar los divisores de velocidad en los registros 0 y 1 de la Uart */
  outportb(puerto_actual+datos,bb);
  outportb(puerto_actual+act_interrup,ba);

  /* Establecer configuracin N81  */
  outportb(puerto_actual+control_linea,N81);
}


void vaciar_buffer()
/*
Ŀ
  Vaciar el buffer de recepcin. Si existan caracteres se desechan   
*/
{
  ncar=0;
  cabeza=cola;
}


/*
 ͻ
                                
  RUTINAS DE LECTURA/ESCRITURA  
                                
 ͼ
*/

int listo_enviar()
/*Ŀ
    Devuelve TRUE cuando est listo para enviar datos.   
    Para ello se comprueba si el bit 5 del registro de   
    estado de lnea est activado.                       
   */
{
  return inportb(puerto_actual+estado_linea) & 0x20;
}

void enviar_car(char car)
/*
Ŀ
  Enviar un carcter por el puerto serie.    
 */
{
  while (!listo_enviar()) /* Esperar hasta que el registro de transmisin  */
  ;                       /*   est vacio                                  */

  outportb(puerto_actual+datos,car);
}

int car_waiting()
/*Ŀ
    Devuelve TRUE si hay un carcter esperando para ser  
    ledo. Para ello se comprueba el bit 0 del registro  
    de estado de lnea de la UART.                       
   */
{
  return (ncar<BUFFER_SERIE && cabeza==cola) ? 0 : 1;
}

char leer_car()
/*
Ŀ
  Leer un carcter por el puerto serie. Si no hay      
  ningn carcter esperando se espera hata que se      
  introduzca alguno.                                   
 */
{
  char c;

  while (!car_waiting());  /*  Esperar hasta que se reciba un byte  */

  c=buffer[cabeza];
  cabeza= (cabeza+1) % BUFFER_SERIE;
  ncar--;

  return c;
}

char leer_car_plazo(int plazo,int *timeout)
/*
Ŀ
  Leer un carcter del buffer de recepcin. Si el carcter no se ha   
  recibido dentro del plazo establecido se devuelve un valor distin-  
  to de 0 en timeout. En caso de llegar un carcter dentro del plazo  
  indicado se devuelve el carcter y se devuelve 0 en timeout.        
                                                                      
    El valor del plazo se especifica en tics. Cada Tic equivale a 55ms
                                                                      
*/
{
  long int tiempo;
  char c;

  tiempo=*tempo+1+plazo;

  while (!car_waiting()) {
    if (tiempo==*tempo) {          /* Error de timeout */
      *timeout=1;
      return 0;
    }
  }

  c=buffer[cabeza];
  cabeza=(cabeza+1) % BUFFER_SERIE;
  ncar--;

  *timeout=0;
  return c;
}

/*
 ͻ
                        
    OTRAS RUTINAS       
                        
 ͼ
*/


void dtr_on(void)
/*
Ŀ
  Activar la seal DTR     
 */
{
  char estado;
  estado=inportb(puerto_actual+control_modem) | 0x1;
  outportb(puerto_actual+control_modem,estado);
}

void dtr_off(void)
/*
Ŀ
 Desactivar la seal DTR   
*/
{
  char estado;
  estado=inportb(puerto_actual+control_modem) & 0xfe;
  outportb(puerto_actual+control_modem,estado);
}


int ok_break(void)
/*
Ŀ
  Se devuelve un valor 0 si no se ha recibido ninguna seal de BREAK  
  Se devuelve un valor distinto de 0 si se ha recibo un BREAK         
*/
{
  if (sbreak==ON) {
    sbreak=OFF;
    return 1;
  }
  else return 0;
}

int wait_break(int plazo)
/*
Ŀ
  Esperar a que venga una seal de BREAK en el plazo indicado.        
  Si es as se devuelve un valor distinto de 0. Si transcurre el plazo
  y no se ha recibido un BREAK se devuelve 0.                         
*/
{
  long int tiempo;
  char c;

  tiempo=*tempo+1+plazo;

  while (!ok_break()) {   /* Mientras no venga BREAK */
    if (kbhit()) {        /* Si tecla pulsada plazo abortado */
      getch();
      return 0;
    }
    if (tiempo==*tempo) {          /* Error de timeout */
      return 0;
    }
  }
  return 1;   /* Break recibido dentro del plazo */
}

/*
 ͻ
                               
  TERMINAR CON EL PUERTO SERIE 
                               
 ͼ
*/

void cerrar_puerto_serie()
/*Ŀ
    Antes de terminar el programa es necesario llamar    
    a este procedimiento para desactivar las rutinas de  
    interrupcin.                                        
   */
{
  /* --- Desactivar interrupcin de Dato recibido --- */
  outportb(puerto_actual+act_interrup,0x00);

  /* --- Desactivar  DTR y GPO2 --- */
  outportb(puerto_actual+control_modem,0x00);

  /* --- Dejar las interrupciones HW como al principio --- */
  outportb(0x21,hwint);
}

