/**************************************************************************/
/*                                                                        */
/*  MICROCONTROLER-BOOT.  GRUPO J&J                     MICROBOTICA       */
/*                                                                        */
/**************************************************************************/
/*                                                                        */
/*  Este programa simplemente enva un fichero en formato .S19 de Moto-   */
/*  rola por el puerto serie.                                             */
/*                                                                        */
/*   Se ha desarrollado para enviar programas al microcontrolador         */
/*   MC68HC11A0 de Motorola. La velocidad de transmisin es de 7680 Baud  */
/*   y la configuracin es N81 (8 bits de datos, 1 bit Stop y sin paridad */
/*   Inicialmente se tiene que enviar al microcontrolador el cdigo $FF   */
/*   (255 en decimal) y a continuacin el cdigo mquina del programa en  */
/*   ensamblador. El MC hace un eco de todos los caracteres enviados.     */
/*   El programa MC-BOOT comprueba si el eco realizado es correcto.       */
/*                                                                        */
/*                                                                        */
/**************************************************************************/


/* Actualizaciones                                                        */
/* mcboot1 -> 2.1 -> preparado para el COM1, 9600, opcion DTR por pantalla*/
/* mcboot2 -> 2.2 -> preparado para el COM2, 9600, opcion DTR por pantalla*/

#define CR 13
#define LF 10

const float VERSION = 2.2;

static int xcursor,ycursor;
void salva_cursor();
void recupera_cursor();
static int puerto=2;
static int baud;
static int dtr=0;

void display_modo(modo)
/*******************************************************************/
/*  Imprimir en pantalla el modo actual de funcionamiento:         */
/*   MODO=1 --> Modo terminal.                                     */
/*   MODO=2 --> Modo GAIA                                          */
/*******************************************************************/
{
  salva_cursor();
  window(1,1,80,25);
  gotoxy(60,1);
  textcolor(15);
  switch(modo) {
    case 1: cprintf("Terminal");
	    break;
    case 2: cprintf("        ");
	    textcolor(7);
	    /*cprintf ("    ");*/
	    break;
  }
  textcolor(7);
  window(1,3,80,21);
  recupera_cursor();

}


void display_baud(baud)
int baud;
/**********************************************/
/*  Imprimir en pantalla la velocidad actual  */
/**********************************************/
{
  salva_cursor();
  window(1,1,80,25);
  gotoxy(70,1);
  textcolor(15);
  cprintf("%u",baud);
  textcolor(7);
  window(1,3,80,21);
  recupera_cursor();
}

void display_com(com)
int com;
/*******************************************************************/
/*  Imprimir en pantalla el com actual                             */
/*******************************************************************/
{
  salva_cursor();
  window(1,1,80,25);
  gotoxy(50,1);
  textcolor(15);
  cprintf("COM%u",com);
  textcolor(7);
  window(1,3,80,21);
  recupera_cursor();

}

void clear_menu()
/************************************/
/* Borrar el menu de abajo del todo */
/************************************/
{
  char i;

  window(1,1,80,25);

  for (i=1; i<80; i++) {
    gotoxy(i,25);
    cprintf(" ");
    gotoxy(i,24);
    cprintf(" ");
  }

  window(1,3,80,21);
}

void salva_cursor()
/**********************************************/
/* Salvar las coordenadas actuales del cursor */
/**********************************************/
{

  xcursor=wherex();
  ycursor=wherey();
}

void recupera_cursor()
/*****************************************/
/*  Recuperar las coordenadas del cursos */
/*****************************************/
{
  gotoxy(xcursor,ycursor);
}

void menu_mcboot()
/*****************************/
/*  Sacar el menu principal  */
/*****************************/
{
  display_modo(1);  /* Modo terminal */
  salva_cursor();
  clear_menu();
  window(1,1,80,25);
  gotoxy(1,24);
  textcolor(10); cprintf("ESC");
  textcolor(11); cprintf("-Terminar  ");
  textcolor(10); cprintf("F1");
  textcolor(11); cprintf ("-Bootstrap  ");
  textcolor(10); cprintf("F2");
  textcolor(11); cprintf("-Expandido ");
  textcolor(10); cprintf("F3");
  textcolor(11); cprintf("-GAIA    ");
  textcolor(10); cprintf("F4");
  textcolor(14); cprintf("-Puerto  ");
  textcolor(10); cprintf("F5");
  textcolor(14); cprintf("-Baudios");
  gotoxy(1,25);
  textcolor(10); cprintf ("F6");
  textcolor(11); cprintf ("-Volcar     ");
  textcolor(10); cprintf ("F7");
  textcolor(11); cprintf ("-DTR ON/OFF");
  window(1,3,80,21);
  textcolor(7);
  recupera_cursor();
}


void presenta()
{
  clrscr();
  textcolor(14); cprintf("MC-BOOT ");
  textcolor(12); cprintf("2.2, ");
  textcolor(11); cprintf(" GRUPO J&J.     MICROBOTICA  \n\r");
  textcolor(9);
  cprintf("\n\r");
  gotoxy(1,23);
  cprintf("\n\r");
  menu_mcboot();
}

unsigned int load_expanded(f)
int f;
/*********************************************************************/
/*  Cargar el fichero f en RAM externa. Debe estar en el formato     */
/*  .S19 de motorola                                                 */
/*  Se devuelve el tamao en bytes del programa cargado en memoria   */
/*********************************************************************/
{
  int tipo;
  unsigned short int tam;
  unsigned short int datos[40];
  unsigned short int dir[3];
  unsigned short int inicio[3];
  unsigned int   direccion=0;
  unsigned int   nbytes=0;
  unsigned int   block=0;
  char c;


  baud=9600;
  baudios(baud);
  display_baud(baud);
  printf ("*");
  do {
    if (kbhit()) {
      getch();
      return 0;
    }
    c=leer_car();
  } while (c!='#');


  enviar(0xff);      /* Enviar carcter comienzo transmisin */
  eco(0xff);         /* Detectar eco                         */
  if (hay_error())
       return 0;     /* Error: el MCU no hace eco            */
  do {

    tipo=leer_bloque_s19(f,&tam,datos,dir); /* Leer bloque fichero .S19 */
    if (block==0) {       /* Si es el primer bloque que se manda se       */
      inicio[0]=dir[0];   /* almacena la direccin de comienzo para luego */
      inicio[1]=dir[1];   /* ejecutar el programa.                        */
    }
    if (tipo==0) {  /* Si hay un error en la transmisin del bloque se  */
      return 0;     /* aborta el programa                               */
    }

    if (tipo==1) {  /* Enviar un bloque de datos */
      direccion=256*dir[1]+dir[0];

      enviar(1);      /* Comunicar LOADER se manda bloque de datos  */
      eco(1);

      enviar(dir[0]); /* Enviar byte bajo direccin comienzo bloque */
      eco(dir[0]);
      enviar(dir[1]); /* Enviar byte alto direccin comienzo bloque */
      eco(dir[1]);

      enviar(tam);    /* Enviar tamao del bloque a enviar          */
      eco(tam);

      enviar_cad_eco(datos,tam,0); /* Enviar los datos por el puerto serie */
      nbytes+=(unsigned int)tam;   /* Incrementar n de bytes mandados     */
      block++;

      textcolor(14);
      cprintf ("\b\b\b\b%4u",block);
    }
    if (tipo==9) {
      enviar(2);         /* Comunicar al LOADER que le enviamos la dir.   */
			 /* de inicio del programa, para que lo ejecute   */
      eco(2);
      enviar(inicio[0]); /*  Mandamos byte bajo                           */
      eco(inicio[0]);
      enviar(inicio[1]); /*  Mandamos byte alto                           */
      eco(inicio[1]);

      return nbytes;        /*  El MCU est ejecutando el prog. Terminamos   */
    }
    if (hay_error()) return 0;   /*  Si hay errores abortar transmisin */
  } while (1);
}

void volcar_fichero()
/*********************************************/
/*  Volcar un fichero hacia el puerto serie  */
/*  Se vuelca a la velocidad de 9600 baudios */
/*********************************************/
{
  int f;
  int estado;
  char cad[2];

  textcolor(10);
  cprintf("\n\rVolcar fichero .S19 al micro ");
  if ( (f=abrir_fichero_s19())==0 ) {
    textcolor(12);
    cprintf ("   ---> ABORTADO <---\n\r");
    textcolor(7);
    return;
  }
  cprintf ("\n\r");

  while ((estado=read(f,cad,1))==1) {
    switch(cad[0]) {
      case CR : cprintf("\n\r"); break;
      case LF : break;
      default : enviar(cad[0]);
		cprintf("%c",cad[0]);
		delay(4);
    }
  }
  textcolor(14);
  cprintf ("\n\rVolcado completado\n\r");
  textcolor(7);
  estado=estado;
}


void especial(c)
char c;
/********************************/
/*  Procesar teclas especiales  */
/********************************/
{
  int oldbaud;

  switch(c) {
    case 59 :                     /* TECLA F1 */
	      oldbaud=baud;
	      baud=7680;
	      baudios(baud);
	      display_baud(baud);
	      if (!modo_boot()) {  /* Si no se carga el programa */
		baud=oldbaud;
		baudios(baud);
		display_baud(baud);
	      }
	      break;
    case 60 :                     /* TECLA F2 */
	      oldbaud=baud;
	      baud=7680;
	      baudios(baud);
	      display_baud(baud);
	      if (!modo_expanded()) {
		baud=oldbaud;
		baudios(baud);
		display_baud(baud);
	      } else {
		baud=9600;
		display_baud(baud);
	      }
	      break;
    case 61 :
	      oldbaud=baud;
	      baud=7680;
	      baudios(baud);
	      display_baud(baud);
	      cargar_gaia();      /* TECLA F3 */
	      menu_mcboot();
	      break;
    case 62 : cerrar_puerto_serie();
	      puerto=(puerto % 4)+1;  /* Tecla F4 */
	      configurar_puerto(puerto);
	      baudios(baud);
	      display_com(puerto);
	      dtr=1;
	      break;
    case 63 : switch(baud) {      /* TECLA F5 */
		case 1200 : baud=7680; break;
		case 7680 : baud=9600; break;
		case 9600 : baud=1200; break;

	      }
	      baudios(baud);
	      display_baud(baud);
	      break;
    case 64 :                    /* TECLA F6 */
	      volcar_fichero();
	      break;
    case 65 : if (dtr) {         /* TECLA F7 */
		dtr=0;
		desactivar_dtr();
		cprintf ("DTR OFF\n\r");
	      }else {
		dtr=1;
		activar_dtr();
		cprintf ("DTR ON\n\r");
	      }
	      break;

    case 46 :                    /* TECLA ALT-C*/
	      clrscr();
	      break;
  }
  if (hay_error()) {
       display_error();  /* Si hay error imprimirlo       */
  }

}

void terminal()
/********************************************************/
/*  Terminal. Todos los caracteres tecleados se envan  */
/*  al MC y todos los caracteres recibidos del MC se    */
/*  sacan por pantalla                                  */
/********************************************************/
{
  char c,d;

  do{
    if (kbhit()) {            /* Si se pulsa una tecla...       */
      c=getch();
      if (c==0) {             /* Si se pulsa una tecla especial */
	c=getch();            /* Leer cdigo extendido          */
	especial(c);          /* Actuar en consecuencia         */
      }
      else {
	if (c!=27)
	   enviar(c);         /* y es ditinta de ESC se enva */
      }
    }
    if (car_waiting()) {      /* Si se recibe un carcter  */
      d=leer_car();           /* Se lee y se imprime       */
      cprintf ("%c",d);
    }
    if (c==27) {              /* Si se pulsa ESC           */
      textcolor(11);
      cprintf ("\n\rSalir de MC-BOOT (S/N)? ");
      c=getch();
      cprintf ("\n\r");
      textcolor(7);
      if (c=='S' || c=='s') c=27;
    }
  } while (c!=27);
  window(1,1,80,25);
  clrscr();
  cprintf("MC-BOOT finalizado....\n\r");
  cprintf("MICROBOTICA (microbot@arrakis.es)\n\r");

}


void main()
{
  puerto=2;              /* seleciona puerto por defecto  */
  baud=9600;
  configurar_puerto(2);  /* configura el puerto 2         */
  baudios(9600);
  desactivar_dtr();      /* DTR OFF */
  presenta();
  display_com(puerto);
  display_baud(baud);
  terminal();
  cerrar_puerto_serie();
}