/************************************************************************/
/*                                                                      */
/*  J&J Soft-Hard Company. ETSI Telecomunicacion. MAYO 1995             */
/*                                                                      */
/************************************************************************/
/*                                                                      */
/*  MCBGAIAC. Procesamiento de los comandos del Maestro de GAIA         */
/*                                                                      */
/************************************************************************/

#define MAXINST 20    /* Nmero de instrucciones */

typedef unsigned short int byte;
enum ttoken {tcomando,tregistro,tidentif,tnumero,tfin};

struct token {           /* Estructura de las "tokens" */
  enum ttoken tipo;      /* Tipo de token              */
  byte dato1;            /* Campo con infor. adicional */
  byte dato2;
} t;


char cadcom[30];     /* Cadena que contiene los comandos introducidos  */
		     /* por el usuario                                 */
char identif[30];    /* Para almacenar identificadores                 */

char carsig=' ';     /*  Siguiente carcter al que se ha leido         */
byte ind=0;          /*  Indice a la cadena cadcom                     */


char error[20][255] = {
 {"Comando incorrrecto"},
 {"Direccin de memoria incorrecta"},
 {"Demasiados parmetros"},
 {"Dato incorrecto"},
 {"Nombre de registro incorrecto"},
 {"Numero decimal demasiado grande"},
 {"Valor incorrecto"},
};


char instruccion[MAXINST][10]={     /* Instrucciones disponibles */
  {"MD"},
  {"MS"},
  {"QUIT"},
  {"CLS"},
  {"LOAD"},
  {"CONT"},
  {"OYE?"},
  {"RESTART"},
  {"REGS"},
  {"REG"},
  {"HELP"},
};

char registro[7][4]={
  {"CCR"},
  {"B"},
  {"A"},
  {"X"},
  {"Y"},
  {"PC"},
  {"SP"},
};


char upper(car)
char car;
/*************************************/
/* Para el carcter car a maysculas */
/*************************************/
{
  return (car>='a' && car<='z') ? car-'a'+'A' : car;
}

void tomar_dir(cad,dir)
char *cad;
unsigned short int *dir;
/************************************************************************/
/*  Tomar la direccin que se encuentra en cad. Se introduce en dir[0]  */
/*  el byte de menor peso y en dir[1] el byte de mayor peso             */
/************************************************************************/
{
  cadtoint(cad,&(dir[1]));
  cadtoint(&(cad[2]),&(dir[0]));
}

void print_error(num)
int num;
/**********************************************/
/*  Sacar el mensaje de error correspondiente */
/**********************************************/
{
  textcolor(9);
  cprintf ("\n\r        ---> %s",error[num-1]);
  textcolor(7);

}


char leer_caracter()
/*********************************************************************/
/*  Leer el siguiente carcter de la cadena que contiene el comando  */
/*********************************************************************/
{
  char c;

  c=carsig;
  carsig=upper(cadcom[ind++]);

  return c;
}

byte es_instruccion(cad)
char *cad;
/*********************************************************************/
/*  Comprobar si la cadena introducida es una instruccin para GAIA  */
/*  Si es una instruccin se devuelve un codigo correspondiente al   */
/*  offset de la instruccion en la tabla + 1.                        */
/*                                                                   */
/*    Si no es una instruccion se devuelve 0.                        */
/*********************************************************************/
{
  byte i;

  for (i=0; i<MAXINST; i++)
     if (strcmp(cad,instruccion[i])==0) return i+1;

  return 0;  /* cad no es una instruccion */
}

byte es_registro(cad)
char *cad;
/**********************************************************************/
/*  Comprobar si la cadena introducida es un registro del MCU         */
/*  Si la cadena es un registro se devuelve su numero correspondiente */
/*  En caso contrario se devuelve 0                                   */
/**********************************************************************/
{
  byte i;

  for (i=0; i<7; i++) {
    if (strcmp(cad,registro[i])==0) return i+1;
  }
  return 0;  /* Cad no es un registro */
}


byte es_numero(car)
char car;
/*******************************************************************/
/*  Se devuelve 1 si el carcter introducido es un dgito decimal  */
/*******************************************************************/
{
  return (car<='9' && car>='0') ? 1 : 0;
}

byte es_numero_hex(car)
char car;
/*********************************************************************/
/* Se devuelve 1 si el carcter introducido en un dgito hexadecimal */
/*********************************************************************/
{
  return (es_numero(car) || (car>='A' && car<='F')) ? 1 : 0;
}


void leer_identificador(cad)
char *cad;
/***********************************************************************/
/*  Se lee un identificador de la cadena cadcom y se introduce en la   */
/*  variable cad                                                       */
/***********************************************************************/
{
  char i=0;

  while (carsig!=' ' && carsig!=0) {
    cad[i++]=leer_caracter();
  }
  cad[i]=0;
}

byte leer_binario()
/***********************************************************************/
/*  Leer un nmero en binario. El nmero no puede ser de ms de 8 bits */
/*  Se devuelve 0 si ha habido un error.                               */
/***********************************************************************/
{
  return 0;
}

byte leer_hexadecimal()
/***********************************************************************/
/* Leer un nmero en hexadecimal. El nmero no puede ser de ms de     */
/* 4 dgitos hexadecimales                                             */
/***********************************************************************/
{
  char numero[30];
  char i=0;
  byte num[30];
  byte iold;     /* Para almacenar la variable ind */

  iold=ind-1;

  leer_identificador(numero);

  /* Comprobar que en la cadena todo son dgitos en hexadecimal */
  while (numero[i]!=0)
  {
    if (!hextoint(numero[i],&num[i])) {
      ind=iold;
      strcpy(identif,numero);
      return 0;
      }
      i++;
  }

  if (i>4) {                /* Nmero hexadecimal demasiado grande!! */
    /* Mensaje de error */
    return 0;
  }

  t.tipo=tnumero;

  switch (i) {
    case 1: t.dato1=num[0];
	    t.dato2=0;
	    break;
    case 2: t.dato1=num[0]*16 + num[1];
	    t.dato2=0;
	    break;
    case 3: t.dato1=num[1]*16 + num[2];
	    t.dato2=num[0];
	    break;
    case 4: t.dato1=num[2]*16  + num[3];
	    t.dato2=num[0]*16 + num[1];
	    break;
  }
  return 1;
}

byte leer_numero()
/*********************************************************************/
/*  Leer un nmero de la cadena cadcom e introducirlo en la variable */
/*  de tipo token. Si se produce algn error se devuelve 0.          */
/*  Si no hay errores se devuelve 1.                                 */
/*********************************************************************/
{
  char numero[30];
  int  n;



  if (carsig=='%') {
    leer_caracter();
    return leer_binario();
  }

  if (carsig=='$') {
    leer_caracter();
    return leer_hexadecimal();
  }

  /* Por defecto se leen nmero en decimal */

  leer_identificador(numero);
  n=atoi(numero);
  if (n>255) {
    print_error(6);
    return 0;
  }
  t.dato1=(unsigned short)n;
  t.tipo=tnumero;
}


byte leer_token()
/**************************************************************************/
/*  Leer el siguiente token de la cadena cadcom. Se actualiza la variable */
/*  t, que es de tipo token.                                              */
/*  Se devuelve 0 si ha habido un error en el token leido, 1 en caso      */
/*  contrario.                                                            */
/**************************************************************************/
{
  byte codigo;
  char car;

  t.dato1=0;
  t.dato2=0;

  while ( carsig==' ')
     leer_caracter();         /* Eliminar espacios */

  if (carsig==0) {    /* Si final de la cadena ... */
    t.tipo=tfin;
    ind=0;
    carsig=' ';
    return 1;
  }

  if (carsig=='$' || carsig=='%' || es_numero(carsig)) {
    if (leer_numero()) return 1;
  }
  else leer_identificador(identif);

  if ( (codigo=es_instruccion(identif)) ) {  /* Si es una instruccion.. */
    t.tipo=tcomando;
    t.dato1=codigo;
    return 1;
  }
  if ( (codigo=es_registro(identif)) ) {     /* Si es un registro.. */
    t.tipo=tregistro;
    t.dato1=codigo;
    return 1;
  }

  /* Es un identificador */

  t.tipo=tidentif;

}

void input(cad,max)
char *cad;
unsigned short int max;
/**********************************************************************/
/* Leer una cadena desde el teclado. Max indica el nmero mximo de   */
/* caracteres que se pueden leer                                      */
/**********************************************************************/
{
  char c;
  char i=0;

  do {
    c=getch();
    switch(c) {
      case 8 : if (i>0) {  /* TECLA DEL */
		 i--;
		 cprintf ("\b \b");
	       }
	       break;
      case 13: cad[i]=0;  /* ENTER = fin */
	       return;
      case 0:  getch();    /* Gragarse las "teclas" especiales */
	       break;
      case 27: break;      /* El ESC no hace nada */

      default : if (i<max) {
		  cprintf ("%c",c);
		  cad[i]=c;
		  i++;
		}
    }
  }while (1);
}

byte fin_comando()
/**************************************************************/
/*  Comprobar que el siguiente token es el de fin de comando  */
/**************************************************************/
{

  leer_token();
  if (t.tipo!=tfin) {
    print_error(3);
    return 0;
  }
  else return 1;
}

void comando_md()
/***************/
/* COMANDO MD  */
/***************/
{
  byte dir[2];

  leer_token();

  if (t.tipo!=tnumero) {
    print_error(2);
    return;
  }

  dir[0]=t.dato1;
  dir[1]=t.dato2;

  if (fin_comando())
    volcar_memoria(dir);
}

void comando_ms()
/**************/
/* COMANDO MS */
/**************/
{
  byte dir[2];
  byte dato;

  leer_token();

  if (t.tipo!=tnumero) {
    print_error(2);
    return;
  }
  dir[0]=t.dato1;
  dir[1]=t.dato2;

  leer_token();

  if (t.tipo!=tnumero) {
    print_error(4);
    return;
  }

  dato=t.dato1;

  if (fin_comando())
    load_block(1,&dato,dir);

}

void comando_load()
/*****************/
/* COMANDO LOAD  */
/*****************/
{

  if (fin_comando()) {
    if (load_exec()) {
      cprintf ("\r\n");
      vaciar_buffer();
      display_modo(2);  /* Pasar al modo GAIA */
    }
  }
}

void comando_cls()
/***************/
/* COMANDO CLS */
/***************/
{
  if (fin_comando()) {
    textcolor(7);
    clrscr();
  }
}

void comando_cont()
/****************/
/* COMANDO CONT */
/****************/
{
  if (fin_comando()) {
    kor();
    display_modo(2);  /* Pasar al modo GAIA */
  }
}

void comando_oye()
/****************/
/* COMANDO OYE  */
/****************/
{
  if ( fin_comando()) {
    estas_ahi();
  }
}

void comando_restart()
/*******************/
/* COMANDO RESTART */
/*******************/
{
/*  if (fin_comando()) {
    boot_gaia();
    if (hay_error())
      display_error();
    else
      presenta_gaia();
  } */
}

void comando_regs()
/****************/
/* COMANDO REGS */
/****************/
{
  if (fin_comando()) {
    mostrar_registros();
  }
}

void comando_reg()
/****************/
/* COMANDO REG  */
/****************/
{
  byte reg[2];
  byte dato[2];

  leer_token();
  if (t.tipo!=tregistro) {
    print_error(5);
    return;
  }

  reg[0]=t.dato1-1;
  reg[1]=0;

  leer_token();
  if (t.tipo!=tnumero) {
    print_error(7);
    return;
  }

  dato[1]=t.dato1;
  dato[0]=t.dato2;

  if (fin_comando()) {
    if (reg[0]>2) {
      switch (reg[0]) {
	case 4 : reg[0]=5; break;
	case 5 : reg[0]=7; break;
	case 6 : reg[0]=9; break;
      }
      load_block(2,dato,reg);
    }
    else load_block(1,&dato[1],reg);
  }
}

void comando_help()
/****************/
/* COMANDO REGS */
/****************/
{

  if (!fin_comando()) return;
  cprintf ("\n\r\n");
  textcolor(11); cprintf ("MS dir byte         ");
  textcolor(9);  cprintf ("Introducir un byte en una direccin de memoria\r\n");
  textcolor(11); cprintf ("MD dir              ");
  textcolor(9);  cprintf ("Mostrar un bloque de memoria\r\n");
  textcolor(11); cprintf ("REGS                ");
  textcolor(9) ; cprintf ("Mostrar el valor de los registros del MCU\r\n");
  textcolor(11); cprintf ("REG registro valor  ");
  textcolor(9);  cprintf ("Dar un valor a un registro del MCU\r\n");
  textcolor(11); cprintf ("LOAD                ");
  textcolor(9);  cprintf ("Cargar un programa en memoria\r\n");
  textcolor(11); cprintf ("CONT                ");
  textcolor(9);  cprintf ("Continuar con la ejecucin de un programa interrumpido\r\n");
  textcolor(11); cprintf ("RESTART             ");
  textcolor(9);  cprintf ("Volver a cargar GAIA\r\n");
  textcolor(11); cprintf ("OYE?                ");
  textcolor(9);  cprintf ("Comprobar si GAIA sigue ah o se ha 'colgado'\r\n");
  textcolor(11); cprintf ("CLS                 ");
  textcolor(9);  cprintf ("Borrar la pantalla\r\n");
  textcolor(11); cprintf ("QUIT                ");
  textcolor(9);  cprintf ("Salir de GAIA\r\n");

}


int procesar_comando()
/**************************************/
/* Esperar un comando y procesarlo    */
/* Si el el comando QUIT se retorna 0 */
/**************************************/
{
  input(cadcom,20);   /* Leer una cadena del teclado */
  ind=0;
  carsig=' ';
  leer_token();
  if (t.tipo==tfin) return 1;  /* Instruccin NULA */
  if (t.tipo!=tcomando) {
    print_error(1);
  }
  switch (t.dato1) {      /* Segn el comando introducido */
    case 1: comando_md();
	    break;
    case 2: comando_ms();
	    break;
    case 3: return 0;
    case 4: comando_cls();
	    break;
    case 5: comando_load();
	    break;
    case 6: comando_cont();
	    break;
    case 7: comando_oye();
	    break;
    case 8: comando_restart();
	    break;
    case 9: comando_regs();
	    break;
    case 10: comando_reg();
	     break;
    case 11: comando_help();
	     break;
  }
  return 1;
}