/***********************************************************************/
/*                                                                     */
/* J&J Soft-Hard Company. ETSI Telecomunicacin. Mayo 1995             */
/*                                                                     */
/***********************************************************************/
/*                                                                     */
/*  MCBGAIA. Programa maestro para GAIA. Se encarga de los "dialogos"  */
/*                                                                     */
/*---------------------------------------------------------------------*/
/*                                                                     */
/***********************************************************************/


/***********************************************************************/
/*                                                                     */
/*  LO QUE QUEDA POR HACER:                                            */
/*                                                                     */
/*     *    Modificar LOADER para que las transmisiones sean a 9600    */
/*                                                                     */
/*     *    Comando de ayuda.                                          */
/*                                                                     */
/*     *    Comandos para transferir bloques de datos entre MCU-PC y   */
/*          PC-MCU                                                     */
/*                                                                     */
/*     *    Hacer el programa del "led intermitente" mediante interrup-*/
/*          ciones de tiempo REAL y probarlo en GAIA. Qu ocurre      */
/*          si terminamos con la ejecucin del programa o si ponemos   */
/*          un punto de ruptura?                                       */
/*                                                                     */
/***********************************************************************/


#include "fcntl.h"

unsigned short int inicio[3];
void modo_comando();
void printhex();
void printchar();
int analiza();
void terminal_gaia();

/*********************************************************/
/* RUTINAS DE DIALOGO DIRECTO CON LOS SERVICIOS DE GAIA  */
/*********************************************************/

void load_block(tam,dato,dir)
unsigned short int tam;
unsigned short int *dato;
unsigned short int *dir;
/***********************************************************************/
/*  Cargar un bloque de datos en la direccion de memoria especificada  */
/*                                                                     */
/*  ENTRADAS:                                                          */
/* 		Tam--> Tamao del bloque de datos                      */
/* 		Dato--> Bloque de datos a enviar                       */
/*		dir--> Direccin donde situar el bloque de datos       */
/*                                                                     */
/***********************************************************************/
{
  unsigned char i=0;

  enviar(1);       /* Codigo 1 : Cargar un bloque en memoria */

  enviar(dir[0]);  /* Enviar byte bajo direccin del bloque  */
  enviar(dir[1]);  /* Enviar byte alto direccion del bloque  */
  enviar(tam);     /* Enviar tamao del bloque               */

  while (tam) {       /* Enviar bloque */
    enviar(dato[i]);
    i++;
    tam--;
  }
}

void exec(dir)
unsigned short int *dir;
/****************************************************************/
/*  Ejecutar un programa que se encuentra en la direccin dir   */
/****************************************************************/
{
  enviar(2);      /* Cdigo 2: Ejecutar un programa */
  enviar(dir[0]);
  enviar(dir[1]);

}


void volcar(tam,dato,dir)
unsigned short int tam;
unsigned short int *dato;
unsigned short int *dir;
/************************************************************************/
/*  Volcar un bloque de datos desde el MCU al PC.                       */
/*                                                                      */
/*  ENTRADAS:                                                           */
/*		tam: Tamao del bloque a volcar. (mx 255)              */
/*		dir: Direccion del bloque a volcar                      */
/*  SALIDAS:                                                            */
/*              dato: Buffer donde almacenar los datos recibidos        */
/*                                                                      */
/************************************************************************/
{
  unsigned char i=0;

  enviar(3);	   /* Codigo 3: Volcar memoria del MCU al PC  */
  enviar(dir[0]);  /* Enviar byte bajo direccin del bloque   */
  enviar(dir[1]);  /* Enviar byte alto direccin del bloque   */
  enviar(tam);	   /* Enviar tamao del bloque                */

  while (tam) {
    dato[i]=(unsigned short int)leer_car();
    i++;
    tam--;
  }
}

void kor()
/**********************************************************************/
/*  kor-> Keep on Running. Continuar ejecutando un programa que haba */
/*  sido parado mediante un break-point                               */
/**********************************************************************/
{
  enviar(4);
  terminal_gaia();
}

int oye()
/*******************************************************************/
/* Comprobar si GAIA est ah. Se devuelve 1 si est ahi.          */
/* 0 si no responde                                                */
/*******************************************************************/
{
  vaciar_buffer();
  enviar(5);
  eco('J');
  eco('&');
  eco('J');
  if (hay_error()) {
    clear_error();
    return 0;
  }
  return 1;
}

/*******************************************************************/
/* RUTINAS QUE OFRECEN UN "SERVICIO" DE MAS ALTO NIVEL AL USUARIO  */
/*******************************************************************/

unsigned int load_prog(f)
int f;
/*******************************************************************/
/* Cargar un programa que se encuentra en un fichero en formato    */
/* .S19 de motorola dado por el handle f                           */
/*  Se devuelve el nmero de bytes cargados                        */
/*  El programa SE EJECUTA y el MC-BOOT pasa al modo terminal      */
/*******************************************************************/
{
  int tipo;
  unsigned short int tam;
  unsigned short int datos[40];
  unsigned short int dir[3];
  unsigned int   nbytes=0;
  unsigned int   block=0;


  do {
    tipo=leer_bloque_s19(f,&tam,datos,dir); /* Leer bloque fichero .S19 */
    if (tipo==0) {  /* Si hay un error en la transmisin del bloque se  */
      return 0;     /* aborta el programa                               */
    }
    if (tipo==1){           /* Enviar un bloque de datos */
      if (block==0) {        /* Si es el primer bloque que se manda se       */
	inicio[0]=dir[0];   /* almacena la direccin de comienzo para luego */
	inicio[1]=dir[1];   /* ejecutar el programa.                        */
      }
      load_block(tam,datos,dir);
      nbytes+=(unsigned int)tam;   /* Incrementar n de bytes mandados */
      block++;
    }
    if (tipo==9) {
      exec(inicio);
      terminal_gaia();
      return nbytes;
    }
    if (hay_error()) return 0;   /*  Si hay errores abortar transmisin */
  } while (tipo==1);
  return nbytes;
}


int boot_gaia()
/***************************************/
/*  Cargar a GAIA y retornar           */
/* Se devuelve TRUE si se ha cargado.  */
/* FALSE en caso contrario             */
/***************************************/
{
  int f;

  if (!cargar_loader()) {   /* Cargar el loader */
    if (hay_error()) {
      return 0;
    }
    else return 0;
  }

  if ((f=open("GAIA.S19",O_RDONLY))==-1) {    /* Abrir fichero de GAIA */
     set_error(8);
     return 0;
  }
  textcolor(10);
  cprintf("\n\rBloques transferidos:    ");
  baudios(9600);
  load_expanded(f);     /* Cargar fichero f1 en modo RAM externa */
  close(f);
  return 1;
}

void cargar_gaia()
/*******************************************/
/*  Cargar a GAIA y pasar al modo comando  */
/*******************************************/
{
  if (!boot_gaia()) {
    if (hay_error()) return;
    else return;

  }
  modo_comando();
}


int load_exec()
/*****************************************************************/
/*  Cargar y ejecutar un programa                                */
/*  Se retorna 0 si no se ha cargado el programa correctamente.  */
/*****************************************************************/
{
  int f;

  textcolor(10);
  cprintf("\n\rCargar programa a travs de ");
  textcolor(14);
  cprintf("GAIA");
  if ( (f=abrir_fichero_s19())==0 ){
    textcolor(12);
    cprintf ("   ---> ABORTADO <---");
    textcolor(7);
    return 0;
  }

  return load_prog(f);

}

void gaia_no_esta()
/*****************************************************************/
/* Se imprime un mensaje de error indicando que GAIA no responde */
/*****************************************************************/
{
  textcolor(12);
  cprintf ("\r\n        ---> GAIA NO RESPONDE <---");
  textcolor(7);
}

void estas_ahi()
/************************************/
/*  Preguntar a GAIA si sigue ahi   */
/************************************/
{
  if (oye()) {
    textcolor(11);
    cprintf ("\r\n      SI??");
    textcolor(7);
  }
  else gaia_no_esta();
}

void terminal_gaia()
/********************************************************/
/*  Terminal. Todos los caracteres tecleados se envan  */
/*  al MC y todos los caracteres recibidos del MC se    */
/*  sacan por pantalla                                  */
/********************************************************/
{
  char c,d;

  textcolor(7);
  cprintf ("\r\n");
  display_modo(1);  /* Modo terminal */

  do{
    if (kbhit()) {            /* Si se pulsa una tecla...       */
      c=getch();
      if (c!=27)
	enviar(c);              /* Se enva */
    }
    if (okbreak()) {
      return;
    }
    if (car_waiting()) {      /* Si se recibe un carcter  */
      d=leer_car();           /* Se lee y se imprime       */
      cprintf ("%c",d);
    }
    if (c==27) {              /* Si se pulsa ESC           */
      textcolor(11);
      cprintf ("\n\rVolver al modo comando (S/N)?");
      c=getch();
      textcolor(7);
      if (c=='S' || c=='s') return;
    }
  } while (1);
}


/*****************************************************/
/*  SERVICIOS OFRECIDOS POR EL MC-BOOT AL USUARIO    */
/*****************************************************/

void volcar_memoria(dir)
unsigned short int *dir;
/*********************************************************************/
/*  Volcar un bloque de memoria.                                     */
/*  ENTRADAS:                                                        */
/*               dir : Direccin de la memoria a volcar              */
/*********************************************************************/
{
  unsigned short int buffer[16];
  unsigned int direccion;
  char i;
  char bloque;

  cprintf ("\r\n\n");

  for (bloque=0; bloque<=15; bloque++) {
    volcar(16,buffer,dir);
    textcolor(14);
    printhex(dir[1]);
    printhex(dir[0]);
    textcolor(7);
    cprintf ("  ");
    for (i=0; i<=15; i++) {  /* Escribir valores en hexadecimal */
      printhex(buffer[i]);
      cprintf (" ");
    }
    cprintf ("   ");
    textcolor(9);
    for (i=0; i<=15; i++) {  /* Escribir valores en ASCII       */
      printchar(buffer[i]);
    }
    cprintf("\r\n");
    dir[0]=(dir[0]+16);
    dir[1]=(dir[1]+div(dir[0],256))%256;
    dir[0]=dir[0]%256;

  }
  textcolor(7);

}

void mostrar_registros()
/************************************************/
/*  Sacar los registros del MCU por la pantalla */
/************************************************/
{
  unsigned short int buffer[16];

  /* Los registros se encuentran a partir de la dirreccion $0000 del MCU */
  volcar(11,buffer,0000);
  textcolor(14);
  cprintf ("\n\r             A    B     X      Y      PC     SP   CCR");
  textcolor(7);
  cprintf ("\n\r            $");
  printhex(buffer[2]);  cprintf ("  $");  /* Reg A   */
  printhex(buffer[1]);  cprintf ("  $");  /* Reg B   */
  printhex(buffer[3]);
  printhex(buffer[4]);  cprintf ("  $");  /* Reg X   */
  printhex(buffer[5]);
  printhex(buffer[6]);  cprintf ("  $");  /* Reg Y   */
  printhex(buffer[7]);
  printhex(buffer[8]);  cprintf ("  $");  /* Reg PC  */
  printhex(buffer[9]);
  printhex(buffer[10]); cprintf ("  $");  /* Reg SP  */
  printhex(buffer[0]);                    /* Reg CCR */

}

char digito_hex(num)
unsigned int;
/************************************************************************/
/*  Tomar un nmero entre 0-15 y convertirlo en un dgito hexadecimal   */
/************************************************************************/
{
  if (num<10 && num>=0)   return (char)num+'0';
  if (num<=15 && num>=10) return (char)(num-10)+'A';

  return 'H';   /* Si se retorna esto es que ha habido un error */
}

void printhex(num)
unsigned short int num;
/*************************************/
/* Imprimir un nmero en hexadecimal */
/*************************************/
{
  char hex1,hex2;

  hex1=digito_hex(div(num,16));
  hex2=digito_hex(num % 16);
  cprintf ("%c%c",hex1,hex2);
}

void printchar(num)
unsigned short int num;
/************************************************************/
/*  Imprimir un carcter. Si no es imprimible se saca un .  */
/************************************************************/
{
  if (num<32 || num>126) cprintf (".");
  else cprintf ("%c",(char)num);
}


/***************************************************/
/*  RUTINAS DE CONTROL DEL INTERFAZ MCBOOT-GAIA    */
/***************************************************/

void menu_gaia()
/*************************/
/*  Sacar menu para GAIA */
/*************************/
{
  salva_cursor();
  clear_menu();
  window(1,1,80,25);
  gotoxy(1,25);
  textcolor(10);  cprintf("ESC");
  textcolor(11); cprintf("-Salir del modo terminal  ");
  window(1,3,80,21);

  recupera_cursor();
}

void presenta_gaia()
/*****************************/
/*  Informacin sobre gaia   */
/*****************************/
{
  menu_gaia();
  textcolor(7);
  cprintf("\n\n\rGAIA Operating System. J&J Soft-Hard Company");
  cprintf("\n\rVersion: 1.0  Mayo 1995\n\r");
  display_modo(2);   /* Modo GAIA */

}

void fin_gaia()
/*****************/
/* Fin de gaia   */
/*****************/
{
  textcolor(7);
  cprintf("\n\rFin de Gaia....\n\r\n");
}

void prompt()
/*************************/
/*  Sacar prompt de GAIA */
/*************************/
{
  textcolor(10);
  cprintf("\r\nGaia> ");
  textcolor(7);
}


void modo_comando()
/************************************/
/*  Se esperan comandos del usuario */
/************************************/
{
  int comando=0;
  unsigned short int dir[3];
  unsigned short int byte;

  delay(1000);
  presenta_gaia();
  do {
    if (!oye()) gaia_no_esta();
    prompt();
    comando=procesar_comando();
  } while (comando!=0);
  fin_gaia();
}
