/*
 ͻ
  S19.C       (c) GRUPO J&J. Julio 1997.                                   
 ͹
                                                                           
  Versin 2.0.                                                             
                                                                           
    Rutinas para el procesamiento de ficheros en el formato .S19 de        
  Motorola.                                                                
                                                                           
                                                                           
 ͼ
*/

#define NULL 0

#include "fcntl.h"
#include "s19.h"



/*			 ͻ
			                         
ĺ   I N T E R F A Z      
			                         
			 ͼ                      */


/* TIPO S19

    Este tipo representa a un fichero en formato S19. Las librerias de
    interfaz lo que hacen es crear y manipular variables del tipo S19.
    Este tipo se puede considerar como un tipo abstracto de datos. No
    importa cmo es la estructura interna. Slo hay que saber cmo se
    opera con este tipo.
								       */

int abrir_s19(char *fich,S19 *ss19,int modo);
/*
Ŀ
  Abrir un fichero .S19 y crear una variable de tipo S19. Para trabajar   
  con cualquier fichero .S19 habr que abrirlo primero.                   
                                                                          
   PARAMETROS ENTRADA:                                                    
                                                                          
      - Fich: Nombre del fichero .S19 a abrir. Si se especifica sin       
              extension por defecto se toma extensin.S19. Se puede       
              especificar la ruta completa del fichero.                   
                                                                          
      - Modo: Indica el modo de apertura del fichero. Existen dos modos   
                                                                          
              - Modo 1: Se comprueba el Checksum de todos los registros   
              - Modo 0: No se comprueba el checksum.                      
                                                                          
   PARAMETROS DE SALIDA:                                                  
                                                                          
      - ss19: Se devuelve una variable del tipo S19 que sera necesaria    
              para trabajar con el fichero abierto.                       
                                                                          
   La funcin devuelve 0 si no ha podido abrir el fichero porque se ha    
   producido algn error.                                                 
                                                                          
                                                                          
   Si el fichero ha sido abierto correctamente se devuelve un 1           
                                                                          
*/


void cerrar_s19(S19 ss19);
/*
Ŀ
  Dejar de trabajar con el archivo S19 especificado. Se libera toda la
  memoria que se habia tomado.                                        
*/


int leerdir_s19(S19 ss19,int nreg, unsigned int *dir);
/*
Ŀ
  Leer el campo direccion del fichero S19 especificado.               
                                                                      
                                                                      
  ENTRADAS:                                                           
                                                                      
       - ss19: Fichero S19 abierto con el que trabajar                
       - nreg: Numero del registro al que se quiere acceder           
                                                                      
  SALIDAS:                                                            
                                                                      
       - dir : Campo direccion del registro especificado              
                                                                      
      Se devuelve 0 en caso de que no exista el registro pedido.      
      Se devuelve 1 en caso de exito.                                 
*/


int leercod_s19(S19 ss19,int nreg, unsigned short int *cod, unsigned short int *tam);
/*
Ŀ
  Leer el campo codigo/datos del registro especificado.               
                                                                      
  ENTRADAS:                                                           
                                                                      
     - ss19 : Fichero S19 abierto con el que trabajar                 
     - nreg : Numero del registro al que se quiere acceder            
                                                                      
  SALIDAS:                                                            
                                                                      
     - cod : Campo cdigo/datos del registro especificado             
     - tam : tamao del campo codigo/datos                            
                                                                      
      Se devuelve 0 en caso de que no exista el registro pedido.      
      Se devuelve 1 en caso de exito.                                 
*/


unsigned int getnbytes19(S19 ss19);
/*
Ŀ
  Devolver el numero de bytes de todo el codigo del fichero S19       
  especificado.                                                       
*/

unsigned int getnregs19(S19 ss19);
/*
Ŀ
  Devolver el numero de registros del tipo 1 del fichero S19  
  especificado.                                               
*/


char *geterrors19();
/*
Ŀ
  Devolver la cadena del error producido en la ultima apertura del        
  fichero .S19                                                            
*/


/*		   ͻ
		                                 
ĺ I M P L E M E N T A C I O N  
		                                 
		   ͼ                        */

/*
  Ŀ
    VARIABLES GLOBALES      
  */

int nerror; /* Numero de error que se ha producido */

char error_s19[8][80]={
  {"Archivo no encontrado o error al abrir el archivo"},  /* 1 */
  {"-----------------------------"},                      /* 2 */
  {"No se puede leer el fichero"},                        /* 3 */
  {"Fichero no est en formato .S19"},                    /* 4 */
  {"Checksum errneo"},                                   /* 5 */
  {"No hay memoria suficiente"},                          /* 6 */
  {"Conflicto con las directivas ORG"},			  /* 7 */
};


void extensions19(char *fich)
/*
Ŀ
  Comprobar la extensin del fichero. Si no tiene ninguna extensin se    
  le poner .S19                                                           
*/
{
  int i;

  i=0;
  while (fich[i]!='.' && fich[i]!=0)  /* Recorrer la cadena buscando un . */
    i++;

  if (fich[i]==0) {     /* no tiene extensin. Hay que aadirla  */
    fich[i]='.';
    fich[i+1]='S';
    fich[i+2]='1';
    fich[i+3]='9';
    fich[i+4]=0;
  }
}

char *geterrors19()
/*
Ŀ
  Devolver la cadena del error producido en la ultima apertura del        
  fichero .S19                                                            
*/
{
  if (nerror==0) return NULL;
  return (char *)&error_s19[nerror-1];
}


int getchar2(int f,char *cad)
/*
Ŀ
  Se leen 2 caracteres del fichero especificado y se meten en la cadena   
  cad. Se devuelve 0 si se ha producido algn error.                      
*/
{
  if (read(f,cad,2)==-1) {
    nerror=3;            /* Error leyendo el fichero */
    return 0;
  }
  cad[2]='\0';
  return 1;
}

int getchar4(int f,char *cad)
/*
Ŀ
  Se leen 4 caracteres del fichero especificado y se meten en la cadena   
  cad. Se devuelve 0 si se ha producido algn error.                      
*/
{
  if (read(f,cad,4)==-1) {
    nerror=3;            /* Error leyendo el fichero */
    return 0;
  }
  cad[4]='\0';
  return 1;
}

int getbyte(int f, unsigned short int *num)
/*
Ŀ
  Leer un entero del fichero especificado. Se leen 2 digitos ASCII en     
  hexadecimal y se devuelve su valor entero.                              
                                                                          
   Se devuelve 0 si se ha producido un error.                             
   Se devuelve 1 en caso contrario.                                       
                                                                          
*/
{
  char cad[3];

  if (getchar2(f,cad)==0) {  /* Error leyendo el fichero */
    nerror=3;
    return 0;
  };
  if (char2tobyte(cad,num)==0) {  /* Error al convertir los digitos */
       nerror=4;
       return 0;
  }

  return 1;
}

int getint(int f, unsigned int *num)
/*
Ŀ
  Leer un entero del fichero especificado. Se leen 4 digitos ASCII en     
  hexadecimal y se devuelve su valor entero.                              
                                                                          
   Se devuelve 0 si se ha producido un error.                             
   Se devuelve 1 en caso contrario.                                       
                                                                          
*/
{
  char cad[5];

  if (getchar4(f,cad)==0) {
    nerror=3;                 /* Error leyendo el fichero */
    return 0;
  }
  if (char4toint(cad,num)==0) {
       nerror=4;              /* Error al convertir los digitos */
       return 0;
  }

  return 1;
}

int leertipo_reg(int f, unsigned short int *tipo)
/*
Ŀ
  Leer tipo de registro. Se devuelve en tipo un 1 si se trata de un   
  registro del tipo 1 y 9 en caso de que se trate de registro del     
  tipo 9.                                                             
                                                                      
      Se devuelve 0 si ha habido algn error                          
      Se devuelve 1 si no ha habido errores                           
                                                                      
*/
{
  char cad[3];

  if (getchar2(f,cad)==0) return 0;
  if (cad[0]!='S') {
    nerror=4;                           /* Error en tipo de registro */
    return 0;
  }
  if (cad[1]!='1' && cad[1]!='9') {
    nerror=4;                           /* Error en tipo de registro */
    return 0;
  }

  if (cad[1]=='1') *tipo=1;
  if (cad[1]=='9') *tipo=9;
  return 1;
}

void liberar_s19(S19 *ss19)
/*
Ŀ
  Liberar toda la memoria utilizada por el fichero s19 especificado   
*/
{
  struct registros1 *punt;   /* Punteros a registros S19 */
  struct registros1 *borra;

  punt=ss19->cabeza;  /* Apuntar al comienzo de la lista */
  while(punt!=NULL){
    borra=punt;
    punt=punt->sig;   /* Apuntar al siguiente registro  */
    free(borra);      /* Borrar el registro actual      */
  }
}

int abrir_s19(char *fich,S19 *ss19,int modo)
/*
Ŀ
  Abrir un fichero .S19 y crear una variable de tipo S19. Para trabajar   
  con cualquier fichero .S19 habr que abrirlo primero.                   
                                                                          
   PARAMETROS ENTRADA:                                                    
                                                                          
      - Fich: Nombre del fichero .S19 a abrir. Si se especifica sin       
              extension por defecto se toma extensin.S19. Se puede       
              especificar la ruta completa del fichero.                   
                                                                          
      - Modo: Indica el modo de apertura del fichero. Existen dos modos   
                                                                          
              - Modo 1: Se comprueba el Checksum de todos los registros   
              - Modo 0: No se comprueba el checksum.                      
                                                                          
   PARAMETROS DE SALIDA:                                                  
                                                                          
      - ss19: Se devuelve una variable del tipo S19 que sera necesaria    
              para trabajar con el fichero abierto.                       
                                                                          
   La funcin devuelve 0 si no ha podido abrir el fichero porque se ha    
   producido algn error.                                                 
                                                                          
                                                                          
   Si el fichero ha sido abierto correctamente se devuelve un 1           
                                                                          
*/
{
  char cad[3];
  unsigned int dir;                 /* Campo direccion           */
  unsigned int maxdir;	            /* Mxima direccion alcanzada*/
  unsigned short int tiporeg;       /* Campo tipo de registro    */
  unsigned short int tam;           /* Tamao campo codigo/datos */
  unsigned short int checksum;      /* Campo de checksum         */
  unsigned short int crc;           /* Checksum calculado        */
  unsigned short int codigo[37];    /* Campo de codigo/datos     */

  struct registros1 *nuevoreg;      /* Puntero a registro S1     */
  struct registros1 *regant;        /* Registro anterior         */

  unsigned int i;
  int fi;                 /* Descriptor del fichero de entrada   */

  crc=0;
  maxdir=0;
  ss19->nbytes=0;          /* Inicialiar variable del tipo S19 */
  ss19->nreg=0;
  ss19->cabeza=NULL;
  nerror=0;                /* Inicialmente no hay errores     */


  /*---------- ABRIR EL FICHERO S19 -----*/

  extensions19(fich);  /* Si no tiene extensin poner .S19 */
  if ((fi=open(fich,O_RDONLY))==-1) {
    nerror=1;                            /* Fichero no encontrado */
    liberar_s19(ss19);
    return 0;
  }


  /*------- LEER TIPO DE REGISTRO -------*/

  if (leertipo_reg(fi,&tiporeg)==0) {
    liberar_s19(ss19);                      /* Ha habido algn error */
    return 0;
  }


  while(tiporeg==1) {

    crc=0;

    /*-------- LEER CAMPO TAMAO ---------*/
    if (getbyte(fi,&tam)==0) {
      liberar_s19(ss19);
      return 0;
    }
    crc=tam&0xFF;

    tam-=3; /* Quitar byte de checksum y 2 bytes de direccion    */

    /*-------- LEER CAMPO DIRECCION -------*/
    if (getint(fi,&dir)==0) {
      liberar_s19(ss19);
      return 0;
    }


    if (maxdir>dir){        /* comprobar si la dir. del nuevo reg. */
      nerror=7;             /* no se solapa con las anteriores      */
      liberar_s19(ss19);    /* Si es asi mensaje de error           */
      return 0;
    }
    maxdir=dir+tam;
    crc=crc + (unsigned short int)(dir & 0xFF) +
	      (unsigned short int)(dir>>8)&0xFF;

    /*-------- LEER CAMPO CODIGO/DATOS ----*/
    for (i=0; i<tam; i++) {
      if (getbyte(fi,&codigo[i])==0) {
	liberar_s19(ss19);
	return 0;
      }
      ss19->nbytes++;
      crc= (crc+codigo[i])&0xFF;
    }

    /*-------- LEER CAMPO CRC ----------*/
    if (getbyte(fi,&checksum)==0) {
      liberar_s19(ss19);
      return 0;
    }
    crc= ~crc & 0xFF;
    if (modo==1 && crc!=checksum) {   /* Comprobar el CRC */
      nerror=5;                       /* Error de CRC     */
      liberar_s19(ss19);
      return 0;
    }
    

    /*-------- LEER CR Y LF ------------*/
    if (getchar2(fi,cad)==0) {
      liberar_s19(ss19);
      return 0;
    }

    /* ------- CREAR EL NUEVO REGISTRO S1 LEIDO -------*/
    if ( (nuevoreg=(struct registros1 *)malloc(sizeof(struct registros1)))==NULL) {
      nerror=6;    /* No hay suficiente memoria */
      liberar_s19(ss19);
      return 0;
    }
    ss19->nreg++;
    nuevoreg->tam=tam;                /* Campo tamao       */
    nuevoreg->dir=dir;                /* Campo direccion    */
    for (i=0; i<tam; i++)             /* Campo cdigo/datos */
      nuevoreg->codigo[i]=codigo[i];
    nuevoreg->sig=NULL;
    if (ss19->nreg==1) ss19->cabeza=nuevoreg;  /* Cabeza de la lista */
    else regant->sig=nuevoreg;           /* Enlazar con registro anterior */
    regant=nuevoreg;

    /*------- LEER TIPO DE REGISTRO -------*/
    if (leertipo_reg(fi,&tiporeg)==0) {
      liberar_s19(ss19);                    /* Ha habido algn error */
      return 0;
    }

  }

  close(fi);
  return 1;
}

void cerrar_s19(S19 ss19)
/*
Ŀ
  Dejar de trabajar con el archivo S19 especificado. Se libera toda la
  memoria que se habia tomado.                                        
*/
{
  liberar_s19(&ss19);    /* Liberar memoria               */
}


int leerdir_s19(S19 ss19,int nreg, unsigned int *dir)
/*
Ŀ
  Leer el campo direccion del fichero S19 especificado.               
                                                                      
                                                                      
  ENTRADAS:                                                           
                                                                      
       - ss19: Fichero S19 abierto con el que trabajar                
       - nreg: Numero del registro al que se quiere acceder           
                                                                      
  SALIDAS:                                                            
                                                                      
       - dir : Campo direccion del registro especificado              
                                                                      
      Se devuelve 0 en caso de que no exista el registro pedido.      
      Se devuelve 1 en caso de exito.                                 
*/
{
  struct registros1 *punt;
  int reg;

  reg=1;      /* Comenzar la busqueda por el registro 1 */

  punt=ss19.cabeza;
  while(punt!=NULL) {
    if (reg==nreg) {
      *dir=punt->dir;
      return 1;
    }
    punt=punt->sig;
    reg++;
  }

  return 0;  /* Registro pedido no se encuentra */
}

int leercod_s19(S19 ss19,int nreg, unsigned short int *cod, unsigned short int *tam)
/*
Ŀ
  Leer el campo codigo/datos del registro especificado.               
                                                                      
  ENTRADAS:                                                           
                                                                      
     - ss19 : Fichero S19 abierto con el que trabajar                 
     - nreg : Numero del registro al que se quiere acceder            
                                                                      
  SALIDAS:                                                            
                                                                      
     - cod : Campo cdigo/datos del registro especificado             
     - tam : tamao del campo codigo/datos                            
                                                                      
      Se devuelve 0 en caso de que no exista el registro pedido.      
      Se devuelve 1 en caso de exito.                                 
*/
{
  struct registros1 *punt;
  int reg;
  int i;

  reg=1;

  punt=ss19.cabeza;
  while(punt!=NULL) {
    if (reg==nreg) {
      *tam=punt->tam;             /* Devolver tamao       */
      for (i=0; i<punt->tam; i++)
	cod[i]=punt->codigo[i];   /* Devolver codigo/datos */
      return 1;
    }
    punt=punt->sig;
    reg++;
  }
  return 0;
}

unsigned int getnbytes19(S19 ss19)
/*
Ŀ
  Devolver el numero de bytes de todo el codigo del fichero S19       
  especificado.                                                       
*/
{
  return ss19.nbytes;
}

unsigned int getnregs19(S19 ss19)
/*
Ŀ
  Devolver el numero de registros del tipo 1 del fichero S19  
  especificado.                                               
*/
{
  return ss19.nreg;
}



