/*
   ͻ
                                                                           
    CTDIALOG. GRUPO J&J.                                   MICROBOTICA     
                                                                           
   ͹
                                                                           
     Aplicacin para 'dialogar' con la tarjeta entrenadora CT6811 del      
    grupo J&J. En la CT6811 se debe haber cargado en la ram interna el     
    programa CTSERVER.                                                     
                                                                           
       Es posible utilizar el programa CTDIALOG con otras tarjetas entrena-
    doras que tengan cargado en CTSERVER.                                  
                                                                           
                                                                           
     Versin 2.0. (Sep-97). Los servicios de dialogo ofrecidos son:        
                                                                           
     1.- Volcado de memoria (comando MD)                                   
     2.- Acceso a posiciones de memoria (Comando MS)                       
     3.- Comprobacin de la conexin (Comando CKC)                         
     4.- Ayuda ON-LINE (Comando HELP)                                      
     5.- Comando de finalizacin (QUIT)                                    
     6.- Desensamblador (Comando DASM)                                     
     7.- Grabacin de programas en memoria EEPROM (Comando EEPROM)         
     8.- Carga de programas en memoria RAM (Comando LOAD)                  
     9.- Ejecutar un programa (Comando G)                                  
    10.- Mostrar informacin sobre el microcontrolador (Comando INFO)      
    11.- Borrar la EEPROM del microcontrolador. (Comando BULK)             
    12.- Acceso a posiciones de memoria eeprom (Comando MSE )              
    13.- Grabacin del registro Config. (Comando CONFIG)                   
                                                                           
                                                                           
   ͼ
									   */

const float version = 2.1;

#include<fcntl.h>


/* Ŀ
    Constantes para el dilogo con el servidor CTSERVER   
    */

#define TMPC  65    /* Transferencia de bloques del micro al PC */
#define TPCM  66    /* Transferencia de bloques del PC al micro */
#define JEXE  67    /* Servicio JUMP & EXECUTE                  */
#define ALIVE 68    /* Servicio de supervivencia                */
#define GBEE  69    /* Servicio de grabacion de EEPROM          */
#define OKALIVE 'J'


void sacar_puerto();
void sacar_posicion_eeprom();
void presenta() ;

/*            Ŀ
  Ĵ TIPOS DEFINIDOS  
                            */

typedef unsigned short int byte;

enum tconex {noconex,perdida, establecida};
/* La conexin entre CTDIALOG y CTSERVER se puede encontrar en 3  */
/* estados diferentes: No conexin, conexin perdida y conexin   */
/* establecida.                                                   */


/*            Ŀ
  Ĵ VARIABLES GLOBALES  
                            */

enum tconex estconex = noconex;   /* Estado de la conexin        */
				  /* Inicialmente no hay conexin */

enum tipo_micro {Ax,E2,xx};

int puerto;
int tope_max=0xB7FF,tope_min=0xB600;     /* Localizacion de la EEPROM
										   por defecto                */
enum tipo_micro micro=Ax;                /* Por defecto tenemos un A1 */

/*
   ͻ
                                                                           
             Rutinas de dilogo con el servidor CTSERVER.                  
                                                                           
   ͼ
									   */

int checkalive()
/* Ŀ
    Utilizar el servicio ALIVE proporcionado por el CTSERVER para        
    comprobar si existe comunicacin con el micro.                       
                                                                         
    SALIDAS:                                                             
              0 --> Servidor no responde                                 
              1 --> El servidor ha respondido                            
   */
{
  int intentos=3;  /* Realizar 3 intentos de conexin con el CTSERVER */

  while (intentos>0) {
    delay(10); enviar(ALIVE);
    if (eco(OKALIVE)) return 1;
    intentos--;
  }
  return 0;
}

void volcar(tam,dato,dir)
unsigned short int *tam;
unsigned short int *dato;
unsigned short int *dir;
/* Ŀ
    Utilizar el servicio TMPC del ctserver para volvar un bloque de      
    datos desde el micro al PC.                                          
                                                                         
    ENTRADAS:  tam: Tamao del bloque a volcar                           
               dir: Direccin del bloque a volcar                        
                                                                         
    SALIDAS:   dato: Buffer donde almacenar los datos recibidos          
   */
{
  unsigned char i=0;
  unsigned int tamano;

  delay(10); enviar(TMPC);	              /* Enviar cdigo de servicio solicitado  */
  delay(10); enviar(dir[0]);  /* Enviar byte bajo direccin   */
  delay(10); enviar(dir[1]);  /* Enviar byte alto direccin   */

  delay(10); enviar(tam[0]);   /* Byte bajo long. del bloque */
  delay(10); enviar(tam[1]);   /* Byte alto long. del bloque */

  tamano = 256*tam[1] + tam[0];

  /* Leer el bloque enviado desde el MCU */
  while (tamano>0) {
    dato[i]=(unsigned short int)leer_car();
    i++;
    tamano--;
  }
}

void load_block(tam,dato,dir)
unsigned short int *tam;
unsigned short int *dato;
unsigned short int *dir;
/* Ŀ
    Utilizar el servicio TPCM del ctserver para cargar un bloque de      
    datos del PC en la memoria del micro.                                
                                                                         
    ENTRADAS:  tam: Tamao del bloque a cargar                           
               dir: Direccin del bloque a cargar                        
               dato: Buffer de datos a cargar en la memoria del micro    
   */
{
  unsigned char i=0;
  unsigned int tamano;

  delay(10); enviar(TPCM);
  delay(10); enviar(dir[0]);   /* Byte bajo direccin */
  delay(10); enviar(dir[1]);   /* Byte alto direccin */

  delay(10); enviar(tam[0]);   /* Byte bajo long. del bloque */
  delay(10); enviar(tam[1]);   /* Byte alto long. del bloque */

  tamano=tam[1]*256 + tam[0];
  while (tamano>0) {           /* Enviar bloque */
    delay(10);
    enviar(dato[i]);
    i++;
    tamano--;
  }
}

void jump_execute(dir)
unsigned short int *dir;
/* Ŀ
    Utilizar el servicio JEXE del ctserver para saltar a una direccin   
    de memoria del micro y comenzar a ejecutar el cdigo.                
                                                                         
    ENTRADAS:  tam: Tamao del bloque a cargar                           
               dir: Direccin del bloque a cargar                        
               dato: Buffer de datos a cargar en la memoria del micro    
   */
{
  delay(10);
  enviar(JEXE);   delay(10);
  enviar(dir[0]); delay(10);
  enviar(dir[1]);
}

void grabar_eeprom(int f)
/* Ŀ
    Utilizar el servicio GBEE del ctserver para grabar un programa en la 
    memoria EEPROM del micro.                                            
                                                                         
   */
{
  unsigned int tamano;
  unsigned int i=0;

  byte datos[40];
  byte dir[2];
  byte tam[2];

  unsigned int   direccion=0;
  unsigned int   totalb=0;
  int  tipo;
  int x,y;
  int xtope,xini;

  textcolor(14);
  cprintf ("Transmitiendo: ");
  x=xini=wherex();
  y=wherey();
  xtope=49;

  do {
    tipo=leer_bloque_s19(f,&tam[0],datos,dir);  /* Leer un bloque */

    if (tipo==0) {  /* Si hay un error en la transmisin del bloque se  */
      return ;     /* aborta el programa                               */
    }

    if (tipo==1) {
      totalb+=tam[0];
      direccion=256*dir[1]+dir[0];
      if (direccion>tope_max || direccion<tope_min) {   /* MODIFICADO por */
	textcolor(12);                                      /* ANDRES         */
	cprintf ("\r\n");
	cprintf ("Direccin ");
	printhex(dir[1]); printhex(dir[0]);
	cprintf (": Intento de acceso a memoria no EEPROM\r\n");
	cprintf ("Grabacin no completada\r\n");
	return;
      }

      /* Enviar a la EEPROM el bloque leido */

      tam[1]=0x00;

      delay(10); enviar(GBEE);
      delay(10); enviar(dir[0]);   /* Byte bajo direccin */
      delay(10); enviar(dir[1]);   /* Byte alto direccin */

      delay(10); enviar(tam[0]);   /* Byte bajo long. del bloque */
      delay(10); enviar(tam[1]);   /* Byte alto long. del bloque */


      tamano=tam[1]*256 + tam[0];
      i=0;
      while (tamano>0) {
	delay(20);
	enviar(datos[i]);
	leer_car();
	delay(20);
	i++;
	tamano--;
	gotoxy(x,y);
	x++;
	if (xtope % 2) cprintf ("į");
	else cprintf (" ");
	if (x==xtope-1) {
	  xtope--;
	  x=xini;
	}
      }
    }

  } while (tipo==1);
  textcolor(15);
  cprintf ("\r\n");
  cprintf ("Grabacin terminada\n\r");
  cprintf ("Nmero bytes grabados: %d\n\r",totalb);
  return;

  
}


void cargar_ram(int f)
/* Ŀ
    Utilizar el servicio TPCM del ctserver para cargar un programa en la 
    memoria RAM del micro.                                               
                                                                         
   */
{
  unsigned int tamano;
  unsigned int i=0;

  byte datos[40];
  byte dir[2];
  byte tam[2];

  unsigned int   direccion=0;
  unsigned int   totalb=0;
  int  tipo;
  int x,y;
  int xtope,xini;

  textcolor(14);
  cprintf ("Transmitiendo: ");
  x=xini=wherex();
  y=wherey();
  xtope=68;

  do {
    tipo=leer_bloque_s19(f,&tam[0],datos,dir);  /* Leer un bloque */

    if (tipo==0) {  /* Si hay un error en la transmisin del bloque se  */
      return ;      /* aborta el programa                               */
    }

    if (tipo==1) {
      totalb+=tam[0];
      direccion=256*dir[1]+dir[0];
      if (direccion<=0x00FF) {
	textcolor(12);
	cprintf ("\r\n");
	cprintf ("Direccin ");
	printhex(dir[1]); printhex(dir[0]);
	cprintf (": Intento de acceso a memoria RAM interna\r\n");
	cprintf ("Carga no completada\r\n");
	return;
      }

      /* Enviar al micro el bloque leido */

      tam[1]=0x00;

      delay(10); enviar(TPCM);
      delay(5); enviar(dir[0]);   /* Byte bajo direccin */
      delay(5); enviar(dir[1]);   /* Byte alto direccin */

      delay(5); enviar(tam[0]);   /* Byte bajo long. del bloque */
      delay(5); enviar(tam[1]);   /* Byte alto long. del bloque */


      tamano=tam[1]*256 + tam[0];
      i=0;
      while (tamano>0) {
	delay(5);
	enviar(datos[i]);
	i++;
	tamano--;
	gotoxy(x,y);
	x++;
	if (xtope % 2) cprintf ("į");
	else cprintf (" ");
	if (x==xtope-1) {
	  xtope--;
	  if (xtope==(xini+2)) {
	    xtope=68;
	    gotoxy(69,y);
	    textcolor(7);
	    cprintf ("%u bytes",totalb);
	    textcolor(14);
	  }
	  x=xini;
	}
      }
    }

  } while (tipo==1);
  textcolor(15);
  cprintf ("\r\n");
  cprintf ("Carga terminada\n\r");
  cprintf ("Nmero bytes cargados: %u\n\r",totalb);
  return;

  
}

/*
   ͻ
                                                                           
            SERVICIOS OFRECIDOS POR EL CTDIALOG AL USUARIO                 
                                                                           
   ͼ
									   */

void volcar_memoria(dir,nbloq)
unsigned short int *dir;
unsigned int nbloq;
/* Ŀ
    Volcar un nmero determinado de bloques. Un bloque son 16 bytes.     
                                                                         
    ENTRADAS:  dir: Direccin de la memoria a volcar                     
               nbloq: Nmero de bloques a volcar                         
   */
{
  unsigned short int buffer[16];
  unsigned int direccion;
  unsigned short int tamano[2] = {16,0};
  char i;
  unsigned int bloque;

  dir[0]=dir[0] & 0xF0;	/* Slo se vuelcan bloques de memoria multiplos de 16 */

  cprintf ("\r\n\n");
  textcolor(10);
  cprintf ("      00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F\n\r");

  for (bloque=0; bloque<nbloq; bloque++) {
    volcar(tamano,buffer,dir);
    textcolor(14);
    printhex(dir[1]);
    printhex(dir[0]);
    textcolor(7);
    cprintf ("  ");
    for (i=0; i<=15; i++) {  /* Escribir valores en hexadecimal */
      printhex(buffer[i]);
      cprintf (" ");
    }
    cprintf ("   ");
    textcolor(9);
    for (i=0; i<=15; i++) {  /* Escribir valores en ASCII       */
      printchar(buffer[i]);
    }
    cprintf("\r\n");
    dir[0]=(dir[0]+16);
    dir[1]=(dir[1]+div(dir[0],256))%256;
    dir[0]=dir[0]%256;
  }
  textcolor(7);
  
}





/*
   ͻ
                                                                           
            IMPLEMENTACION DE LOS DIFERENTES COMANDOS                      
                                                                           
   ͼ
									   */

void ejecuta_md(dir,nbloq)
unsigned short int *dir;
unsigned int nbloq;
/* Ŀ
    COMANDO MD: Volcar un bloque de memoria desde el micro al pc         
                Si la conexin no est establecida no se ejecuta.        
                                                                         
    ENTRADAS:  dir: Direccin de la memoria a volcar                     
   */
{
  unsigned int bloque;
  char c;

  bloque=nbloq;

  if (checkalive()) {
    for (bloque=(nbloq/20); bloque>0; bloque--) {
	volcar_memoria(dir,20);
	textcolor(7);
	cprintf ("\n\rPulse una tecla. <ESC> cancelar.\r\n");
	c=getch();
	if (c==27) {
	  cprintf ("\n\rAbortado\n\r");
	  return;
	}
    }
    if (nbloq % 20) volcar_memoria(dir,nbloq % 20);

    textcolor(7);
    if (nbloq==1) cprintf ("\n\r1 bloque volcado\n\r");
    else cprintf ("\n\r%d bloques volcados\n\r",nbloq);
  }
  else printf ("\n\rNo hay conexin\n\r");
}


void ejecutar_ms(dato,dir)
unsigned short int dato;
unsigned short int *dir;
/* Ŀ
    COMANDO MS: Almacenar un byte en una direccin de memoria del micro  
                Si la conexin no est establecida no se ejecuta.        
                                                                         
    ENTRADAS:  dir: Direccin de la memoria a grabar                     
               dato: Dato a enviar                                       
   */
{
  unsigned short int tam[2];

  tam[0]=1;
  tam[1]=0;

  if (checkalive()) {
    load_block(tam,&dato,dir);
  }
  else cprintf ("\n\rNo hay conexin\n\r");
}


void ejecutar_mse(dato,dir)             /* ANDRES !!!!! */
unsigned short int dato;
unsigned short int *dir;
/* Ŀ
    COMANDO MSE: Almacenar un byte en una direccin de memoria EEPROM.   
                Si la conexin no est establecida no se ejecuta.        
                                                                         
    ENTRADAS:  dir: Direccin de la memoria a grabar                     
               dato: Dato a enviar                                       
   */
{
  unsigned int direccion;
  byte tam[2];
  byte buffer[1];

  if (checkalive()) {
      direccion=256*dir[1]+dir[0];
	  if (direccion>tope_max || direccion<tope_min) {
		 textcolor(12);
		 cprintf ("\r\n");
		 cprintf ("Direccin ");
		 printhex(dir[1]); printhex(dir[0]);
		 cprintf (": Intento de acceso a memoria no EEPROM\r\n");
		 cprintf ("Grabacin no completada\r\n");
		 return;
	  }

	  tam[0]=1;
	  tam[1]=0;

	  delay(10); enviar(GBEE);
	  delay(10); enviar(dir[0]);   /* Byte bajo direccin */
	  delay(10); enviar(dir[1]);   /* Byte alto direccin */

      delay(10); enviar(tam[0]);   /* Byte bajo long. del bloque */
      delay(10); enviar(tam[1]);   /* Byte alto long. del bloque */

	  delay(20);
	  buffer[0]=dato;
	  enviar(buffer[0]);

  }
  else cprintf ("\n\rNo hay conexin\n\r");
}


void ejecuta_go(dir)
unsigned short int *dir;
/* Ŀ
    COMANDO GO: Saltar a la direccin especificada.                      
                La conexin con el CTSERVER se pierde                    
                                                                         
    ENTRADAS:  dir: Direccin a donde saltar                             
   */
{
  if (checkalive()) {
    jump_execute(dir);
  }
  else cprintf ("\n\rNo hay conexin\n\r");
}

void ejecuta_dasm(dir)
unsigned short int *dir;
{
  unsigned short int buffer[80];
  unsigned short int tam[2] = {70,0};
  unsigned int tamano,direccion;

  if (checkalive()) {
    volcar(tam,buffer,dir);
    tamano=tam[1]*256+tam[0];
    direccion=dir[1]*256+dir[0];
    dasm(buffer,tamano,direccion);
  }
  else cprintf ("\n\rNo hay conexin\n\r");
}


void ejecutar_ckc()
/* Ŀ
    COMANDO CKC: Comprobar el estado de la conexin.                     
   */
{
  int con;

  cprintf ("\n\r");
  sacar_puerto();
  con=comprobar_conexion();
  if (con) cprintf ("Estado de la conexin: ESTABLECIDA");
  else cprintf ("Estado de la conexin: NO HAY CONEXION");
  cprintf ("\n\r");
}


void ejecutar_info()            /* modificado por Andres !!!!! */
/* Ŀ
    COMANDO INFO: Mostrar informacion sobre el microcontrolador.         
   */
{
  byte tamano[2];
  byte dir[2];
  byte buffer[1];

  dir[0]=0x3F;    /* byte bajo de la direccion */
  dir[1]=0x10;    /* byte alto de la direccion */
  tamano[0]=1;    /* byte bajo del tamao */
  tamano[1]=0;    /* byte alto del tamao */

  presenta();
  if (checkalive()) {
	  textcolor(15);
	  volcar(tamano,buffer,dir);
	  cprintf("\n\r");
	  if ( !(buffer[0]&01) ) {
		  cprintf("bit 0: EEON =0 -> EEPROM desactivada.");
		  cprintf("\r\n");
	  }
	  else {
		  cprintf("bit 0: EEON =1 -> EEPROM activada.");
		  cprintf("\r\n");
	  }
	  if ( !(buffer[0]&02) ) {
		  cprintf("bit 1: ROMON=0 -> ROM desactivada.");
		  cprintf("\r\n");
	  }
	  else {
		  cprintf("bit 1: ROMON=1 -> ROM activada.");
		  cprintf("\r\n");
	  }
	  if ( !(buffer[0]&04) ) {
		  cprintf("bit 2: NOCOP=0 -> COP activado.");
		  cprintf("\r\n");
	  }
	  else {
		  cprintf("bit 2: NOCOP=1 -> COP desactivado.");
		  cprintf("\r\n");
	  }
	  if ( !(buffer[0]&0x08) ) {
		  cprintf("bit 3: NOSEC=0 -> SECURITY activada. Modelo fabricado con la mscara\r\n");
		  cprintf("                  de seguridad\r\n");
	  }
	  else {
		  cprintf("bit 3: NOSEC=1 -> SECURITY desactivada.( Solo disponible en modelos \r\n");
		  cprintf("                  fabricados con la mscara de seguridad ). \r\n");
	  }
	  textcolor(14);
	  cprintf("\r\nRegistro CONFIG: %X ",buffer[0]);
	  cprintf("\n\r");
	  switch (micro) {
		case E2:
		  cprintf("El modelo de micro se corresponde con el E2\r\n");
		  buffer[0]=buffer[0] & 0xF0;  /* me quedo con los bits superiores */
		  tope_min=(buffer[0]+8)*256;
		  tope_max=(buffer[0]+0xF)*256+0xFF;
		  break;
		case Ax :
		  cprintf("El modelo de micro se corresponde con la familia Ax\r\n");
		  tope_min=0xB600; tope_max=0xB7ff;
		  break;
		default:
		  cprintf("El modelo de micro se corresponde familia desconocida \r\n");
		  tope_min=0xB600; tope_max=0xB7ff;
		  break;
	  }

	  sacar_posicion_eeprom();
	  textcolor(7);
  }
  else cprintf ("\n\rNo hay conexin\n\r");
}



void ejecutar_bulk()            /* modificado por Andres !!!!! */
/* Ŀ
    COMANDO BULK: Borrar la EEPROM del microcontrolador.                 
   */
{
  byte tamano[2];
  byte dir1[2];
  byte dir2[2];
  byte buffer[1];

  if (checkalive()) {
				  /* Apunto al registro $103B  */
  dir1[0]=0x3B;   /* byte bajo de la direccion */
  dir1[1]=0x10;   /* byte alto de la direccion */

				  /* Apunto a una zona de la EEPROM */

  dir2[0]=tope_min & 0x00FF;            /* byte bajo de la direccion */
  dir2[1]=(tope_min & 0xFF00) >> 8;     /* byte alto de la direccion */

  tamano[0]=1;    /* byte bajo del tamao */
  tamano[1]=0;    /* byte alto del tamao */

  buffer[0]=0x06;
  load_block(tamano,buffer,dir1);
  load_block(tamano,buffer,dir2);
  buffer[0]=0x07;
  load_block(tamano,buffer,dir1);
  buffer[0]=0x00;
  load_block(tamano,buffer,dir1);
  }
  else cprintf ("\n\rNo hay conexin\n\r");
}


void ejecutar_config(dato)            /* modificado por Andres !!!!! */
unsigned short int dato;
/* Ŀ
    COMANDO CONFIG: Programar el registro CONFIG.                        
   */
{
  byte tamano[2];
  byte dir1[2];
  byte dir2[2];
  byte buffer[1];


  if (checkalive()) {
				  /* Apunto al registro $103B  */
  dir1[0]=0x3B;   /* byte bajo de la direccion */
  dir1[1]=0x10;   /* byte alto de la direccion */

				  /* Apunto al registro CONFIG */
  dir2[0]=0x3F;   /* byte bajo de la direccion */
  dir2[1]=0x10;   /* byte alto de la direccion */

  tamano[0]=1;    /* byte bajo del tamao */
  tamano[1]=0;    /* byte alto del tamao */

		  /* primero borro el registro */

  buffer[0]=0x16;
  load_block(tamano,buffer,dir1);
  load_block(tamano,buffer,dir2);
  buffer[0]=0x17;
  load_block(tamano,buffer,dir1);

		/* Ahora grabo el nuevo valor */

  buffer[0]=0x02;
  load_block(tamano,buffer,dir1);
  buffer[0]=dato;
  load_block(tamano,buffer,dir2);
  buffer[0]=0x03;
  load_block(tamano,buffer,dir1);
  buffer[0]=0x00;
  load_block(tamano,buffer,dir1);

  }
  else cprintf ("\n\rNo hay conexin\n\r");
}


void ejecutar_eeprom(fich)
char *fich;
/* Ŀ
    COMANDO EEPROM: Grabar un programa en la EEPROM.                     
   */
{
  int f;

  if (checkalive()) {
    extension(fich);   /* Aadir extensin si no tiene */

    if ((f=open(fich,O_RDONLY))==-1) {
      textcolor(12);
      cprintf ("\n\r        ---> %s : Fichero no encontrado",fich);
      textcolor(7);
    }
    else {
      cprintf ("\r\n");
      textcolor(9);  cprintf ("Fichero a grabar en memoria EEPROM: ");
      textcolor(7);  cprintf ("%s\r\n",fich);
      grabar_eeprom(f);
      close(f);
    }
  }
  else cprintf ("\n\rNo hay conexin\n\r");
}


void ejecutar_load(fich)
char *fich;
/* Ŀ
    COMANDO LOAD: Cargar un programa en la memoria RAM.                  
   */
{
  int f;

  if (checkalive()) {
    extension(fich);   /* Aadir extensin si no tiene */

    if ((f=open(fich,O_RDONLY))==-1) {
      textcolor(12);
      cprintf ("\n\r        ---> %s : Fichero no encontrado",fich);
      textcolor(7);
    }
    else {
      cprintf ("\r\n");
      textcolor(9);  cprintf ("Fichero a cargar: ");
      textcolor(7);  cprintf ("%s\r\n",fich);
      cargar_ram(f);
      close(f);
    }
  }
  else cprintf ("\n\rNo hay conexin\n\r");
}


void presenta()
{
  cprintf ("\r\n\n");
  textcolor(12); cprintf ("CTDIALOG ");
  textcolor(11); cprintf ("Versin %0.1f ",version);
  textcolor(10);
  cprintf (" GRUPO J&J. MICROBOTICA \n\r");
  textcolor(9);
  cprintf ("Teclee HELP para obtener ayuda\r\n\n");
}


void establecer_conexion()
/* Ŀ
    Establecer la conexin con el servidor CTSERVER que debe estar eje-  
    cutndose en la entrenadora. Segn que exista o no conexin se       
    informa al usuario y se modifica la variable global estconex, que    
    indica el estado de la conexin actual.                              
   */
{
  textcolor(7);
  cprintf ("Estableciendo conexin con tarjeta..... ");
  delay(300);
  if (checkalive()) {
    cprintf("conexin establecida\n\r");
    estconex=establecida;
  }
  else {
    cprintf("CONEXION NO ESTABLECIDA\n\r");
    estconex=noconex;
  }
}

int comprobar_conexion()
/* Ŀ
    Comprobar si existe conexin. Se actualiza la variable global        
    estconex.                                                            
    Se devuelve 0 si no hay conexin. 1 si la hay.                       
   */
{
  int alive;

  alive=checkalive();
  if (estconex==establecida && !alive) {
    estconex=perdida;
    textcolor(7);
    cprintf ("\n\rConexin perdida\n\r");
    return 0;
  }
  if (estconex!=establecida && alive) {
    estconex=establecida;
    cprintf ("\n\rConexin reestablecida\n\r");
    return 1;
  }
  if (alive) return 1;
  else return 0;
}

void prompt()
/* Ŀ
    Imprimir el prompt actual. En caso de que la conexin se haya es-    
    tablecido, el prompt ser >. Si la conexin no se ha establecido el  
    prompt es de la forma *>                                             
   */
{
  textcolor(7);
  if (estconex==establecida) cprintf ("\n\r>");
  else cprintf ("\n\r*>");
}

void analiza_parametros(na,arg)
int na;
char *arg[];
/* Ŀ
    Analizar los parmetros de la lnea de argumentos. Si se encuentran  
    errores se aborta el programa. Se actualizan las siguientes varia-   
    bles:                                                                
                                                                         
    puerto --> Numero del puerto especificado.                           
   */
{
  int i;
  char temp[10];

  if (na==1) return;  /* No hay parmetros. Se toma un puerto por defecto */

  if (na>2) {
    printf ("Demasiados parmetros\n");
    exit(1);
  }

  if (strcmp(arg[1],"-com1")==0) {  /* COM 1 */
    puerto=1;
    return;
  }

  if (strcmp(arg[1],"-com2")==0) {  /* COM 2 */
    puerto=2;
    return;
  }

  printf ("Parmetro incorrecto. Los parmetros permitidos son: \n");
  printf (" -com1 : Utilizar el COM1 (valor por defecto)\n");
  printf (" -com2 : Utilizar el COM2\n");
  exit(0);
}

void sacar_puerto()
/* Ŀ
    Sacar el puerto actual en uso.  
   */
{
  textcolor(7);
  cprintf ("Puerto actual: COM%u\r\n",puerto);
}

void sacar_posicion_eeprom()              /* Aadido por ANDRES  */
/* Ŀ
    Sacar la posicion de la eeprom  
   */
{
  cprintf ("Posicin eeprom: %X - %X \r\n",tope_min,tope_max);
}


void modelo_micro()              /* Aadido por ANDRES  */
/* Ŀ
    Averigua que micro tenemos      
   */
{
  byte tamano[2];
  byte dir[2];
  byte dato_inicial[1];

				  /* Vamos a observar la posicin 'BF64' */
  dir[0]=0x64;    /* byte bajo de la direccion */
  dir[1]=0xBF;    /* byte alto de la direccion */
  tamano[0]=1;    /* byte bajo del tamao */
  tamano[1]=0;    /* byte alto del tamao */

  if (checkalive()) {
	  volcar(tamano,dato_inicial,dir);
	  dato_inicial[0]=dato_inicial[0]&0xFF;
	  switch (dato_inicial[0] ) {
		case 0xb6: micro=Ax; break;
		case 0xf8: micro=E2; break;
		default  : micro=xx; break;
	  }
  }
  else cprintf ("\n\rNo hay conexin\n\r");
}





/*            Ŀ
  Ĵ Programa PRINCIPAL  
                           */

main(argc,argv)
int argc;
char *argv[];
{
  char d;
  int alive;

  /* Inicializar parmetros por defecto */

  puerto =1;   /* Por defecto COM1 */

  analiza_parametros(argc,argv);

  configurar_puerto(puerto);
  baudios(9600);


  vaciar_buffer();
  sacar_puerto();
  establecer_conexion();
  modelo_micro();
  ejecutar_info();

  do {
    vaciar_buffer();
    comprobar_conexion();
    prompt();
  } while(procesar_comando());
  cerrar_puerto_serie();
  textcolor(7);
  cprintf("\r\n");
  if (estconex!=establecida) {
    cprintf ("Programa terminado\r\n");
	cprintf ("\r\nMicrobtica (microbot@arrakis.es)");
  }
  else {
	cprintf ("Conexin terminada\r\n");
	cprintf ("\r\nMicrobtica (microbot@arrakis.es)\r\n");
  }
  return;
}





