-------------------------------------------------------------
-- perif_access.vhd. Feb-2002                              --
-- Ivan Gonzalez                                           --
-- Juan Gonzalez                                           --
-- Licencia GPL.                                           --
-------------------------------------------------------------
-- PROYECTO LABOBOT                                        --
-------------------------------------------------------------
-- TOP-level. Se implementan 4 unidades de PWM y la mßqui- --
-- na de estados para la transferencia de informacion      --
-- entre la CPU y la FPGA.                                 --
--                                                         --
--------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;  
use IEEE.numeric_std.all;		  
use IEEE.std_logic_unsigned.all;

entity perif_access is 
  port (
    -- Señales usadas para el interfaz entre FPGA y CPU	 
    data  : inout std_logic_vector(31 downto 0);
    adr   : in std_logic;
    cs0   : in std_logic;	 
    ws    : in std_logic;
    rs    : in std_logic;	 
    ack   : out std_logic;
    
    -- Señales empleadas para las unidades PWM	
    reset : in std_logic;
    clk   : in std_logic;
    pwm1  : out std_logic;
    pwm2  : out std_logic;
    pwm3  : out std_logic;
    pwm4  : out std_logic);
end perif_access;						   

architecture Final of perif_access is
  type state is (IDLE, RDOK, WROK);
  signal estado     : state;
  signal n_estado   : state;	
  signal load       : std_logic;
  -- Registros de posicion de los servos (Interfaz con CPU)
  signal reg_pos    : std_logic_vector(31 downto 0);
  
  ---------------------------------------------     
  -- Señales especificas de las unidades PWM --
  ---------------------------------------------
  -- Posiciones de los servos
  signal pos1  : std_logic_vector(7 downto 0);
  signal pos2  : std_logic_vector(7 downto 0);
  signal pos3  : std_logic_vector(7 downto 0);
  signal pos4  : std_logic_vector(7 downto 0);
  -- Reloj para el PWM. Funciona a una frecuencia 4 veces 
  -- inferior que la señal de reloj global (clk)		
  signal clk_4 : std_logic;
      	
  ----------------------------------
  -- Declaracion de componentes   --
  ----------------------------------
	
  -- Unidad de PWM	
  component pwm_unit is
    port (pwm_pos  : in std_logic_vector (7 downto 0);  -- Posicion
	  pwm_clk  : in std_logic;                      -- Reloj
	  pwm_reset: in std_logic;                      -- Reset.
	  pwm_o    : out std_logic);                    -- Se±al PWM  
    end component;
	
  -- Divisor de la frecuencia del reloj	
  component prescaler is
    port (pres_clk  : in std_logic;
          pres_reset: in std_logic;
          pres_q    : out std_logic);  
  end component;
	
		
begin
  -------------------------------------
  -- MAQUINA DE ESTADOS DE INTERFAZ  --
  -------------------------------------
	
  process(clk, reset, load, reg_pos) is		   
    variable i : integer;		
  begin									
    if(reset = '1') then  -- Si hay reset...
      estado <= IDLE;            -- Estado inicial
      reg_pos<=(others => '0');  -- Inicializar registro de posicion
    elsif (clk = '1' and clk'event) then  -- Flanco de subida			
      estado <= n_estado;
      if(load = '1') then	
        reg_pos <= data;   -- Lectura de la posicion de los servos
      end if;
    end if;		

    -- Cada grupo de 8 bits del registro de posicion se envia a
    -- una unidad pwm diferente		
    pos1<=reg_pos(7 downto 0);
    pos2<=reg_pos(15 downto 8);
    pos3<=reg_pos(23 downto 16);
    pos4<=reg_pos(31 downto 24);		
  end process;   		

  process(estado, cs0, ws, rs)
  begin
    case estado is
      when IDLE =>
	ack <= '1';
	load <= '0';
	if(cs0 = '0' and ws = '0' and rs = '1') then -- Ciclo de escritura
	  n_estado <= WROK;
	  load <= '1';
	elsif(cs0 = '0' and rs = '0' and ws = '1') then -- Ciclo de lectura
	  n_estado <= RDOK;
	else			 		  
	  n_estado <= IDLE;
	end if;			
      when WROK =>
        ack <= '0';
	load <= '0';
			
	if(cs0 = '1') then
	  n_estado <= IDLE;
	else
	  n_estado <= WROK;
	end if;
      when RDOK =>
	ack <= '0';
	load <= '0';
	if(cs0 = '1') then
	  n_estado <= IDLE;
	else 
	  n_estado <= RDOK;
	end if;	
    end case;				 
  end process;	
	
  process(estado, reg_pos)
  begin
    if(estado = RDOK) then
         data <= reg_pos;
    else data <= (others => 'Z');
    end if;
  end process;
	
-------------------
--  PRESCALER    --
-------------------	
  PRES1: prescaler port map (pres_clk=>clk,
                             pres_reset=>reset,
	                     pres_q=>clk_4);
---------------------
-- UNIDADES DE PWM --
---------------------              
  PWM_U1: pwm_unit port map (pwm_pos => pos1,
                             pwm_reset=>reset,
                             pwm_clk=>clk_4,
  	                     pwm_o=>pwm1); 
			     
  PWM_U2: pwm_unit port map (pwm_pos => pos2,
                             pwm_reset=>reset,
                             pwm_clk=>clk_4,
  	                     pwm_o=>pwm2); 
			     
  PWM_U3: pwm_unit port map (pwm_pos => pos3,
                             pwm_reset=>reset,
                             pwm_clk=>clk_4,
  	                     pwm_o=>pwm3); 
			     
  PWM_U4: pwm_unit port map (pwm_pos => pos4,
                             pwm_reset=>reset,
                             pwm_clk=>clk_4,
  	                     pwm_o=>pwm4); 
				 	 			  
end Final;

