/*
**************************************************************************
* XPUCHO                                     Andrs Prieto-Moreno Torres *
**************************************************************************
* Programa que ofrece un interfaz grfico para programar los movimientos *
* del Perro Robot, llamado por el autor PuchoBot.                        *
**************************************************************************
* 15 de Octubre de 1999                                                  *
**************************************************************************
*/


#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "forms.h"
#include "panel.h"
#include "xpucho.h"
#include "serie.h"

FD_panel  *fd_panel;   // la hacemos global para poder acceder a ella  

#define MODE644 S_IWUSR|S_IRUSR|S_IRGRP|S_IRGRP
#define PERIOD  0.1    // periodo de acceso a la CT6811

struct t_paso {       // definicin del tipo 'paso'
  int   pos_motor[13];
  int   ojo[3];
  float retardo;
};  

// el movimiento es una secuencia de pasos, el orden es cronolgico, por
// eso se puede estructurar como un ARRAY

struct t_paso movimiento[1001] ;  // movimiento <- conjunto de pasos
struct t_paso paso;               // variable_temporal

int actual;  //  contador de pasos

/**************************************************************
* SUBRUTINAS DEL PROGRAMA                                     *
***************************************************************/

int pregunta(char *cadena)
/*...............................................................
  . Subrutina que hace una pregunta y espera a que se pulse el  .
  . Botn 'S' o el 'No'. Una respuesta afirmativa devuelve un  .
  . 1 y una negativa un 0. Como argumento de entrada hay que    .
  . pasar el texto de la pregunta                               .
  ...............................................................
*/
{
  FL_FORM *cuadro;  // creamos una 'form' simple
  FL_OBJECT *si,*no,*temp,*icono;
  
  fl_deactivate_form(fd_panel->panel);	  // desactivamos el panel
  cuadro=fl_bgn_form(FL_UP_BOX,320,120);
    temp=fl_add_box(FL_NO_BOX,160,40,0,0,cadena);
    si= fl_add_button(FL_NORMAL_BUTTON,40,70,80,30,"SI");
    no= fl_add_button(FL_NORMAL_BUTTON,200,70,80,30,"NO");
    icono=fl_add_pixmap(FL_NORMAL_PIXMAP,10,10,50,50,"");	
    fl_set_pixmap_data(icono,inter_xpm);    // ponemos un icono
    fl_set_object_lsize(temp,14);           // letra mayor en el mensaje
    fl_set_object_lsize(no,14);
    fl_set_object_lsize(si,14);    
    fl_set_button_shortcut(no,"Nn",1);    // activamos teclas de salida
    fl_set_button_shortcut(si,"Ss",1);
  fl_end_form();

  fl_ringbell(5);   // lanzo un avisador acstico
  fl_show_form(cuadro,FL_PLACE_CENTER,FL_TRANSIENT,"Pregunta");
 
  temp=fl_do_forms();
  if ( temp==si ) { // Se ha pulsado SI
        fl_hide_form(cuadro);
        fl_activate_form(fd_panel->panel);
	return 1;
  }
  fl_hide_form(cuadro);                    // se ha pulsado NO
  fl_activate_form(fd_panel->panel);       // volvemos a activar el panel
  return 0;
}


void mensaje(char *cadena)
/*...............................................................
  . Subrutina que escribe un mensaje de advertencia.            .
  ...............................................................
*/
{
  FL_FORM *cuadro;  // creamos una 'form' simple
  FL_OBJECT *temp,*cont,*icono;
  
  fl_deactivate_form(fd_panel->panel);	  // desactivamos el panel
  cuadro=fl_bgn_form(FL_UP_BOX,320,120);
    temp=fl_add_box(FL_NO_BOX,160,40,0,0,cadena);
    cont= fl_add_button(FL_RETURN_BUTTON,95,70,130,30,"CONTINUAR");
    icono=fl_add_pixmap(FL_NORMAL_PIXMAP,10,10,50,50,"");	
    fl_set_pixmap_data(icono,mensa_xpm);    // ponemos un icono
    fl_set_object_lsize(temp,14);           // letra mayor en el mensaje
    fl_set_object_lsize(cont,14);
  fl_end_form();

  fl_ringbell(5);   // lanzo un avisador acstico
  fl_show_form(cuadro,FL_PLACE_CENTER,FL_TRANSIENT,"Mensaje");
 
  temp=fl_do_forms();                   // esperamos para continuar
  fl_hide_form(cuadro);    
  fl_activate_form(fd_panel->panel);    // volvemos a activar el panel
}

int pide_limites_ciclo(int *inf,int *sup,int *inter)
/*...............................................................
  . Subrutina que pide los parmetros del ciclo de repeticin   .
  . Estos son el paso inicial, el final y el nmero de etapas   .
  . entre paso y paso. Por defecto los valores son 0,1000 y 0   .
  . respectivamente,                                            .                                         
  . Si se cancela se devuelve un 0, en caso contrario un 1      .
  ...............................................................
*/
{
  FL_FORM *cuadro;  // creamos una 'form' simple
  FL_OBJECT *obj,*n_pasos,*l_superior,*l_inferior;
  FL_OBJECT *b_aceptar,*b_cancelar,*icono;
  
  fl_deactivate_form(fd_panel->panel);	  // desactivamos el panel

  cuadro = fl_bgn_form(FL_NO_BOX, 420, 320);
    obj = fl_add_box(FL_UP_BOX,0,0,420,320,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_DARKTOMATO);
    obj = fl_add_frame(FL_ENGRAVED_FRAME,0,70,420,190,"");
    obj=l_superior= fl_add_valslider(FL_HOR_BROWSER_SLIDER,160,150,200,30,"");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_slider_precision(obj, 0);
    fl_set_slider_bounds(obj, 0, 1000);
    fl_set_slider_value(obj, *sup);
    fl_set_slider_increment(obj, 1, 0);
    obj = fl_add_text(FL_NORMAL_TEXT,20,100,110,30,"Posicin inicial");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHEAT);
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    obj=n_pasos= fl_add_valslider(FL_HOR_BROWSER_SLIDER,160,200,200,30,"");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_slider_precision(obj, 0);
    fl_set_slider_bounds(obj, 1, 25);
    fl_set_slider_value(obj, *inter);
    fl_set_slider_increment(obj, 1, 0);
    obj = fl_add_text(FL_NORMAL_TEXT,20,200,130,30,"Nmero subpasos");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHEAT);
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    obj=l_inferior = fl_add_valslider(FL_HOR_BROWSER_SLIDER,160,100,200,30,"");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_slider_precision(obj, 0);
    fl_set_slider_bounds(obj, 0, 1000);
    fl_set_slider_value(obj, *inf);
    fl_set_slider_step(obj, 1);
    fl_set_slider_increment(obj, 1, 0);
    obj = b_cancelar=fl_add_button(FL_NORMAL_BUTTON,230,270,100,30,"Cancelar");
    fl_set_object_lsize(obj,FL_LARGE_SIZE);
    fl_set_button_shortcut(obj,"Cc",1);
    icono = fl_add_pixmap(FL_NORMAL_PIXMAP,10,10,60,50,"");
    fl_set_pixmap_data(icono,cicli_xpm);
    obj = fl_add_text(FL_NORMAL_TEXT,90,20,310,30,"Reproduccin cclica");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHEAT);
    fl_set_object_lsize(obj,FL_HUGE_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    obj = b_aceptar=fl_add_button(FL_NORMAL_BUTTON,90,270,100,30,"Aceptar");
    fl_set_object_lcolor(obj,FL_RIGHT_BCOL);
    fl_set_object_lsize(obj,FL_LARGE_SIZE);
    fl_set_button_shortcut(obj,"Aa",1);
    obj = fl_add_text(FL_NORMAL_TEXT,20,150,100,30,"Posicin final");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcolor(obj,FL_WHEAT);
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
  fl_end_form();

  fl_show_form(cuadro,FL_PLACE_CENTER,FL_TRANSIENT,"Reproduccin cclica");
 
  do {
   obj=fl_do_forms();                   // esperamos para continuar
   if (obj==l_superior) { *sup=fl_get_slider_value(obj);  }
   if (obj==l_inferior) { *inf=fl_get_slider_value(obj);  }
   if (obj==n_pasos)    { *inter=fl_get_slider_value(obj); }
   if (obj==b_cancelar) { 
     fl_hide_form(cuadro);    
     fl_activate_form(fd_panel->panel);  
     return 0; }
  } while (obj!=b_aceptar);

  fl_hide_form(cuadro);    
  fl_activate_form(fd_panel->panel);    // volvemos a activar el panel
  return 1;
}


void actualizar_paso(struct t_paso act)
/*..............................................................
  . Recibe como parmetro el paso  actual y coloca los motores .
  . ,el valor del contador del tiempo, y los ojos .            .
  ..............................................................
*/
{
  fl_set_slider_value(fd_panel->motor_a1,act.pos_motor[1]);
  fl_call_object_callback(fd_panel->motor_a1);
  fl_set_slider_value(fd_panel->motor_a2,act.pos_motor[2]);
  fl_call_object_callback(fd_panel->motor_a2);
  fl_set_slider_value(fd_panel->motor_a3,act.pos_motor[3]);
  fl_call_object_callback(fd_panel->motor_a3);
  fl_set_slider_value(fd_panel->motor_a4,act.pos_motor[4]);
  fl_call_object_callback(fd_panel->motor_a4);
  fl_set_slider_value(fd_panel->motor_b1,act.pos_motor[5]);
  fl_call_object_callback(fd_panel->motor_b1);
  fl_set_slider_value(fd_panel->motor_b2,act.pos_motor[6]);
  fl_call_object_callback(fd_panel->motor_b2);
  fl_set_slider_value(fd_panel->motor_b3,act.pos_motor[7]);
  fl_call_object_callback(fd_panel->motor_b3);
  fl_set_slider_value(fd_panel->motor_b4,act.pos_motor[8]);
  fl_call_object_callback(fd_panel->motor_b4);
  fl_set_slider_value(fd_panel->motor_c1,act.pos_motor[9]);
  fl_call_object_callback(fd_panel->motor_c1);
  fl_set_slider_value(fd_panel->motor_c2,act.pos_motor[10]);
  fl_call_object_callback(fd_panel->motor_c2);
  fl_set_slider_value(fd_panel->motor_c3,act.pos_motor[11]);
  fl_call_object_callback(fd_panel->motor_c3);
  fl_set_slider_value(fd_panel->motor_c4,act.pos_motor[12]);
  fl_call_object_callback(fd_panel->motor_c4);

  fl_set_button(fd_panel->ojo_a1, 1-((act.ojo[1] & 0x40) ==0x40));
  fl_set_button(fd_panel->ojo_a2, 1-((act.ojo[1] & 0x80) ==0x80));
  fl_set_button(fd_panel->ojo_a3, 1-((act.ojo[1] & 0x10) ==0x10));
  fl_set_button(fd_panel->ojo_a4, 1-((act.ojo[1] & 0x20) ==0x20));
  fl_set_button(fd_panel->ojo_a5, 1-((act.ojo[1] & 0x04) ==0x04));
  fl_set_button(fd_panel->ojo_a6, 1-((act.ojo[1] & 0x08) ==0x08));
  fl_set_button(fd_panel->ojo_a7, 1-((act.ojo[1] & 0x01) ==0x01));
  fl_set_button(fd_panel->ojo_a8, 1-((act.ojo[1] & 0x02) ==0x02));
  fl_call_object_callback(fd_panel->ojo_a8); // actualizacion

  fl_set_button(fd_panel->ojo_b1, 1-((act.ojo[2] & 0x40) ==0x40));
  fl_set_button(fd_panel->ojo_b2, 1-((act.ojo[2] & 0x80) ==0x80));
  fl_set_button(fd_panel->ojo_b3, 1-((act.ojo[2] & 0x10) ==0x10));
  fl_set_button(fd_panel->ojo_b4, 1-((act.ojo[2] & 0x20) ==0x20));
  fl_set_button(fd_panel->ojo_b5, 1-((act.ojo[2] & 0x04) ==0x04));
  fl_set_button(fd_panel->ojo_b6, 1-((act.ojo[2] & 0x08) ==0x08));
  fl_set_button(fd_panel->ojo_b7, 1-((act.ojo[2] & 0x01) ==0x01));
  fl_set_button(fd_panel->ojo_b8, 1-((act.ojo[2] & 0x02) ==0x02));
  fl_call_object_callback(fd_panel->ojo_b8); // actualizacion

  fl_set_counter_value(fd_panel->boton_tiempo,act.retardo);
  fl_call_object_callback(fd_panel->boton_tiempo);
}



void inicializar()
/*...........................................................
  . Inicializa los motores, deslizadores,   contadores del  .
  . programa.                                               .
  ...........................................................
*/
{
  int i,j;
  
  // ponemos los motores en un estado inicial

    paso.pos_motor[1]=193;
    paso.pos_motor[2]=151;
    paso.pos_motor[3]=94;
    paso.pos_motor[4]=151;
    paso.pos_motor[5]=23;
    paso.pos_motor[6]=103;
    paso.pos_motor[7]=126;
    paso.pos_motor[8]=76;
    paso.pos_motor[9]=211;
    paso.pos_motor[10]=108;
    paso.pos_motor[11]=103;
    paso.pos_motor[12]=37;

  // inicializa el valor del retardo entre pasos
  
  paso.retardo=0.3;
  
  // inicializo el valor del contador de pasos
	
  actual=0;
  fl_set_counter_value(fd_panel->boton_pasos,actual);

  // inicializa el valor de los ojos
  
  paso.ojo[1]=0x0;
  paso.ojo[2]=0x0;

  // actualizo los cambios
  
  actualizar_paso(paso);

  // borra la base de datos
  
  for (i=0; i<=1000; i++) {
    for ( j=1; j<=12; j++ ) {
      movimiento[i].pos_motor[j]=paso.pos_motor[j];
    }
    movimiento[i].retardo=0.3;
    movimiento[i].ojo[1]=0x0;
    movimiento[i].ojo[2]=0x0;
  }
}  
 

void abrir_fichero()
{
   const char *fname;
   int fd;  // descriptor del fichero
   
   fl_set_fselector_fontsize(14);
   fl_set_fselector_title("Abrir fichero");
   fname=fl_show_file_selector("Abrir fichero","","*.bot","");
   if (fname==NULL) {return;}
   fd=open(fname,O_RDONLY,MODE644);
   if (fd==-1) {
     mensaje("Error al abrir el archivo");
     return;
   }
   read(fd,&movimiento,sizeof(movimiento));
   if (fd==-1) {
     mensaje("Error al leer el archivo");
     close(fd);
     return;
   }
   close(fd);   
}


void guardar_fichero()
/*................................................................
  . Rutina que guarda un fichero con la secuencia de movimientos .
  ................................................................*/
{
   const char *fname;  // nombre del fichero
   int   fd;           // descriptor del fichero
   int   temp;
   
   fl_set_fselector_fontsize(14);
   fl_set_fselector_title("Guardar fichero");
   fname=fl_show_file_selector("Guardar fichero","","*.bot","");
   if (fname==NULL) {return;}
   fd=open(fname,O_WRONLY|O_CREAT|O_EXCL,MODE644);
   if (fd==-1) {
     if (pregunta(" Sobreescribir archivo?")==0) { return;} 
     fd=open(fname,O_WRONLY|O_CREAT|O_TRUNC,MODE644);     
   }
   if (fd==-1) {
     mensaje("Error al abrir el archivo");
     return;
   }
   write(fd,&movimiento,sizeof(movimiento));
   if (fd==-1) {
     mensaje("Error al escribir el archivo");
     close(fd);
     return;
   }
   close(fd);   
}


void convertir_fichero()
/*.................................................................
  . Rutina que guarda la secuencia de movimientos en formato ASM  .
  .................................................................
  . movixxx FCB fa1,fa2,fa3,fa4,fb1,fb2,fb3,fb4,                  .
  .             fc1,fc2,fc3,fc4,ojoa,ojob,retardo                 . 
  .................................................................*/
{
   const char *fname;  // nombre del fichero
   int   fd;           // descriptor del fichero
   char  temp[73];
   int   i=0;  
   
   fl_set_fselector_fontsize(14);
   fl_set_fselector_title("Convertir fichero a ensamblador");
   fname=fl_show_file_selector("Convertir fichero","","*.asm","");
   if (fname==NULL) {return;}
   fd=open(fname,O_WRONLY|O_CREAT|O_EXCL,MODE644);
   if (fd==-1) {
     if (pregunta(" Sobreescribir archivo?")==0) { return;} 
     fd=open(fname,O_WRONLY|O_CREAT|O_TRUNC,MODE644);     
   }
   if (fd==-1) {
     mensaje("Error al abrir el archivo");
     return;
   }
   for (i=0;i<1000;i++) {
     sprintf(temp,"movi%-3i FCB %3i,%3i,%3i,%3i,%3i,%3i,%3i,%3i,%3i,%3i,%3i,%3i,$%-2X,$%-2X,%-3.0f\n",
             i,movimiento[i].pos_motor[1],movimiento[i].pos_motor[2],
             movimiento[i].pos_motor[3],movimiento[i].pos_motor[4],
             movimiento[i].pos_motor[5],movimiento[i].pos_motor[6],
             movimiento[i].pos_motor[7],movimiento[i].pos_motor[8],
             movimiento[i].pos_motor[9],movimiento[i].pos_motor[10],
             movimiento[i].pos_motor[11],movimiento[i].pos_motor[12],
             movimiento[i].ojo[1],movimiento[i].ojo[2],movimiento[i].retardo*10);
     write(fd,&temp,sizeof(temp)-1);
     if (fd==-1) {
       mensaje("Error al escribir el archivo");
       close(fd);
       return;
     }
   }  
   close(fd);   
}


void enviar_trama(char placa,char comando,char dato1, char dato2)
/*..................................................................
  . Envia una trama a la RED de BT6811. Esta trama se enva por el .
  . puerto serie y el programa de la CT6811 lo reenva a travs del.
  . bus SPI, al que estn conectadas el resto de los mdulos de    .
  . control (BT6811s)                                              .
  ..................................................................*/
{
  enviar_car(placa);
  enviar_car(comando);
  enviar_car(dato1);
  enviar_car(dato2);
}  


/******************************************************************
 * FUNCIONES CALLBACKS UTILIZADAS POR LOS OBJECTOS DE LAS XFORMS  *
 ******************************************************************/
 
void salir_programa()
{
  if ( pregunta("Salir del programa ?")==1 ) { 
    fl_finish();
    cerrar_puerto_serie();
    exit(0);
  }
}	  


void boton_pulsado(FL_OBJECT *obj, long arg)
/*.....................................................................
  . Los botones de la consola llaman a esta rutina, de forma que cada .
  . vez que se pulsa uno hay que ver cual ha sido para poder realizar .
  . la correspondiente accin.                                        .
  .....................................................................*/  
{
    int cont,temp,i=0;
    
       
   /* boton grabar */
   
   if (obj==fd_panel->b_grabar) {
     movimiento[actual]=paso;    // guardo los valores
     actual++;                    // aumenta el registro
     if (actual>1000)  actual=1000; 
      ;
     fl_set_counter_value(fd_panel->boton_pasos,actual);
   }    
   
   /* boton borrar */
   
   if (obj==fd_panel->b_borrar) {
     i=actual;
     do {
       movimiento[i]=movimiento[i+1];    // guardo los valores
       i++; 
     } while (i<1000) ;      
     fl_set_counter_value(fd_panel->boton_pasos,actual);
     actualizar_paso(movimiento[actual]);
   }    

   /* boton insertar */
   
   if (obj==fd_panel->b_insertar) {
     i=1000;
     do {
       movimiento[i]=movimiento[i-1];    // guardo los valores
       i--; 
     } while (i>actual) ;      
     fl_set_counter_value(fd_panel->boton_pasos,actual);
     actualizar_paso(movimiento[actual]);
   }    

   /* boton repro */
   
   if (obj==fd_panel->b_repro) {
     temp=actual;
     for (i=actual; i<=1000; i++) {
       fl_set_counter_value(fd_panel->boton_pasos,i);
       fl_call_object_callback(fd_panel->boton_pasos);
       cont=movimiento[i].retardo*1000000;
       usleep(cont);
       if (fl_check_forms()==fd_panel->b_parar) i=1000;   // parar
     }
     actual=temp;
     fl_ringbell(5);
     fl_set_button(fd_panel->b_repro,0);
     fl_set_counter_value(fd_panel->boton_pasos,actual);
     fl_call_object_callback(fd_panel->boton_pasos);
   }

   /* boton atras */
   
   if (obj==fd_panel->b_atras) {
     temp=actual;
     for (i=actual; i>=0; i--) {
       fl_set_counter_value(fd_panel->boton_pasos,i);
       fl_call_object_callback(fd_panel->boton_pasos);
       cont=movimiento[i].retardo*1000000;
       usleep(cont);
       if (fd_panel->b_parar==fl_check_forms()) i=0;  // parar
     }
     actual=temp;
     fl_ringbell(5);
     fl_set_button(fd_panel->b_atras,0);
     fl_set_counter_value(fd_panel->boton_pasos,actual);
     fl_call_object_callback(fd_panel->boton_pasos);
   }

   /* boton off */
   
   if (obj==fd_panel->b_off) {
     fl_set_button(fd_panel->MA1,0);
     fl_set_button(fd_panel->MA2,0);
     fl_set_button(fd_panel->MA3,0);
     fl_set_button(fd_panel->MA4,0);
     fl_set_button(fd_panel->MB1,0);
     fl_set_button(fd_panel->MB2,0);
     fl_set_button(fd_panel->MB3,0);
     fl_set_button(fd_panel->MB4,0);
     fl_set_button(fd_panel->MC1,0);
     fl_set_button(fd_panel->MC2,0);
     fl_set_button(fd_panel->MC3,0);
     fl_set_button(fd_panel->MC4,0);
     fl_call_object_callback(fd_panel->MC4); // actualizacion
   }    

   /* boton on */
   
   if (obj==fd_panel->b_on) {
     fl_set_button(fd_panel->MA1,1);
     fl_set_button(fd_panel->MA2,1);
     fl_set_button(fd_panel->MA3,1);
     fl_set_button(fd_panel->MA4,1);
     fl_set_button(fd_panel->MB1,1);
     fl_set_button(fd_panel->MB2,1);
     fl_set_button(fd_panel->MB3,1);
     fl_set_button(fd_panel->MB4,1);
     fl_set_button(fd_panel->MC1,1);
     fl_set_button(fd_panel->MC2,1);
     fl_set_button(fd_panel->MC3,1);
     fl_set_button(fd_panel->MC4,1);
     fl_call_object_callback(fd_panel->MC4); // actualizacion
   }    

   /* boton tiempo */

   if (obj==fd_panel->boton_tiempo) {
     paso.retardo=fl_get_counter_value(fd_panel->boton_tiempo);
   }

   /* boton pasos */
   
   if (obj==fd_panel->boton_pasos) {
     actual=fl_get_counter_value(fd_panel->boton_pasos);
     actualizar_paso(movimiento[actual]);
   }
}


void poner_posicion(FL_OBJECT *obj, long arg)
/*.....................................................................
  .  Rutina que coloca el robot en una posicin determinada           .
  .  Cada botn de la barra inferior selecciona una posicin distinta .
  .....................................................................*/
{

   int i;
   int temp[13];
   
   /* posicion levantado */
   
   if (obj==fd_panel->levantado) {
     temp[1]=193; temp[2]=151;  temp[3]=94;  temp[4]=151;
     temp[5]=23;  temp[6]=103;  temp[7]=126; temp[8]=76;
     temp[9]=211; temp[10]=108; temp[11]=103; temp[12]=37;
   }    
   
   /* posicion sentado */
   
   if (obj==fd_panel->sentado) {
     temp[1]=193;  temp[2]=151;  temp[3]=94; temp[4]=16;
     temp[5]=23; temp[6]=103; temp[7]=126; temp[8]=64;
     temp[9]=211;  temp[10]=106; temp[11]=238; temp[12]=37;
   }    

   /* posicion tumbado */
   
   if (obj==fd_panel->tumbado) {
     temp[1]=193; temp[2]=151;  temp[3]=184;  temp[4]=16;
     temp[5]=23 ; temp[6]=103;  temp[7]=16;   temp[8]=64;
     temp[9]=211; temp[10]=106; temp[11]=238; temp[12]=37;
   }    
   
   /* posicion equilibrio A */
   
   if (obj==fd_panel->equilibrio_a) {
     temp[1]=185; temp[2]=151;  temp[3]=94;   temp[4]=151;
     temp[5]=15;  temp[6]=103;  temp[7]=126;  temp[8]=76;
     temp[9]=218; temp[10]=108; temp[11]=103; temp[12]=44;
   }    

   /* posicion equilibrio B */
   
   if (obj==fd_panel->equilibrio_b) {
     temp[1]=193; temp[2]=151;  temp[3]=94;   temp[4]=151;
     temp[5]=29 ; temp[6]=103;  temp[7]=126;  temp[8]=76;
     temp[9]=203; temp[10]=108; temp[11]=103; temp[12]=30;
   }    
   
   /* posicion equilibrio C */
   
   if (obj==fd_panel->equilibrio_c) {
     temp[1]=119; temp[2]=90; temp[3]=138; temp[4]=16;
     temp[5]=67; temp[6]=167; temp[7]=67; temp[8]=64;
     temp[9]=211;  temp[10]=106; temp[11]=238; temp[12]=37;
   }    

   /* posicion equilibrio D */
   
   if (obj==fd_panel->equilibrio_d) {
     temp[1]=193;  temp[2]=110; temp[3]=53; temp[4]=151;
     temp[5]=23; temp[6]=103; temp[7]=126;   temp[8]=76;
     temp[9]=211;  temp[10]=62; temp[11]=154; temp[12]=25;
   }    

   for (i=1;i<=12;i++) {
     paso.pos_motor[i]=temp[i];
   }
   actualizar_paso(paso);
}



void camara_lenta(FL_OBJECT *obj,long arg)
/*..............................................................
  . Rutina que repite una secuencia de manera ciclica.         .
  . Se puden elegir el principio y el fin del ciclo y los pasos.
  . entre pasos consecutivas.                                 .
  ..............................................................*/
{
  static int inf,sup,inter;     // variables usadas en este trozo
  int    temp,i,j,k,cont,fin=0;
  int    paso[13]={0,0,0,0,0,0,0,0,0,0,0,0,0};
     
  if (pide_limites_ciclo(&inf,&sup,&inter)==0) {return;};
  if (sup>inf) {
    temp=actual;
    do {
      for (i=inf; i<sup; i++) {
        fl_set_counter_value(fd_panel->boton_pasos,i);
        fl_call_object_callback(fd_panel->boton_pasos);
        cont=movimiento[i].retardo*1000000;
        usleep(cont);

        for (k=1; k<inter;k++) {
           paso[1]=paso[1]+(movimiento[i+1].pos_motor[1]-
                            movimiento[i].pos_motor[1])/inter;
           fl_set_slider_value(fd_panel->motor_a1,movimiento[i].pos_motor[1]+paso[1]);
           fl_call_object_callback(fd_panel->motor_a1);

           paso[2]=paso[2]+(movimiento[i+1].pos_motor[2]-
                            movimiento[i].pos_motor[2])/inter;
           fl_set_slider_value(fd_panel->motor_a2,movimiento[i].pos_motor[2]+paso[2]);
           fl_call_object_callback(fd_panel->motor_a2);

           paso[3]=paso[3]+(movimiento[i+1].pos_motor[3]-
                            movimiento[i].pos_motor[3])/inter;
           fl_set_slider_value(fd_panel->motor_a3,movimiento[i].pos_motor[3]+paso[3]);
           fl_call_object_callback(fd_panel->motor_a3);

           paso[4]=paso[4]+(movimiento[i+1].pos_motor[4]-
                      movimiento[i].pos_motor[4])/inter;
           fl_set_slider_value(fd_panel->motor_a4,movimiento[i].pos_motor[4]+paso[4]);
           fl_call_object_callback(fd_panel->motor_a4);

           paso[5]=paso[5]+(movimiento[i+1].pos_motor[5]-
                            movimiento[i].pos_motor[5])/inter;
           fl_set_slider_value(fd_panel->motor_b1,movimiento[i].pos_motor[5]+paso[5]);
           fl_call_object_callback(fd_panel->motor_b1);

           paso[6]=paso[6]+(movimiento[i+1].pos_motor[6]-
                            movimiento[i].pos_motor[6])/inter;
           fl_set_slider_value(fd_panel->motor_b2,movimiento[i].pos_motor[6]+paso[6]);
           fl_call_object_callback(fd_panel->motor_b2);

           paso[7]=paso[7]+(movimiento[i+1].pos_motor[7]-
                            movimiento[i].pos_motor[7])/inter;
           fl_set_slider_value(fd_panel->motor_b3,movimiento[i].pos_motor[7]+paso[7]);
           fl_call_object_callback(fd_panel->motor_b3);

           paso[8]=paso[8]+(movimiento[i+1].pos_motor[8]-
                            movimiento[i].pos_motor[8])/inter;
           fl_set_slider_value(fd_panel->motor_b4,movimiento[i].pos_motor[8]+paso[8]);
           fl_call_object_callback(fd_panel->motor_b4);

           paso[9]=paso[9]+(movimiento[i+1].pos_motor[9]-
                            movimiento[i].pos_motor[9])/inter;
           fl_set_slider_value(fd_panel->motor_c1,movimiento[i].pos_motor[9]+paso[9]);
           fl_call_object_callback(fd_panel->motor_c1);

           paso[10]=paso[10]+(movimiento[i+1].pos_motor[10]-
                            movimiento[i].pos_motor[10])/inter;
           fl_set_slider_value(fd_panel->motor_c2,movimiento[i].pos_motor[10]+paso[10]);
           fl_call_object_callback(fd_panel->motor_c2);

           paso[11]=paso[11]+(movimiento[i+1].pos_motor[11]-
                            movimiento[i].pos_motor[11])/inter;
           fl_set_slider_value(fd_panel->motor_c3,movimiento[i].pos_motor[11]+paso[11]);
           fl_call_object_callback(fd_panel->motor_c3);

           paso[12]=paso[12]+(movimiento[i+1].pos_motor[12]-
                            movimiento[i].pos_motor[12])/inter;
           fl_set_slider_value(fd_panel->motor_c4,movimiento[i].pos_motor[12]+paso[12]);
           fl_call_object_callback(fd_panel->motor_c4);

           cont=movimiento[i].retardo*1000000;
           usleep(cont);
           if (fl_check_forms()==fd_panel->b_parar) { 
             i=sup;  fin=1; k=inter;     // parar la reproduccin
             fl_set_button(fd_panel->b_repro,0);
             fl_ringbell(5); } 
        }
        paso[1]=0 ;paso[2]=0; paso[3]=0; paso[4]=0;
        paso[5]=0 ;paso[6]=0; paso[7]=0; paso[8]=0;
        paso[9]=0 ;paso[10]=0;paso[11]=0;paso[12]=0;
        if ((k==1) && (fl_check_forms()==fd_panel->b_parar)) { 
          i=sup;  fin=1;      // parar la reproduccin
          fl_set_button(fd_panel->b_repro,0);
          fl_ringbell(5); } 
      }
      fl_set_counter_value(fd_panel->boton_pasos,sup);
      fl_call_object_callback(fd_panel->boton_pasos);
      cont=movimiento[i].retardo*1000000;
      usleep(cont);
    } while ( fin==0 );
    actual=temp;         // recupera el paso actual
    fl_set_counter_value(fd_panel->boton_pasos,actual);
    fl_call_object_callback(fd_panel->boton_pasos);
  }
  if (inf>sup) {
    temp=actual;
    do {
      for (i=inf; i>sup; i--) {
        fl_set_counter_value(fd_panel->boton_pasos,i);
        fl_call_object_callback(fd_panel->boton_pasos);
        cont=movimiento[i].retardo*1000000;
        usleep(cont);

        for (k=1; k<inter;k++) {
           paso[1]=paso[1]+(movimiento[i-1].pos_motor[1]-
                            movimiento[i].pos_motor[1])/inter;
           fl_set_slider_value(fd_panel->motor_a1,movimiento[i].pos_motor[1]+paso[1]);
           fl_call_object_callback(fd_panel->motor_a1);

           paso[2]=paso[2]+(movimiento[i-1].pos_motor[2]-
                            movimiento[i].pos_motor[2])/inter;
           fl_set_slider_value(fd_panel->motor_a2,movimiento[i].pos_motor[2]+paso[2]);
           fl_call_object_callback(fd_panel->motor_a2);

           paso[3]=paso[3]+(movimiento[i-1].pos_motor[3]-
                            movimiento[i].pos_motor[3])/inter;
           fl_set_slider_value(fd_panel->motor_a3,movimiento[i].pos_motor[3]+paso[3]);
           fl_call_object_callback(fd_panel->motor_a3);

           paso[4]=paso[4]+(movimiento[i-1].pos_motor[4]-
                      movimiento[i].pos_motor[4])/inter;
           fl_set_slider_value(fd_panel->motor_a4,movimiento[i].pos_motor[4]+paso[4]);
           fl_call_object_callback(fd_panel->motor_a4);

           paso[5]=paso[5]+(movimiento[i-1].pos_motor[5]-
                            movimiento[i].pos_motor[5])/inter;
           fl_set_slider_value(fd_panel->motor_b1,movimiento[i].pos_motor[5]+paso[5]);
           fl_call_object_callback(fd_panel->motor_b1);

           paso[6]=paso[6]+(movimiento[i-1].pos_motor[6]-
                            movimiento[i].pos_motor[6])/inter;
           fl_set_slider_value(fd_panel->motor_b2,movimiento[i].pos_motor[6]+paso[6]);
           fl_call_object_callback(fd_panel->motor_b2);

           paso[7]=paso[7]+(movimiento[i-1].pos_motor[7]-
                            movimiento[i].pos_motor[7])/inter;
           fl_set_slider_value(fd_panel->motor_b3,movimiento[i].pos_motor[7]+paso[7]);
           fl_call_object_callback(fd_panel->motor_b3);

           paso[8]=paso[8]+(movimiento[i-1].pos_motor[8]-
                            movimiento[i].pos_motor[8])/inter;
           fl_set_slider_value(fd_panel->motor_b4,movimiento[i].pos_motor[8]+paso[8]);
           fl_call_object_callback(fd_panel->motor_b4);

           paso[9]=paso[9]+(movimiento[i-1].pos_motor[9]-
                            movimiento[i].pos_motor[9])/inter;
           fl_set_slider_value(fd_panel->motor_c1,movimiento[i].pos_motor[9]+paso[9]);
           fl_call_object_callback(fd_panel->motor_c1);

           paso[10]=paso[10]+(movimiento[i-1].pos_motor[10]-
                            movimiento[i].pos_motor[10])/inter;
           fl_set_slider_value(fd_panel->motor_c2,movimiento[i].pos_motor[10]+paso[10]);
           fl_call_object_callback(fd_panel->motor_c2);

           paso[11]=paso[11]+(movimiento[i-1].pos_motor[11]-
                            movimiento[i].pos_motor[11])/inter;
           fl_set_slider_value(fd_panel->motor_c3,movimiento[i].pos_motor[11]+paso[11]);
           fl_call_object_callback(fd_panel->motor_c3);

           paso[12]=paso[12]+(movimiento[i-1].pos_motor[12]-
                            movimiento[i].pos_motor[12])/inter;
           fl_set_slider_value(fd_panel->motor_c4,movimiento[i].pos_motor[12]+paso[12]);
           fl_call_object_callback(fd_panel->motor_c4);

           cont=movimiento[i].retardo*1000000;
           usleep(cont);
           if (fl_check_forms()==fd_panel->b_parar) { 
             i=sup;  fin=1; k=inter;     // parar la reproduccin
             fl_set_button(fd_panel->b_repro,0);
             fl_ringbell(5); } 
        }
        paso[1]=0 ;paso[2]=0; paso[3]=0; paso[4]=0;
        paso[5]=0 ;paso[6]=0; paso[7]=0; paso[8]=0;
        paso[9]=0;paso[10]=0;paso[11]=0;paso[12]=0;
        if ((k==1) && (fl_check_forms()==fd_panel->b_parar)) { 
          i=inf;  fin=1;      // parar la reproduccin
          fl_set_button(fd_panel->b_repro,0);
          fl_ringbell(5); } 
      }
      fl_set_counter_value(fd_panel->boton_pasos,sup);
      fl_call_object_callback(fd_panel->boton_pasos);
      cont=movimiento[i].retardo*1000000;
      usleep(cont);
    } while ( fin==0 );
    actual=temp;         // recupera el paso actual
    fl_set_counter_value(fd_panel->boton_pasos,actual);
    fl_call_object_callback(fd_panel->boton_pasos);
  }
}



void posicionar_futaba(FL_OBJECT *obj, long arg)
/*....................................................................
  . Rutina que lee el valor de la barra deslizadora y lo asigna al   .
  . motor correspondiente.                                           .
  . Se utiliza el argumento 'ARG' para distinguir la barra utilizada .
  ....................................................................*/
{ 
//  char tarjeta='a';
  
  paso.pos_motor[arg]=fl_get_slider_value(obj);
  if (arg<=4) { enviar_trama('a','p',(char) arg,(char) paso.pos_motor[arg]); }
  else if (arg<=8) { enviar_trama('b','p',(char) (arg-4),(char) paso.pos_motor[arg]);}
       else { enviar_trama('c','p',(char) (arg-8),(char) paso.pos_motor[arg]);}
}


void poner_ojo(FL_OBJECT *obj, long arg)
/*............................................................
  . Rutina que enciende el LED correspondiente de cada ojo   .
  ............................................................
  . La asignacin entre el botn y el bit de control, es la  .
  . que corresponde a conectar un perifrico preparado para  .  
  . el puerto C en el conector de expansin X de la BT6811   .
  ............................................................*/
{
  int byte1=0;
  
  /* Mscaras de la BT6811 A */
  
  byte1=(1-fl_get_button(fd_panel->ojo_a1))*0x40+    /* PX7 -> PC1 */
        (1-fl_get_button(fd_panel->ojo_a2))*0x80+    /* PX8 -> PC2 */
        (1-fl_get_button(fd_panel->ojo_a3))*0x10+    /* PX5 -> PC3 */
        (1-fl_get_button(fd_panel->ojo_a4))*0x20+    /* PX6 -> PC4 */
        (1-fl_get_button(fd_panel->ojo_a5))*0x04+    /* PX3 -> PC5 */
        (1-fl_get_button(fd_panel->ojo_a6))*0x08+    /* PX4 -> PC6 */
        (1-fl_get_button(fd_panel->ojo_a7))*0x01+    /* PX1 -> PC7 */
        (1-fl_get_button(fd_panel->ojo_a8))*0x02;    /* PX2 -> PC8 */
  paso.ojo[1]=byte1;
  enviar_trama('a','l',(char) byte1,'.');
        
  /* Mscara de la BT6811 B */

  byte1=(1-fl_get_button(fd_panel->ojo_b1))*0x40+    /* PX7 -> PC1 */
        (1-fl_get_button(fd_panel->ojo_b2))*0x80+    /* PX8 -> PC2 */
        (1-fl_get_button(fd_panel->ojo_b3))*0x10+    /* PX5 -> PC3 */
        (1-fl_get_button(fd_panel->ojo_b4))*0x20+    /* PX6 -> PC4 */
        (1-fl_get_button(fd_panel->ojo_b5))*0x04+    /* PX3 -> PC5 */
        (1-fl_get_button(fd_panel->ojo_b6))*0x08+    /* PX4 -> PC6 */
        (1-fl_get_button(fd_panel->ojo_b7))*0x01+    /* PX1 -> PC7 */
        (1-fl_get_button(fd_panel->ojo_b8))*0x02;    /* PX2 -> PC8 */
  paso.ojo[2]=byte1;
  enviar_trama('b','l',(char) byte1,'.');
}


void apaga_motor(FL_OBJECT *obj,long arg)
/*..................................................................
  . Esta rutina conecta o desconecta el motor marcado              .
  . La rutina lee el estado de todos los motores, y manda el estado.
  . a los servidores de los motores                                .
  ..................................................................*/
{
  int byte1=0;
  
  /* Mscaras de la BT6811 A */
  
  byte1=fl_get_button(fd_panel->MA1)*0x01+
        fl_get_button(fd_panel->MA2)*0x02+
        fl_get_button(fd_panel->MA3)*0x04+
        fl_get_button(fd_panel->MA4)*0x08;
  enviar_trama('a','e',(char) byte1,'.');
        
  /* Mscara de la BT6811 B */
  
  byte1=fl_get_button(fd_panel->MB1)*0x01+
        fl_get_button(fd_panel->MB2)*0x02+
        fl_get_button(fd_panel->MB3)*0x04+
        fl_get_button(fd_panel->MB4)*0x08;
  enviar_trama('b','e',(char) byte1,'.');

  /* Mscara de la BT6811 C */
  
  byte1=fl_get_button(fd_panel->MC1)*0x01+
        fl_get_button(fd_panel->MC2)*0x02+
        fl_get_button(fd_panel->MC3)*0x04+
        fl_get_button(fd_panel->MC4)*0x08;
  enviar_trama('c','e',(char) byte1,'.');
    
}


void menu_pulsado(FL_OBJECT *obj, long arg)
/*.................................................................
  . Esta funcion analiza la opcin del menu desplegable que se ha .
  . pulsado. Cuando se localiza una opcin se procede a su reali- .
  . zacin.                                                       .
  .................................................................*/
{
 int  menu_item;
  
 menu_item=fl_get_menu(obj);
 switch (menu_item) {
   case 1: abrir_fichero();break;
   case 2: guardar_fichero();break;
   case 3: convertir_fichero();break;
   case 4: salir_programa();break;  // salimos del programa 
   default: break;
  }
} 

   
/*****************************************************************
 * PROCEDIMIENTO PRINCIPAL: Arranca el panel y la interaccin    *
 *****************************************************************/

main(int argc, char *argv[])
{

  if (abrir_puerto_serie(COM1)==0) {
    printf("Error al abrir el puerto serie: %s\n",getserial_error());
    exit(1);
  }
  baudios(9600);
  resetct6811();

  fl_initialize(&argc,argv,"XpuchoBot 1.0.",0,0);             
  fd_panel=create_form_panel();
  fl_show_form(fd_panel->panel,FL_PLACE_CENTER,FL_FULLBORDER, "XPuchoBot 1.0.");
  inicializar();   // inicializamos el panel
  do { fl_do_forms(); } while (1);   // activamos el panel y evitamos que
                                     // al pulsar 'PARAR' abandonemos el 
                                     // programa, debido a que no tiene 
                                     // asociada una rutina CALLBACK
  
}
