/*
**************************************************************************
* CONTROL                                    Andrs Prieto-Moreno Torres *
**************************************************************************
*   Programa  que  ofrece  un  interfaz  grfico  para  mover  el  robot *
* en distintas direcciones.                                              *
**************************************************************************
* 1 de Agosto de 2000                                                    *
**************************************************************************
*/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "forms.h"
#include "control.h"
#include "serie.h"
#include "cuadro.h"

FD_cuadro  *fd_cuadro;   // la hacemos global para poder acceder a ella

#define MODE644 S_IWUSR|S_IRUSR|S_IRGRP|S_IRGRP
#define PERIOD  0.1    // periodo de acceso a la CT6811

struct t_paso {        // definicin del tipo 'paso'
  int   pos_motor[13];
  int   ojo[3];
  float retardo;
};  

// el movimiento es una secuencia de pasos, el orden es cronolgico, por
// eso se puede estructurar como un ARRAY

struct t_paso avanzar[7]={{{0,193,151, 94,151, 23,103,126, 76,211,108,103, 37},{ 0, 0}, 0.3},
                          {{0,193,126, 53,151, 23,103,126, 76,211, 99,147, 37},{ 0, 0}, 0.3},
                          {{0,193,126, 53,179, 23, 87,106,103,211, 99,147, 37},{ 0, 0}, 0.3},
			  {{0,193,131, 65,209, 23,131, 78,103,211, 83,131, 37},{ 0, 0}, 0.3},
			  {{0,193,154,103,209, 23,158,170,103,211,119, 88, 37},{ 0, 0}, 0.3},
			  {{0,193,190,140, 92, 23,163,197, 28,211,138, 60, 37},{ 0, 0}, 0.3},
			  {{0,193,151, 94,151, 23,103,126, 76,211,108,103, 37},{ 0, 0}, 0.3}};

struct t_paso parar[1] = {{{0,193,151, 94, 151, 23,103,126, 76,211,108,103, 37},{ 0, 0}, 0.5}};

struct t_paso izquierda[6 ]={{{0,193,151, 94,151, 23,103,126, 76,211,108,103, 37},{ 0, 0}, 0.3},
                             {{0,193,117, 46,151, 23,103,126, 76,211, 74,129, 37},{ 0, 0}, 0.3},
                             {{0,204,117, 46,151, 23,103, 19, 76,211, 74,129, 37},{ 0, 0}, 0.3},
			     {{0,204,117, 46,151, 23,156,151, 76,211, 74,129, 37},{ 0, 0}, 0.3},
			     {{0,181,117, 46,110, 23,156,151, 25,211, 74,158, 37},{ 0, 0}, 0.3},
			     {{0,193,151, 94,151, 23,103,126, 76,211,108,103, 37},{ 0, 0}, 0.3}};
			

struct t_paso derecha[6]={{{0,193,151, 94,151, 23,103,126, 76,211,108,103, 37},{ 0, 0}, 0.3},
                          {{0,193,151, 94,151, 23,103, 76, 76,211,108,103, 37},{ 0, 0}, 0.3},
                          {{0,193,151, 94,200, 23,103, 76, 76,211,108,103, 37},{ 0, 0}, 0.3},
			  {{0,200,151, 94,200, 23,103, 76, 76,224,108,103, 37},{ 0, 0}, 0.3},
			  {{0,200,108,  0,170, 23,103,129, 76,224,108,103, 37},{ 0, 0}, 0.3},
			  {{0,193,151, 94,151, 23,103,126, 76,211,108,103, 37},{ 0, 0}, 0.3}};
			

struct t_paso retroceder[9]={{{0,193,151, 94,151, 23,103,126, 76,211,108,103, 37},{0 ,0} ,0.3},
                             {{0,193,151, 94,188, 23,103,126,113,211,108,103, 37},{0 ,0} ,0.3},
                             {{0,193,151, 94,188, 23, 67, 90,113,211,108,103, 37},{0 ,0} ,0.3},
                             {{0,193,124, 71,188, 23, 67, 90,113,211, 78,133, 37},{0 ,0} ,0.3},
                             {{0,193,124, 71,145, 23,110,138, 64,211, 78,133, 37},{0 ,0} ,0.3},
                             {{0,193,124, 71,145, 23,110,138, 64,211,159, 67, 37},{0 ,0} ,0.3},
                             {{0,193,154, 71,145, 23,110,138, 64,211,159, 67, 37},{0 ,0} ,0.3},
                             {{0,193,197,151,113, 23,126,158, 42,211,159, 67, 37},{0 ,0} ,0.3},
                             {{0,193,151, 94,151, 23,103,126, 76,211,108,103, 37},{0 ,0} ,0.3}};



void actualizar_paso(struct t_paso aux);
void enviar_trama(char placa,char comando,char dato1, char dato2);


/**************************************************************
* SUBRUTINAS DEL PROGRAMA                                     *
***************************************************************/

void inicializar()
/*...........................................................
  . Inicializa la posicin del robot                        .
  ...........................................................
*/
{
  int i,j;
  struct t_paso paso;
  
  // ponemos los motores en un estado inicial

  paso.pos_motor[1]=193;
  paso.pos_motor[2]=151;
  paso.pos_motor[3]=94;
  paso.pos_motor[4]=151;
  paso.pos_motor[5]=23;
  paso.pos_motor[6]=103;
  paso.pos_motor[7]=126;
  paso.pos_motor[8]=76;
  paso.pos_motor[9]=211;
  paso.pos_motor[10]=108;
  paso.pos_motor[11]=103;
  paso.pos_motor[12]=37;

  // inicializa el valor del retardo entre pasos
  
  paso.retardo=0.3;
  
  // inicializa el valor de los ojos
  
  paso.ojo[1]=0x0;
  paso.ojo[2]=0x0;

  // actualizo los cambios

  actualizar_paso(paso);
}
 

void actualizar_paso(struct t_paso aux) 
{
  int i=1;

  // actualizamos los ojos
  enviar_trama('a','l',aux.ojo[0],'.');
  enviar_trama('b','l',aux.ojo[1],'.');

  // actualizamos los motores
  for (i=1; i<=12; i++) {
    if (i<5) { enviar_trama('a','p',i,aux.pos_motor[i]);}
    else if ((i>=5) && (i<9)) { enviar_trama('b','p',i-4,aux.pos_motor[i]);}
    else { enviar_trama('c','p',i-8,aux.pos_motor[i]);}
  }
}


void enviar_trama(char placa,char comando,char dato1, char dato2)
/*..................................................................
  . Envia una trama a la RED de BT6811. Esta trama se enva por el .
  . puerto serie y el programa de la CT6811 lo reenva a travs del.
  . bus SPI, al que estn conectadas el resto de los mdulos de    .
  . control (BT6811s)                                              .
  ..................................................................*/
{
  enviar_car(placa);
  enviar_car(comando);
  enviar_car(dato1);
  enviar_car(dato2);
}  


/******************************************************************
 * FUNCIONES CALLBACKS UTILIZADAS POR LOS OBJECTOS DE LAS XFORMS  *
 ******************************************************************/


void control(FL_OBJECT *obj, long arg)
/*.....................................................................
  . Los botones del pad de control llaman a esta rutina, de forma que .
  . vez que se pulsa uno hay que ver cual ha sido para poder realizar .
  . la correspondiente accin.                                        .
  .....................................................................*/  
{
  int npaso;            // variable auxiliar que indica nmero de pasos
  struct t_paso *paso;  // puntero auxiliar a la estructura paso
  int i;

  /* boton avanzar */
   
  if (obj==fd_cuadro->avanzar) {
    npaso=6;      // La secuencia avanzar tiene 6 pasos
    paso=avanzar; // Actualizamos variable auxiliar con la secuencia avanzar
  }
   
  /* boton retroceder */
   
  if (obj==fd_cuadro->retroceder) {
    npaso=8;         // La secuencia retroceder tiene 8 pasos
    paso=retroceder;  // Actualizamos variable auxiliar con la secuencia retroceder
  }
   /* boton parar */
   
  if (obj==fd_cuadro->parar) {
    npaso=0;     // La secuencia parar tiene 1 pasos
    paso=parar;  // Actualizamos variable auxiliar con la secuencia parar
  }

  /* boton derecha */
   
  if (obj==fd_cuadro->derecha) {
    npaso=5;     // La secuencia derecha tiene 5 pasos
    paso=derecha; // Actualizamos variable auxiliar con la secuencia derecha
  }

  /* boton izquierda */
   
  if (obj==fd_cuadro->izquierda) {
    npaso=5;       // La secuencia izquierda tiene 5 pasos
    paso=izquierda; // Actualizamos variable auxiliar con la secuencia izquierda
  }

  for (i=0; i<=npaso; i++) {
    actualizar_paso(paso[i]);
    usleep(paso[i].retardo*1000000);
  }
}


void activar(FL_OBJECT *obj, long arg)
/*....................................................................
  . Cuando el botn est  pulsado el robot est activado, en caso    .
  . contrario se encuentra desactivado y el robot no funciona.       .
  ....................................................................*/
{
  if ( fl_get_button(obj) ) {
    enviar_trama('a','e',0x0F,'.');
    enviar_trama('b','e',0x0F,'.');
    enviar_trama('c','e',0x0F,'.');
  }
  else {
    enviar_trama('a','e',0x0,'.');
    enviar_trama('b','e',0x0,'.');
    enviar_trama('c','e',0x0,'.');
  }
}


void salir(FL_OBJECT *obj, long arg)
/*....................................................................
  .  Rutina que sale del programa                                    .
  ....................................................................*/
{
  exit(0);
}


void poner_posicion(FL_OBJECT *obj, long arg)
/*.....................................................................
  .  Rutina que coloca el robot en una posicin determinada           .
  .  Cada botn de la barra inferior selecciona una posicin distinta .
  .....................................................................*/
{
   int i;
   int temp[13];
   struct t_paso paso;
   
   /* posicion levantado */
   
   if (obj==fd_cuadro->levantado) {
     temp[1]=193; temp[2]=151 ; temp[3]=94 ; temp[4]=151 ;
     temp[5]=23 ; temp[6]=103 ; temp[7]=126; temp[8]=76 ;
     temp[9]=211; temp[10]=108; temp[11]=103; temp[12]=37;
   }    
   
   /* posicion sentado */
   
   if (obj==fd_cuadro->sentado) {
     temp[1]=193; temp[2]=151 ; temp[3]=94 ; temp[4]=16  ;
     temp[5]=23 ; temp[6]=103  ; temp[7]=126; temp[8]=64 ;
     temp[9]=211; temp[10]=106; temp[11]=238 ; temp[12]=37;
   }    

   /* posicion equilibrio A */
   
   if (obj==fd_cuadro->equilibrio_a) {
     temp[1]=185; temp[2]=151  ; temp[3]=94; temp[4]=151 ;
     temp[5]=15 ; temp[6]=103 ; temp[7]=126; temp[8]=76 ;
     temp[9]=218; temp[10]=108; temp[11]=103; temp[12]=44;
   }    

   /* posicion equilibrio B */
   
   if (obj==fd_cuadro->equilibrio_b) {
     temp[1]=193; temp[2]=151  ; temp[3]=94; temp[4]=151 ;
     temp[5]=29 ; temp[6]=103 ; temp[7]=126  ; temp[8]=76;
     temp[9]=203; temp[10]=108; temp[11]=103; temp[12]=30;
   }    

   for (i=1; i<=12; i++) {
     paso.pos_motor[i]=temp[i];
   }
   actualizar_paso(paso);
}


/*****************************************************************
 * PROCEDIMIENTO PRINCIPAL: Arranca el panel y la interaccin    *
 *****************************************************************/

main(int argc, char *argv[])
{
  if (abrir_puerto_serie(COM1)==0) {
    printf("Error al abrir el puerto serie: %s\n",getserial_error());
    exit(1);
  }
  baudios(9600);
  resetct6811();

  fl_initialize(&argc,argv,"Control 1.0.",0,0);
  fd_cuadro=create_form_cuadro();
  fl_show_form(fd_cuadro->cuadro,FL_PLACE_CENTER,FL_FULLBORDER, "Control 1.0.");
  inicializar();                     // inicializamos el panel
  do { fl_do_forms(); } while (1);   // activamos el panel y evitamos que
                                     // al pulsar 'PARAR' abandonemos el 
                                     // programa, debido a que no tiene 
                                     // asociada una rutina CALLBACK
}



























