// Yep, this is actually -*- c++ -*-
/*********************************************************************************************************
 *  RepRap 3rd Generation Firmware (R3G)
 *
 *  Slave Extruder Firmware for Extruder Controller v2.x
 *
 *  Board documentation at: http://make.rrrf.org/ec-2.0
 *  Specification for this protocol is located at: http://docs.google.com/Doc?id=dd5prwmp_14ggw37mfp
 *  
 *  License: GPLv2
 *  Authors: Marius Kintel, Adam Mayer, and Zach Hoeken
 *
 *  Version History:
 *
 *  0001: Initial release of the protocol and firmware.
 *
 *********************************************************************************************************/

#ifndef __AVR_ATmega168__
#error Oops!  Make sure you have 'Arduino' selected from the boards menu.
#endif

//include some basic libraries.
#include <WProgram.h>
#include <Servo.h>
#include <SimplePacket.h>
#include <stdint.h>

#include "Configuration.h"
#include "Datatypes.h"
#include "RS485.h"
#include "Variables.h"
#include "ThermistorTable.h"

//this is our firmware version
#define FIRMWARE_VERSION 0001

//set up our firmware for actual usage.
#include "WProgram.h"
void setup();
void initialize();
void init_serial();
void loop();
void abort_print();
void init_extruder();
void read_quadrature();
void enable_motor_1();
void disable_motor_1();
void reverse_motor_1();
void cancellable_delay(unsigned int duration);
void enable_motor_2();
void disable_motor_2();
void enable_fan();
void disable_fan();
void open_valve();
void close_valve();
byte is_tool_ready();
void set_temperature(int temp);
int get_temperature();
int read_thermistor();
int read_thermocouple();
int sample_temperature(byte pin);
void manage_temperature();
void manage_motor1_speed();
void init_commands();
void process_packets();
void send_reply();
void handle_query();
void enableTimer1Interrupt();
void disableTimer1Interrupt();
void setTimer1Resolution(byte r);
void setTimer1Ceiling(unsigned int c);
unsigned int getTimer1Ceiling(unsigned long ticks);
byte getTimer1Resolution(unsigned long ticks);
void setTimer1Ticks(unsigned long ticks);
void setupTimer1Interrupt();
void delayMicrosecondsInterruptible(unsigned int us);
void setup()
{
  //setup our firmware to a default state.
  init_serial(); //dont want to re-initialize serial!
  initialize();

  //this is a simple text string that identifies us.
  //Serial.print("R3G Slave v");
  //Serial.println(FIRMWARE_VERSION, DEC);
}

//this function takes us back to our default state.
void initialize()
{
  is_tool_paused = false;
  init_extruder();
}

//start our hardware serial drivers
void init_serial()
{
  pinMode(RX_ENABLE_PIN, OUTPUT);
  pinMode(TX_ENABLE_PIN, OUTPUT);
  digitalWrite(RX_ENABLE_PIN, LOW); //always listen

  Serial.begin(SERIAL_SPEED);
}

//handle various things we're required to do.
void loop()
{
  //check for and handle any packets that come in.
  process_packets();
  	
  //did we trigger a reversal?
  if (motor1_reversal_state)
    reverse_motor_1();

  //manage our extruder stuff.
  if (!is_tool_paused)
    manage_temperature();
}

//handle the abortion of a print job
void abort_print()
{
  //TODO: shut down all things

  //initalize everything to the beginning
  initialize();
}

// Yep, this is actually -*- c++ -*-
void init_extruder()
{
  //reset motor1
  motor1_control = MC_PWM;
  motor1_dir = MC_FORWARD;
  motor1_pwm = 0;
  motor1_target_rpm = 0;
  motor1_current_rpm = 0;
  
  //reset motor2
  motor2_control = MC_PWM;
  motor2_dir = MC_FORWARD;
  motor2_pwm = 0;
  motor2_target_rpm = 0;
  motor2_current_rpm = 0;
	
  //free up 9/10
  servo1.detach();
  servo2.detach();

  //init our PID stuff.
  speed_error = 0;
  iState = 0;
  dState = 0;
  pGain = SPEED_INITIAL_PGAIN;
  iGain = SPEED_INITIAL_IGAIN;
  dGain = SPEED_INITIAL_DGAIN;
	
  //encoder pins are for reading.
  pinMode(ENCODER_A_PIN, INPUT);
  pinMode(ENCODER_B_PIN, INPUT);

  //pullups on our encoder pins
  digitalWrite(ENCODER_A_PIN, HIGH);
  digitalWrite(ENCODER_B_PIN, HIGH);

  //attach our interrupt handler
  attachInterrupt(0, read_quadrature, CHANGE);

  //setup our motor control pins.
  pinMode(MOTOR_1_SPEED_PIN, OUTPUT);
  pinMode(MOTOR_2_SPEED_PIN, OUTPUT);
  pinMode(MOTOR_1_DIR_PIN, OUTPUT);
  pinMode(MOTOR_2_DIR_PIN, OUTPUT);

  //turn them off and forward.
  digitalWrite(MOTOR_1_SPEED_PIN, LOW);
  digitalWrite(MOTOR_2_SPEED_PIN, LOW);
  digitalWrite(MOTOR_1_DIR_PIN, HIGH);
  digitalWrite(MOTOR_2_DIR_PIN, HIGH);

  //setup our various accessory pins.
  pinMode(HEATER_PIN, OUTPUT);
  pinMode(FAN_PIN, OUTPUT);
  pinMode(VALVE_PIN, OUTPUT);

  //turn them all off
  digitalWrite(HEATER_PIN, LOW);
  digitalWrite(FAN_PIN, LOW);
  digitalWrite(VALVE_PIN, LOW);

  //setup our debug pin.
  pinMode(DEBUG_PIN, OUTPUT);
  digitalWrite(DEBUG_PIN, LOW);

  //default to zero.
  set_temperature(0);

  setupTimer1Interrupt();
}

void read_quadrature()
{  
  // found a low-to-high on channel A
  if (digitalRead(ENCODER_A_PIN) == HIGH)
  {   
    // check channel B to see which way
    if (digitalRead(ENCODER_B_PIN) == LOW)
      QUADRATURE_INCREMENT
else
  QUADRATURE_DECREMENT
}
// found a high-to-low on channel A
  else                                        
  {
    // check channel B to see which way
    if (digitalRead(ENCODER_B_PIN) == LOW)
      QUADRATURE_DECREMENT
else
  QUADRATURE_INCREMENT
}
}

void enable_motor_1()
{
  if (motor1_control == MC_PWM)
  {
    //nuke any previous reversals.
    motor1_reversal_state = false;
    
    if (motor1_dir == MC_FORWARD)
      digitalWrite(MOTOR_1_DIR_PIN, HIGH);
    else
      digitalWrite(MOTOR_1_DIR_PIN, LOW);

    analogWrite(MOTOR_1_SPEED_PIN, motor1_pwm);
  }
  else if (motor1_control == MC_ENCODER)
  {
    speed_error = 0;
    disableTimer1Interrupt();
    setTimer1Ticks(motor1_target_rpm);
    enableTimer1Interrupt();
  }
  else if (motor1_control == MC_STEPPER)
  {
    setTimer1Ticks(stepper_ticks);
    enableTimer1Interrupt();
  }
}

void disable_motor_1()
{
  if (motor1_control == MC_PWM)
  {
    analogWrite(MOTOR_1_SPEED_PIN, 0);
    
    if (motor1_dir == MC_FORWARD)
      motor1_reversal_state = true;
  }
  else if (motor1_control == MC_ENCODER)
  {
    speed_error = 0;
    disableTimer1Interrupt();
  }
  else if (motor1_control == MC_STEPPER)
  {
    disableTimer1Interrupt();
    digitalWrite(MOTOR_1_SPEED_PIN, LOW);
    digitalWrite(MOTOR_2_SPEED_PIN, LOW);
  }
}

void reverse_motor_1()
{
  //wait for it to stop.
  if (DELAY_FOR_STOP > 0)
    cancellable_delay(DELAY_FOR_STOP);

  //reverse our motor for a bit.
  if (MOTOR_REVERSE_DURATION > 0 && motor1_reversal_state)
  {
    digitalWrite(MOTOR_1_DIR_PIN, LOW);
    analogWrite(MOTOR_1_SPEED_PIN, motor1_pwm);
    cancellable_delay(MOTOR_REVERSE_DURATION);
  }

  //wait for it to stop.
  if (DELAY_FOR_STOP > 0 && motor1_reversal_state)
    cancellable_delay(DELAY_FOR_STOP);
  
  //reverse our motor for a bit.
  if (MOTOR_FORWARD_DURATION > 0 && motor1_reversal_state)
  {
    digitalWrite(MOTOR_1_DIR_PIN, HIGH);
    analogWrite(MOTOR_1_SPEED_PIN, motor1_pwm);
    cancellable_delay(MOTOR_FORWARD_DURATION);
  }

  //finally stop it.
  if (motor1_reversal_state)
    analogWrite(MOTOR_1_SPEED_PIN, 0);
  
  //we're done.
  motor1_reversal_state = false;
}

//basically we want to delay unless there is a start command issued.
void cancellable_delay(unsigned int duration)
{
  if (motor1_reversal_state)
  {
    for (unsigned int i=0; i<duration; i++)
    {
      delay(1);
      process_packets();
      
      //did we start up?  break!
      if (!motor1_reversal_state)
        break;
    }    
  }
}

void enable_motor_2()
{
  if (motor2_control == MC_PWM)
  {
    if (motor2_dir == MC_FORWARD)
      digitalWrite(MOTOR_2_DIR_PIN, HIGH);
    else
      digitalWrite(MOTOR_2_DIR_PIN, LOW);

    analogWrite(MOTOR_2_SPEED_PIN, motor2_pwm);
  }
  else if (motor2_control == MC_ENCODER)
  {
    speed_error = 0;
    setTimer1Ticks(motor2_target_rpm/16);
    enableTimer1Interrupt();
  }
}

void disable_motor_2()
{
  if (motor2_control == MC_PWM)
    analogWrite(MOTOR_2_SPEED_PIN, 0);
  else if (motor2_control == MC_ENCODER)
  {
    speed_error = 0;
    disableTimer1Interrupt();
  }
}

void enable_fan()
{
  digitalWrite(FAN_PIN, HIGH);
}

void disable_fan()
{
  digitalWrite(FAN_PIN, LOW);
}

void open_valve()
{
  digitalWrite(VALVE_PIN, HIGH);
}

void close_valve()
{
  digitalWrite(VALVE_PIN, LOW);
}

byte is_tool_ready()
{
  //are we within 5% of the temperature?
  if (current_temperature > (int)(target_temperature * 0.95))
    return 1;
  else
    return 0;
}

void set_temperature(int temp)
{
  target_temperature = temp;
  max_temperature = (int)((float)temp * 1.1);
}

/**
 *  Samples the temperature and converts it to degrees celsius.
 *  Returns degrees celsius.
 */
int get_temperature()
{
#ifdef THERMISTOR_PIN
  return read_thermistor();
#endif
#ifdef THERMOCOUPLE_PIN
  return read_thermocouple();
#endif
}

/*
* This function gives us the temperature from the thermistor in Celsius
 */
#ifdef THERMISTOR_PIN
int read_thermistor()
{
  int raw = sample_temperature(THERMISTOR_PIN);

  int celsius = 0;
  byte i;

  for (i=1; i<NUMTEMPS; i++)
  {
    if (temptable[i][0] > raw)
    {
      celsius  = temptable[i-1][1] + 
        (raw - temptable[i-1][0]) * 
        (temptable[i][1] - temptable[i-1][1]) /
        (temptable[i][0] - temptable[i-1][0]);

      if (celsius > 255)
        celsius = 255; 

      break;
    }
  }

  // Overflow: We just clamp to 0 degrees celsius
  if (i == NUMTEMPS)
    celsius = 0;

  return celsius;
}
#endif

/*
* This function gives us the temperature from the thermocouple in Celsius
 */
#ifdef THERMOCOUPLE_PIN
int read_thermocouple()
{
  return ( 5.0 * sample_temperature(THERMOCOUPLE_PIN) * 100.0) / 1024.0;
}
#endif

/*
* This function gives us an averaged sample of the analog temperature pin.
 */
int sample_temperature(byte pin)
{
  int raw = 0;

  //read in a certain number of samples
  for (byte i=0; i<TEMPERATURE_SAMPLES; i++)
    raw += analogRead(pin);

  //average the samples
  raw = raw/TEMPERATURE_SAMPLES;

  //send it back.
  return raw;
}


/*!
 Manages motor and heater based on measured temperature:
 o If temp is too low, don't start the motor
 o Adjust the heater power to keep the temperature at the target
 */
void manage_temperature()
{
  //make sure we know what our temp is.
  current_temperature = get_temperature();

  //put the heater into high mode if we're not at our target.
  if (current_temperature < target_temperature)
    analogWrite(HEATER_PIN, heater_high);
  //put the heater on low if we're at our target.
  else if (current_temperature < max_temperature)
    analogWrite(HEATER_PIN, heater_low);
  //turn the heater off if we're above our max.
  else
    analogWrite(HEATER_PIN, 0);
}


//this handles the timer interrupt event
void manage_motor1_speed()
{
  // somewhat hacked implementation of a PID algorithm as described at:
  // http://www.embedded.com/2000/0010/0010feat3.htm - PID Without a PhD, Tim Wescott 

  int abs_error = abs(speed_error);
  int pTerm = 0;
  int iTerm = 0;
  int dTerm = 0;
  int speed = 0;

  //hack for extruder not keeping up, overflowing, then shutting off.
  if (speed_error < -5000)
    speed_error = -500;
  if (speed_error > 5000)
    speed_error = 500;

  if (speed_error < 0)
  {
    //calculate our P term
    pTerm = abs_error / pGain;

    //calculate our I term
    iState += abs_error;
    iState = constrain(iState, iMin, iMax);
    iTerm = iState / iGain;

    //calculate our D term
    dTerm = (abs_error - dState) * dGain;
    dState = abs_error;

    //calculate our PWM, within bounds.
    speed = pTerm + iTerm - dTerm;
  }

  //our debug loop checker thingie
  /*
    cnt++;
   if (cnt > 250)
   {
   Serial.print("e:");
   Serial.println(speed_error);
   Serial.print("spd:");
   Serial.println(speed);
   cnt = 0;
   }
   */

  //figure out our real speed and use it.
  motor1_pwm = constrain(speed, MIN_SPEED, MAX_SPEED);

  analogWrite(MOTOR_1_SPEED_PIN, motor1_pwm);
}

// Yep, this is actually -*- c++ -*-
// These are our query commands from the host
#define SLAVE_CMD_VERSION                0
#define SLAVE_CMD_INIT                   1
#define SLAVE_CMD_GET_TEMP               2
#define SLAVE_CMD_SET_TEMP               3
#define SLAVE_CMD_SET_MOTOR_1_PWM        4
#define SLAVE_CMD_SET_MOTOR_2_PWM        5
#define SLAVE_CMD_SET_MOTOR_1_RPM        6
#define SLAVE_CMD_SET_MOTOR_2_RPM        7
#define SLAVE_CMD_SET_MOTOR_1_DIR        8
#define SLAVE_CMD_SET_MOTOR_2_DIR        9
#define SLAVE_CMD_TOGGLE_MOTOR_1        10
#define SLAVE_CMD_TOGGLE_MOTOR_2        11
#define SLAVE_CMD_TOGGLE_FAN            12
#define SLAVE_CMD_TOGGLE_VALVE          13
#define SLAVE_CMD_SET_SERVO_1_POS       14
#define SLAVE_CMD_SET_SERVO_2_POS       15
#define SLAVE_CMD_FILAMENT_STATUS       16
#define SLAVE_CMD_GET_MOTOR_1_PWM       17
#define SLAVE_CMD_GET_MOTOR_2_PWM       18
#define SLAVE_CMD_GET_MOTOR_1_RPM       19
#define SLAVE_CMD_GET_MOTOR_2_RPM       20
#define SLAVE_CMD_SELECT_TOOL           21
#define SLAVE_CMD_IS_TOOL_READY         22

//initialize the firmware to default state.
void init_commands()
{
  finishedCommands = 0;
  masterPacket.init();
}

//handle our packets.
void process_packets()
{
  //do we have any data to process?
  if (Serial.available() > 0)
  {
    unsigned long start = millis();
    unsigned long end = start + PACKET_TIMEOUT;

    //is there serial data available?
    //read through our available data
    while (!masterPacket.isFinished())
    {
      //only try to grab it if theres something in the queue.
      if (Serial.available() > 0)
      {
        digitalWrite(DEBUG_PIN, HIGH);

        //grab a byte and process it.
        byte d = Serial.read();
        masterPacket.process_byte(d);

        //keep us goign while we have data coming in.
        start = millis();
        end = start + PACKET_TIMEOUT;

        digitalWrite(DEBUG_PIN, LOW);
      }

      //are we sure we wanna break mid-packet?
      //have we timed out?
      if (millis() >= end)
        return;
    }

    //do we have a finished packet?
    if (masterPacket.isFinished())
    {
      //only process packets intended for us.
      if (masterPacket.get_8(0) == RS485_ADDRESS)
      {
        //take some action.
        handle_query();                

        //send reply over RS485
        send_reply();

        //how many have we processed?
        finishedCommands++;

        //okay, we'll come back later.
        return;
      }
    }

    //always clean up the packet.
    masterPacket.init();  
  }
}

void send_reply()
{
  //might be needed to allow for pin switching / etc.  testing needed.
  delayMicrosecondsInterruptible(50);
  
  digitalWrite(TX_ENABLE_PIN, HIGH); //enable tx

  delayMicrosecondsInterruptible(10);

  //okay, send our response
  masterPacket.sendReply();

  digitalWrite(TX_ENABLE_PIN, LOW); //disable tx
}

//this is for handling query commands that need a response.
void handle_query()
{
  byte temp;

  //which one did we get?
  switch (masterPacket.get_8(1))
  {
  //WORKING
  case SLAVE_CMD_VERSION:
    //get our host version
    master_version = masterPacket.get_16(2);

    //send our version back.
    masterPacket.add_16(FIRMWARE_VERSION);
    break;

  //WORKING
  case SLAVE_CMD_INIT:
    //just initialize
    initialize();
    break;

  //WORKING
  case SLAVE_CMD_GET_TEMP:
    masterPacket.add_16(current_temperature);
    break;

  //WORKING
  case SLAVE_CMD_SET_TEMP:
    target_temperature = masterPacket.get_16(2);
    break;

  //WORKING
  case SLAVE_CMD_SET_MOTOR_1_PWM:
    motor1_control = MC_PWM;
    motor1_pwm = masterPacket.get_8(2);
    break;

  //WORKING
  case SLAVE_CMD_SET_MOTOR_2_PWM:
    motor2_control = MC_PWM;
    motor2_pwm = masterPacket.get_8(2);
    break;

  //NEEDS TESTING
  case SLAVE_CMD_SET_MOTOR_1_RPM:
    motor1_target_rpm = masterPacket.get_32(2) * 16;
    #if MOTOR_STYLE == 1
      motor1_control = MC_ENCODER;
    #else
      motor1_control = MC_STEPPER;
      stepper_ticks = motor1_target_rpm / (MOTOR_STEPS * MOTOR_STEP_MULTIPLIER);
      stepper_high_pwm = motor1_pwm;
      stepper_low_pwm = round((float)motor1_pwm * 0.4);
    #endif
    break;

  //NEEDS TESTING
  case SLAVE_CMD_SET_MOTOR_2_RPM:
    motor2_control = MC_ENCODER;
    motor2_target_rpm = masterPacket.get_32(2);
    break;

  //WORKING
  case SLAVE_CMD_SET_MOTOR_1_DIR:
    temp = masterPacket.get_8(2);
    if (temp & 1)
      motor1_dir = MC_FORWARD;
    else
      motor1_dir = MC_REVERSE;    
    break;

  //WORKING
  case SLAVE_CMD_SET_MOTOR_2_DIR:
    temp = masterPacket.get_8(2);
    if (temp & 1)
      motor2_dir = MC_FORWARD;
    else
      motor2_dir = MC_REVERSE;    
    break;

  //WORKING
  case SLAVE_CMD_TOGGLE_MOTOR_1:
    temp = masterPacket.get_8(2);
    if (temp & 2)
      motor1_dir = MC_FORWARD;
    else
      motor1_dir = MC_REVERSE;

    if (temp & 1)
      enable_motor_1();
    else
      disable_motor_1();
    break;

  //WORKING
  case SLAVE_CMD_TOGGLE_MOTOR_2:
    temp = masterPacket.get_8(2);
    if (temp & 2)
      motor2_dir = MC_FORWARD;
    else
      motor2_dir = MC_REVERSE;

    //TODO: check to see if we're not in stepper mode.
    if (temp & 1)
      enable_motor_2();
    else
      disable_motor_2();
    break;

  //WORKING
  case SLAVE_CMD_TOGGLE_FAN:
    temp = masterPacket.get_8(2);
    if (temp & 1)
      enable_fan();
    else
      disable_fan();
    break;

  //WORKING
  case SLAVE_CMD_TOGGLE_VALVE:
    temp = masterPacket.get_8(2);
    if (temp & 1)
      open_valve();
    else
      close_valve();
    break;

  //WORKING
  case SLAVE_CMD_SET_SERVO_1_POS:
    servo1.attach(9);
    servo1.write(masterPacket.get_8(2));
    break;

  //WORKING
  case SLAVE_CMD_SET_SERVO_2_POS:
    servo2.attach(10);
    servo2.write(masterPacket.get_8(2));
    break;

  //WORKING
  case SLAVE_CMD_FILAMENT_STATUS:
    //TODO: figure out how to detect this.
    masterPacket.add_8(255);
    break;

  //WORKING
  case SLAVE_CMD_GET_MOTOR_1_PWM:
    masterPacket.add_8(motor1_pwm);
    break;

  //WORKING
  case SLAVE_CMD_GET_MOTOR_2_PWM:
    masterPacket.add_8(motor2_pwm);
    break;

  //NEEDS TESTING
  case SLAVE_CMD_GET_MOTOR_1_RPM:
    masterPacket.add_32(motor1_current_rpm);
    break;

  //NEEDS TESTING
  case SLAVE_CMD_GET_MOTOR_2_RPM:
    masterPacket.add_32(motor1_current_rpm);
    break;

  //WORKING
  case SLAVE_CMD_SELECT_TOOL:
    //do we do anything?
    break;

  //WORKING
  case SLAVE_CMD_IS_TOOL_READY:
    masterPacket.add_8(is_tool_ready());
    break;
  }
}

// Yep, this is actually -*- c++ -*-
//these routines provide an easy interface for controlling timer1 interrupts

//this handles the timer interrupt event
SIGNAL(SIG_OUTPUT_COMPARE1A)
{
  #if MOTOR_STYLE == 1
    //do encoder stuff here.
  #else
    //increment our index
    if (motor1_dir == MC_FORWARD)
      stepper_index = (stepper_index - 1) % 8;
    else
      stepper_index = (stepper_index + 1) % 8;
      
    //coil 1
    digitalWrite(MOTOR_1_DIR_PIN,   coil_a_direction & (1 << stepper_index));
    if (coil_a_enabled & (1 << stepper_index))
      analogWrite(MOTOR_1_SPEED_PIN, stepper_high_pwm);
    else
      analogWrite(MOTOR_1_SPEED_PIN, stepper_low_pwm);

    //coil 2
    digitalWrite(MOTOR_2_DIR_PIN,   coil_b_direction & (1 << stepper_index));
    if (coil_b_enabled & (1 << stepper_index))
      analogWrite(MOTOR_2_SPEED_PIN, stepper_high_pwm);
    else
      analogWrite(MOTOR_2_SPEED_PIN, stepper_low_pwm);
  #endif
}

void enableTimer1Interrupt()
{
  //enable our interrupt!
  TIMSK1 |= (1<<OCIE1A);
}

void disableTimer1Interrupt()
{
  TIMSK1 &= ~(1<<ICIE1);
  TIMSK1 &= ~(1<<OCIE1A);
}

void setTimer1Resolution(byte r)
{
  //from table 15-5 in that atmega168 datasheet:
  // we're setting CS12 - CS10 which correspond to the binary numbers 0-5
  // 0 = no timer
  // 1 = no prescaler
  // 2 = clock/8
  // 3 = clock/64
  // 4 = clock/256
  // 5 = clock/1024

  if (r > 5)
    r = 5;

  TCCR1B &= B11111000;
  TCCR1B |= r;
}

void setTimer1Ceiling(unsigned int c)
{
  OCR1A = c;
}


unsigned int getTimer1Ceiling(unsigned long ticks)
{
  // our slowest speed at our highest resolution ( (2^16-1) * 0.0625 usecs = 4095 usecs)
  if (ticks <= 65535L)
    return (ticks & 0xffff);
  // our slowest speed at our next highest resolution ( (2^16-1) * 0.5 usecs = 32767 usecs)
  else if (ticks <= 524280L)
    return ((ticks / 8) & 0xffff);
  // our slowest speed at our medium resolution ( (2^16-1) * 4 usecs = 262140 usecs)
  else if (ticks <= 4194240L)
    return ((ticks / 64) & 0xffff);
  // our slowest speed at our medium-low resolution ( (2^16-1) * 16 usecs = 1048560 usecs)
  else if (ticks <= 16776960L)
    return (ticks / 256);
  // our slowest speed at our lowest resolution ((2^16-1) * 64 usecs = 4194240 usecs)
  else if (ticks <= 67107840L)
    return (ticks / 1024);
  //its really slow... hopefully we can just get by with super slow.
  else
    return 65535;
}

byte getTimer1Resolution(unsigned long ticks)
{
  // these also represent frequency: 1000000 / ticks / 2 = frequency in hz.

  // our slowest speed at our highest resolution ( (2^16-1) * 0.0625 usecs = 4095 usecs (4 millisecond max))
  // range: 8Mhz max - 122hz min
  if (ticks <= 65535L)
    return 1;
  // our slowest speed at our next highest resolution ( (2^16-1) * 0.5 usecs = 32767 usecs (32 millisecond max))
  // range:1Mhz max - 15.26hz min
  else if (ticks <= 524280L)
    return 2;
  // our slowest speed at our medium resolution ( (2^16-1) * 4 usecs = 262140 usecs (0.26 seconds max))
  // range: 125Khz max - 1.9hz min
  else if (ticks <= 4194240L)
    return 3;
  // our slowest speed at our medium-low resolution ( (2^16-1) * 16 usecs = 1048560 usecs (1.04 seconds max))
  // range: 31.25Khz max - 0.475hz min
  else if (ticks <= 16776960L)
    return 4;
  // our slowest speed at our lowest resolution ((2^16-1) * 64 usecs = 4194240 usecs (4.19 seconds max))
  // range: 7.812Khz max - 0.119hz min
  else if (ticks <= 67107840L)
    return 5;
  //its really slow... hopefully we can just get by with super slow.
  else
    return 5;
}

void setTimer1Ticks(unsigned long ticks)
{
  // ticks is the delay between interrupts in 62.5 nanosecond ticks.
  //
  // we break it into 5 different resolutions based on the delay. 
  // then we set the resolution based on the size of the delay.
  // we also then calculate the timer ceiling required. (ie what the counter counts to)
  // the result is the timer counts up to the appropriate time and then fires an interrupt.

  setTimer1Ceiling(getTimer1Ceiling(ticks));
  setTimer1Resolution(getTimer1Resolution(ticks));
}

void setupTimer1Interrupt()
{
  //clear the registers
  TCCR1A = 0;
  TCCR1B = 0;
  TCCR1C = 0;
  TIMSK1 = 0;

  //waveform generation = 0100 = CTC
  TCCR1B &= ~(1<<WGM13);
  TCCR1B |=  (1<<WGM12);
  TCCR1A &= ~(1<<WGM11); 
  TCCR1A &= ~(1<<WGM10);

  //output mode = 00 (disconnected)
  TCCR1A &= ~(1<<COM1A1); 
  TCCR1A &= ~(1<<COM1A0);
  TCCR1A &= ~(1<<COM1B1); 
  TCCR1A &= ~(1<<COM1B0);

  //start off with a slow frequency.
  setTimer1Resolution(5);
  setTimer1Ceiling(65535);
  disableTimer1Interrupt();
}

void delayMicrosecondsInterruptible(unsigned int us)
{
  // for a one-microsecond delay, simply return.  the overhead
  // of the function call yields a delay of approximately 1 1/8 us.
  if (--us == 0)
    return;

  // the following loop takes a quarter of a microsecond (4 cycles)
  // per iteration, so execute it four times for each microsecond of
  // delay requested.
  us <<= 2;

  // account for the time taken in the preceeding commands.
  us -= 2;

  // busy wait
  __asm__ __volatile__ ("1: sbiw %0,1" "\n\t" // 2 cycles
"brne 1b" : 
  "=w" (us) : 
  "0" (us) // 2 cycles
    );
}

int main(void)
{
	init();

	setup();
    
	for (;;)
		loop();
        
	return 0;
}

