#include <openrave-core.h>
#include <vector>
#include <cstring>
#include <sstream>
#include <math.h>
#include <pthread.h>

#include <boost/thread/thread.hpp>
#include <boost/bind.hpp>

using namespace OpenRAVE;
using namespace std;


void SetViewer(EnvironmentBasePtr penv, const string& viewername)
{
    RaveViewerBasePtr viewer = penv->CreateViewer(viewername);
    BOOST_ASSERT(!!viewer);

    // attach it to the environment:
    penv->AttachViewer(viewer);

    // finally you call the viewer's infinite loop (this is why you need a separate thread):
    bool showgui = true;
    viewer->main(showgui);

}

void servo_update(EnvironmentBasePtr penv, KinBody::JointPtr joint, dReal ref)
{
  const dReal KP = 8.0;
  dReal error;
  std::vector<dReal> angle;
  std::vector<dReal> pJointVelocities(1);
  const dReal WMAX=4.0;

  ref = ref * PI/180.0;

  //-- Get the current angle
  joint->GetValues(angle);

  error = angle[0] - ref;
  dReal velocity = -error*KP;

  if (velocity > WMAX) velocity = WMAX;
  if (velocity < -WMAX) velocity = -WMAX;

  pJointVelocities[0]=velocity;
  penv->GetPhysicsEngine()->SetJointVelocity (joint, pJointVelocities);
}

int main(int argc, char ** argv)
{
    // create the main environment
    EnvironmentBasePtr penv = CreateEnvironment(true);
    penv->StopSimulation();

    boost::thread thviewer(boost::bind(SetViewer,penv,"qtcoin"));

    {
        // lock the environment to prevent changes
        EnvironmentMutex::scoped_lock lock(penv->GetMutex());

        // load the scene
        penv->Load("./test30.xml");

    }

    //-- Get the physical engine
    PhysicsEngineBasePtr pe = penv->GetPhysicsEngine();

    //-- KinBody
    std::vector<KinBodyPtr> bodies;
    penv->GetBodies(bodies);

    //-- Joints
    KinBody::JointPtr joint0 = bodies[0]->GetJoints()[0];
    KinBody::JointPtr joint1 = bodies[0]->GetJoints()[1];


    //-- Constants
    const float T=2;
    const int N=20;
    const float p = T/N;
    const float A0 = 30;
    const float A1 = 30;
    const float phase_ini0 = 0;
    const float phase_ini1 = -140;

    float angle0,angle1;
    int sample_count=0;
    float phase;
    int simticks=p/0.01;

    cout << "Sampling period in ticks: " << simticks << endl;

    char tecla;

    cout << "PRESS A KEY TO START THE SIMULATION!" << endl;
    cin >> tecla;

    do {
      phase = 2*PI/N * sample_count;
      sample_count=(sample_count + 1)%N;
      angle0 = A0 * sin(phase + phase_ini0*PI/180.0);
      angle1 = A1 * sin(phase + phase_ini1*PI/180.0);

      penv->StopSimulation();
      for (int count=0; count<simticks; count++) {
        penv->StepSimulation(0.01);
        usleep(5000);
        servo_update(penv,joint0,angle0);
        servo_update(penv,joint1,angle1);
      }
      penv->StartSimulation(0.01);

    } while(1);

    penv->Destroy(); // destroy
    return 0;
}
