#include <openrave-core.h>
#include <vector>
#include <cstring>
#include <sstream>
#include <math.h>
#include <pthread.h>

#include <boost/thread/thread.hpp>
#include <boost/bind.hpp>

using namespace OpenRAVE;
using namespace std;


void SetViewer(EnvironmentBasePtr penv, const string& viewername)
{
    RaveViewerBasePtr viewer = penv->CreateViewer(viewername);
    BOOST_ASSERT(!!viewer);

    // attach it to the environment:
    penv->AttachViewer(viewer);

    // finally you call the viewer's infinite loop (this is why you need a separate thread):
    bool showgui = true;
    viewer->main(showgui);

}

void servo_pos(EnvironmentBasePtr penv, KinBody::JointPtr joint, dReal ref, int simticks)
{
    const dReal KP = 8.3;
    dReal error;
    std::vector<dReal> angle;
    std::vector<dReal> pJointVelocities(1);

    for (int count=0; count<simticks; count++) {
        penv->StepSimulation(0.01);
        usleep(20000);

        //-- Get the current angle
        joint->GetValues(angle);

        error = angle[0] - ref;
        dReal velocity = -error*KP;

        pJointVelocities[0]=velocity;
        penv->GetPhysicsEngine()->SetJointVelocity (joint, pJointVelocities);
    }

}

int main(int argc, char ** argv)
{
    // create the main environment
    EnvironmentBasePtr penv = CreateEnvironment(true);
    penv->StopSimulation();

    boost::thread thviewer(boost::bind(SetViewer,penv,"qtcoin"));

    {
        // lock the environment to prevent changes
        EnvironmentMutex::scoped_lock lock(penv->GetMutex());

        // load the scene
        penv->Load("./test08.xml");

    }

    //-- KinBody
    std::vector<KinBodyPtr> bodies;
    penv->GetBodies(bodies);
    KinBodyPtr kinbody = bodies[0];

    //-- Get the physical engine
    PhysicsEngineBasePtr pe = penv->GetPhysicsEngine();

    //-- Get the joints
    const std::vector<KinBody::JointPtr>& vjoints = kinbody->GetJoints();
    KinBody::JointPtr joint = vjoints[0];

    const float T=1.5;
    const int N=20;
    const float p = T/N;

    cout << "Sample period: " << p << endl;

    cout << "Simu. time: " <<   penv->GetSimulationTime() << endl;
    penv->StepSimulation(0.01);
    cout << "Simu. time: " <<   penv->GetSimulationTime() << endl;

    float angle;
    int sample_count=0;
    float phase;
    int simticks=p/0.01;

    cout << "Sampling period in ticks: " << simticks << endl;

    do {

      phase = 2*PI/N * sample_count;
      sample_count=(sample_count + 1)%N;
      angle = 80 * sin(phase);

      penv->StopSimulation();
      servo_pos(penv,joint, angle*PI/180.0,simticks);
      penv->StartSimulation(0.01);

    } while(1);

    penv->Destroy(); // destroy
    return 0;
}
