#include <openrave-core.h>
#include <vector>
#include <cstring>
#include <sstream>

#include <boost/thread/thread.hpp>
#include <boost/bind.hpp>

using namespace OpenRAVE;
using namespace std;


void SetViewer(EnvironmentBasePtr penv, const string& viewername)
{
    RaveViewerBasePtr viewer = penv->CreateViewer(viewername);
    BOOST_ASSERT(!!viewer);

    // attach it to the environment:
    penv->AttachViewer(viewer);

    // finally you call the viewer's infinite loop (this is why you need a separate thread):
    bool showgui = true;
    viewer->main(showgui);

}

int main(int argc, char ** argv)
{
    // create the main environment
    EnvironmentBasePtr penv = CreateEnvironment(true);

    boost::thread thviewer(boost::bind(SetViewer,penv,"qtcoin"));

    {
        // lock the environment to prevent changes
        EnvironmentMutex::scoped_lock lock(penv->GetMutex());

        // load the scene
        penv->Load("./prueba05.xml");

    }

    //-- Obtener los robots
    std::vector<RobotBasePtr> robots;
    penv->GetRobots(robots);

    //-- Robot 0
    RobotBasePtr robot = robots[0];
    cout << "Robot: " << robot->GetName() << endl;

    //-- KinBody
    std::vector<KinBodyPtr> bodies;
    penv->GetBodies(bodies);
    KinBodyPtr kinbody = bodies[0];
    cout << "KinBody: " << kinbody->GetName() << endl;

    //-- Set the joint value
    std::vector< dReal > vJointValues(1,0.2);
    kinbody->SetJointValues(vJointValues);

    //-- Motor fisico
    PhysicsEngineBasePtr pe = penv->GetPhysicsEngine();

    //-- Obtener articulaciones
    const std::vector<KinBody::JointPtr> & vjoints = kinbody->GetJoints();
    KinBody::JointPtr joint = vjoints[0];
    cout << "Joint: " << joint->GetName() << endl;

    std::vector<dReal> pJointVelocity;
    pe->GetJointVelocity(joint, pJointVelocity);
    cout << "VEL:" << pJointVelocity[0] << endl;

    //-- Set the joint velocity
    const std::vector<dReal> pJointVelocities(1,2.0);
    pe->SetJointVelocity (joint, pJointVelocities);


    thviewer.join(); // wait for the viewer thread to exit
    penv->Destroy(); // destroy
    return 0;
}
