/********************************************************************/
/** Module servos.c                                                 */
/**-----------------------------------------------------------------*/
/** Module for positioning up to 8 servos Futaba 3003 or compatible */
/** The pwm signal is generated by means of the timer 0 overflow    */
/** Interruption. The signals are written to pins RB0 - RB7         */
/**-----------------------------------------------------------------*/
/** (c) Andres Prieto-Moreno                                        */
/** (c) Dr. Juan Gonzalez-Gomez.                                    */
/** Feb/2011                                                        */
/**-----------------------------------------------------------------*/
/** GPL license                                                     */
/********************************************************************/

//-- Specify the PIC micro-controller used in the Skycube board
#include <pic16f876a.h>

//---- Some constants related to servo timming --------------------------
#define DELAY_0_3ms 0xEA    // Time: 0.3ms with prescaler set to 64
#define DELAY_2_2ms 0x52    // Time: 2.2ms, prescaler set to 64

//-- Servo home position (in time). It corresponds to 0 degrees
#define SERVO_HOME      0xB7//-- Specify the PIC micro-controller used in the Skycube board
#include <pic16f876a.h>

//-- The servo module is being used
#include <servos.h>

/************************************************************/
/* EXPORTED VARIABLES. They are visible outside this module */
/************************************************************/

//-- Mask for enabling/disabling the PWM generation for each servo
//-- Bits to '0' means that the servos are disablee (PWM signal not generated)
//-- Bits to '1' means that the servos are enabled (PWM signal is generated)
unsigned char enable_mask;

// -- Servo position, in time units
volatile unsigned char servo_pos[8];

//-- Current servo index
unsigned char index;

//-- Flag to indicate if a new time-window of 2.5ms is started
unsigned char tic;

//-- Current servo in binary. Only one bit is active that indicates the
//-- current servo. It is used for optimization
unsigned char current_servo;

/***********************************/
/* MODULE INTERNAL VARIABLES       */
/***********************************/

//-- State of the Finit state machine that controlls the PWM generation
//-- of the 8 servos
static unsigned char state;

//-- Temporal calculations
static unsigned char pos;

/********************************************************************/
/** INTERRUPTION ROUTINE SERVICE                                    */
/** The overflow timer0 interrupt is used                           */
/** Each servo is controlled by a time-window of 2.5. Each          */
/** time-window can be in three different states:                   */
/**  -> State 0: The PWM signal of the current servo is set to 1    */
/**       It last 0.3ms. It is the minimal pulse width in time      */
/**  -> State 1: The pulse width in time is set for this servo      */
/**              This width depend on the desired servo position    */
/**               se situe el servo                                 */
/**  -> State 2: All the PWM signals are set to 0. The time is      */
/**        calculated so that the total time since the state 0      */
/**         started is 2.5ms (the time-window width)                */
/********************************************************************/

void servos_intr()
{
  // Clear TIMER0 overflow flag
  T0IF=0;

  //-- Initial state. Pulse width of 0.3ms
  if (state==0) {
    //-- Set the PWM signal of the current servo to '1', if it is enable
    PORTB=enable_mask & current_servo;

    //-- Launch the timer for calculating a fixed time of 0.3ms
    TMR0=DELAY_0_3ms;

    //-- Set the next state for the following Timer0 interruption
    state=1;

    //-- A new time-window of 2.5ms has started
    tic=1;
  }
  //-- State 1: Variable Pulse width
  else if (state==1) {

    //-- Servo pulse width. It depends on the servo position
    TMR0=servo_pos[index];

    //-- Set the next state for the following Timer0 interruption
    state=2;
  }
  //-- Final state. Set the pwm signal to 0. Calculate the remaining time
  //-- to complete a 2.5ms time-windows
  else if (state==2) {
    //--  All the pwm signal should be 0
    PORTB=0;

    //-- Calculate the time of this state so that the next timer0 interruption
    //-- is generated after 2.5ms from the beginin (state 0)
    TMR0=DELAY_2_2ms-servo_pos[index];

    //-- The next state will be the initial
    state=0;

    //-- Change the servo to the next one
    current_servo=current_servo<<1;
    index++;

    //-- If the 8 PWM signals have been generated, start again from servo 0
    //-- el servo 0
    if (current_servo==0x00) {
      current_servo=1;
      index=0;
    }
  }
  else {
    // --  It should never be reached!!!
    //-- Just in case it happens, return to the initial state
    state=0;
  }
}

/****************************************/
/**  Timer 0 configuration              */
/****************************************/
void timer0_configure() {
  // TOCS= Timer mode bit5=0
  // PS2:PS0 prescaler set to 64
  // PSA=0  The prescaler is asigned to timer0
  OPTION_REG=0x05;

  //
  T0IF=0;  //-- Clear timer0 overflow interruption flag
  T0IE=1;  //-- Enable timer0 overflow interruption
  TMR0=0;  //-- Timer0 register is initialized
}

/*----------------------------------------------------*/
/*    I N T E R F A C E     F U N C T I O N S         */
/*----------------------------------------------------*/

/******************************************************/
/** Convert between degrees and time units            */
/** INPUTS:                                           */
/**    -degrees: Servo position in degrees (-90,90)   */
/******************************************************/
int servos_degrees2time(int degrees)
{
  int result;
  volatile int g;
  volatile float temp;
  volatile int t2;

  //-- The equation for calculating the pulse width in time (timer0 units)
  //-- in fuction of the servo position in degrees is: time = degrees*0.8 + 72;

  //-- In the SDCC compiler (version 2.8.0 and 2.9.0) there is a bug that prevent to
  //-- perform operations with negativle floating point numbers.
  //-- For that reason the calculation is divided into two parts, depending on the
  //-- sign of the result

  if (degrees>=0)
    result = (int) (degrees*0.8 + 72);
  else {
    //-- Calculate the value of pos_grados*0.8 as an positive integer number
    g = -degrees;
    temp = g*0.8;
    t2 = (int)temp;

    //-- And then the substraction is done
    result = 72 - t2;
   }

  //-- Return the result
  return 0xFF-result;
}

/*****************************************************************/
/** Set the servo position in degrees                            */
/** INPUTS:                                                      */
/**    -servo: Servo number (0-7). The labels S1-S8 can be used  */
/**    -degrees: Position angle in degrees (-90,90)              */
/*****************************************************************/
void servos_set(int servo, int degrees)
{
  int pos;

  //-- Convert from degrees to time
  pos = servos_degrees2time(degrees);

  //-- Set the new servo position (in time)
  servo_pos[servo]=pos;
}

/*****************************************************************/
/** Set the servo position (in time units)                       */
/** INPUTS:                                                      */
/**    -servo: Servo number (0-7). The labels S1-S8 can be used  */
/**    -time: Servo position in time                             */
/*****************************************************************/
void servos_set_raw(int servo, int time)
{
  servo_pos[servo]=time;
}

/*************************************************************************/
/** Set servo enable/disable mask                                        */
/** INPUTS:                                                              */
/**   -mask: The byte used for the enable/disable mask                   */
/**          Bits set to '1' means the corresponding servo is enable     */
/**          Bits set to '0' means the corresponding servo is disable    */
/*************************************************************************/
void servos_enable_mask(unsigned char mask)
{
  enable_mask = mask;
}

/*******************************************************************/
/** Enable one servo                                               */
/** INPUTS:                                                        */
/**   -servo:  Servo number (0-7). The labels S1-S8 can be used    */
/*******************************************************************/
void servos_enable(unsigned char servo)
{
  enable_mask = enable_mask | (1<<servo);
}

/****************************************************************/
/** Disable one servo                                           */
/** ENTRADAS:                                                   */
/**   -servo:  Servo number (0-7). The labels S1-S8 can be used */
/****************************************************************/
void servos_disable(unsigned char servo)
{
  enable_mask = enable_mask & ~(1<<servo);
}

/*****************************************************************/
/** Servo module initialization                                  */
/** This function should be called BEFORE any other interface    */
/** function is invoked                                          */
/** Configurar el modulo servos para funcionar                   */
/*****************************************************************/
void servos_init(void)
{
  unsigned char i;

  //-- Configure the whole PORTB as a digital output
  TRISB=0x00;

  //-- Variable initialization
  index=0;
  current_servo=1;
  state=0;
  tic=0;

  //-- Move the servo to their home positions (in the middle, 0 degrees)
  for (i=0; i<8; i++) {
    servo_pos[i]=SERVO_HOME;
  }

  //-- By default, al the PWM signals are disabled
  enable_mask=0x00;

  //-- Timer0 configuration
  timer0_configure();
}

