/*******************************************************************/
/** OSCILLATOR MODULE                                              */
/**----------------------------------------------------------------*/
/** Library for generating the oscillation of the VIRTUAL MODULES  */
/** This library is written for working with 3 VIRTUAL MODULES     */
/**----------------------------------------------------------------*/
/** (c) Dr. Juan Gonzalez-Gomez.                                   */
/** Feb/2011                                                       */
/**----------------------------------------------------------------*/
/** GPL license                                                    */
/*******************************************************************/

//-- Specify the PIC micro-controller used in the Skycube board
#include <pic16f876a.h>

//-- The servo and sine module are being used
#include "servos.h"
#include "sin.h"

/******************************/
/** EXPORTED VARIABLES        */
/******************************/

int oscillator_periods=0;     //-- Number of oscillation periods performed
int oscillator_new_period=0;  //-- Flag. A new period is started

/**********************************************/
/** GLOBAL VARIABLES                          */
/**********************************************/

//-- Oscillations amplitudes. Degrees. Range: [0,90]
//----- Servos   1  2  3  4  5  6  7  8
static int A[8]={0, 0, 0, 0, 0, 0, 0, 0};

//-- Oscillations initial phases. Degrees. Range: [0,360]
static int PHASE[8]={0,0,0,0,0,0,0,0};

//-- Offset. Degrees. Range [-90,90]
static int OFFSET[8]={0,0,0,0,0,0,0,0};

//-- Phase increment (INC)
static int INCm=0;

//-- Initial phase (for all the modules). Degrees. Range: [0,360]
static int PHASE0=0;

//-- Phase variale. It changes with time
static int phase;

//-- Global variables imported from the SERVOS module
extern volatile unsigned char servo_pos[8];
extern unsigned char index;
extern unsigned char tic;
extern unsigned char current_servo;

//-- External VIRTUAL MODULES VARIABLES
//-- They should be defined in the MAIN program
extern unsigned char v[];
extern __code int INC[];

//-- Virtual module 1 -------
extern __code int A1[];
extern __code int O1[];
extern __code int P1[];

//-- Virtual module 2 -------
extern __code int A2[];
extern __code int O2[];
extern __code int P12[];

//-- Virtual module 3 -------
extern __code int A3[];
extern __code int O3[];
extern __code int P13[];


/*******************************************************/
/**  INTERRUPTION ROUTINE SERVICE                      */
/*******************************************************/
void intr(void) interrupt 0
{
  //-----------------------------------------------------------
  //-- Interruption routine service of timer0 for generating
  //-- de PWM signals for the servos. It is implemented in the
  //-- servo module
  //-----------------------------------------------------------
  if (T0IF==1)
    servos_intr();
}

/*****************************************************************************/
/** Module initialization. It should be called before any other function of  */
/** this module.  After calling it the interruptions are enabled and the     */
/** servos will start moving                                                 */
/*****************************************************************************/
void oscillator_init()
{
  //-- Initialize the servo module
  servos_init();

  //-- Phase variable initialized
  phase=0;

  //-- Enable all the servos. However in this example only one servo is moved
  servos_enable_mask(SERVOS_ALL);

  //-- Enable the global interrupts. Now the PWM signals for controlling the
  //-- servos are being generated
  GIE=1;
}

/*************************************************************/
/** Calculate the real servo parameters according to the     */
/** virtual servo parameters                                 */
/** INPUTS:                                                  */
/**   -Sequence decriptor. It is used to index the           */
/**    virtual modules array of parameters. This way it is   */
/**    very easy to generated differnt sequences of movement */
/*************************************************************/
void oscillator_set_params(unsigned char m)
{
  unsigned char j;

  //-- The current implementation is done for only 3 modules

  INCm    = INC[m];
  PHASE0  = P1[m];

  //------- Virtual module 1 ---------------
  j=v[1];  //-- Get the real servo number

  if (j!=0) {
    A[j-1]     =A1[m];
    OFFSET[j-1]=O1[m];
    PHASE[j-1]  =0;
  }

  //------- Virtual module 2 ----------------
  j=v[2];  //-- Get the real servo number

  if (j!=0) {
    A[j-1]     =A2[m];
    OFFSET[j-1]=O2[m];
    PHASE[j-1] =P12[m];
  }

  //------- Virtual module 3 ----------------
  j=v[3];  //-- Get the real servo number

  if (j!=0) {
    A[j-1]     =A3[m];
    OFFSET[j-1]=O3[m];
    PHASE[j-1] =P13[m];
  }

  phase=0;


}


/***************************************************************************/
/**  Get the next position for the servos. The calculations are perfomed   */
/**  for every time-window (every 2.5ms)                                   */
/***************************************************************************/
void oscillator_pos()
{
  int pos_sin;
  unsigned char upos;
  unsigned char i;
  unsigned char s;

    //-- The external variable tic is set to '1' every 2.5ms (it is updated
    //--  by the timer0 interruption service routine in the SERVOS module)
    if (tic==1) {

      //-- Reset the tic variable. After 2.5ms it will be set to 1 again
      tic=0;

      //-- Get the current servo number (index variable) and its binary mask
      //-- Both of them are in the SERVOS module
      //-- This operation is VERY IMPORTANT! This two global variables are
      //-- changed asynchronously by the interruptions. It is necessary to
      //-- have an unmodified local copy into the i and s variables
      i=index;
      s=current_servo;

      //-- Calculate the next position for the servo i
      pos_sin=sin(A[i],phase+PHASE[i]+PHASE0)+OFFSET[i];

      //-- Convert from degrees to time
      upos = servos_degrees2time(pos_sin);

      //-- Update the servo position
      servo_pos[i]=upos;

      //-- If the current servo is the first, increment the phase variable
      if (i==0) {
        //-- Increment the phase variable
        phase=phase + INCm;

        if (phase>=360) {
          oscillator_new_period=1;
          oscillator_periods++;
        }

        //-- The negative phases are converted into positive
        //-- BUG: para fases negativas deveria ser: fase=360-fase
        if (phase<0) phase=359;

        //-- Keep the phase in the range [0,360]
        phase = phase % 360;
      }
    }

}




