/*******************************************************************/
/** Test-oscillations-1.c                                          */
/**----------------------------------------------------------------*/
/** Example of generating sinusoidal oscillations for 8 servos     */
/**----------------------------------------------------------------*/
/** (c) Dr. Juan Gonzalez-Gomez.                                   */
/** (c) Andres Prieto-Moreno                                       */
/** Feb/2011                                                       */
/**----------------------------------------------------------------*/
/** GPL license                                                    */
/*******************************************************************/

//-- Specify the PIC micro-controller used in the Skycube board
#include <pic16f876a.h>

//-- The servo and sine module are being used
#include "servos.h"
#include "sin.h"

/*******************************************/
/**  OSCILLATIONS PARAMETERS               */
/*******************************************/

//-- Phase increment (INC) (in degreess). Range: 1-180
//--
//--   This parameters determines the frequency of the oscillation as the starting
//-- direction. It is an ingetener number (positive and negatives values are ok)
//--   The period of the oscillation (T) can be calculated with this expresion:
//--  T=360/W, where W is express in degrees/sec. For the futaba servos the angular
//--  speed is approximately 260 deg/sec. There, the final formula is:
//--
//--  T=7.2/INC
//--
//--  A table with different values of INC, T and f is shown:
//--
//--      INC      T (sec)   f(Hz)
//--    -----------------------------
//--       0       no osc.     0
//--       1       7.2         0.14
//--       2       3.7         0.28
//--       3       2,4         0.42
//--       4       1.8         0.55
//--       5       1.4         0.71
//--       6       1.2         0.83
//--       7       1.03        0.97
//--       8       0.9         1.11
//--       9       0.8         1.25
//--      10       0.72        1.39
//--      ..       ...         ...

//-- Amplitude: A[i].       Range: [0,90] (degrees)
//-- Phase:     PHASE[I].   Range: [0,360] (degrees)
//-- Offset:    OFFSET[i]:  Range: [-90,90] (degrees)

//-- Phase inc: INC. The same for all the oscillators
//-- Initial phase: PHASE0. Range: [0,360] (degrees). The same for all osc.

//-- The i servo angle is calculate as follows:
//--   servo_i_pos = A[i] * sin(phase + PHASE[I] + PHASE0) + OFFSET[i]
//-- Everty 20ms the phase is incremented: phase = phase + INC


/**********************************************/
/** GLOBAL VARIABLES                          */
/**********************************************/

//-- Oscillations amplitudes. Degrees. Range: [0,90]
//----- Servos   1  2  3  4  5  6  7  8
static int A[8]={80,40,80,80,80,80,80,80};

//-- Oscillations initial phases. Degrees. Range: [0,360]
static int PHASE[8]={0,45,90,135,180,225,270,315};

//-- Offset. Degrees. Range [-90,90]
static int OFFSET[8]={0,30,0,0,0,0,0,0};

//-- Phase increment (INC)
static int INC=3;

//-- Initial phase (for all the modules). Degrees. Range: [0,360]
static int PHASE0=0;

//-- Phase variale. It changes with time
static int phase;

//-- Global variables imported from the SERVOS module
extern volatile unsigned char servo_pos[8];
extern unsigned char index;
extern unsigned char tic;
extern unsigned char current_servo;

/*******************************************************/
/**  INTERRUPTION ROUTINE SERVICE                      */
/*******************************************************/
void intr(void) interrupt 0
{
  //-----------------------------------------------------------
  //-- Interruption routine service of timer0 for generating
  //-- de PWM signals for the servos. It is implemented in the
  //-- servo module
  //-----------------------------------------------------------
  if (T0IF==1)
    servos_intr();
}


/***************************************************************************/
/**  Get the next position for the servos. The calculations are perfomed   */
/**  for every time-window (every 2.5ms)                                   */
/***************************************************************************/
void oscillator_calculate()
{
  int pos_sin;
  unsigned char upos;
  unsigned char i;
  unsigned char s;

    //-- The external variable tic is set to '1' every 2.5ms (it is updated
    //--  by the timer0 interruption service routine in the SERVOS module)
    if (tic==1) {

      //-- Reset the tic variable. After 2.5ms it will be set to 1 again
      tic=0;

      //-- Get the current servo number (index variable) and its binary mask
      //-- Both of them are in the SERVOS module
      //-- This operation is VERY IMPORTANT! This two global variables are
      //-- changed asynchronously by the interruptions. It is necessary to
      //-- have an unmodified local copy into the i and s variables
      i=index;
      s=current_servo;

      //-- Calculate the next position for the servo i
      pos_sin=sin(A[i],phase+PHASE[i]+PHASE0)+OFFSET[i];

      //-- Convert from degrees to time
      upos = servos_degrees2time(pos_sin);

      //-- Update the servo position
      servo_pos[i]=upos;

      //-- If the current servo is the first, increment the phase variable
      if (i==0) {
        //-- Increment the phase variable
        phase=phase + INC;

        //-- The negative phases are converted into positive
        //-- BUG: para fases negativas deveria ser: fase=360-fase
        if (phase<0) phase=359;

        //-- Keep the phase in the range [0,360]
        phase = phase % 360;
      }
    }

}

//----------------------------
//- MAIN PROGRAM
//----------------------------

void main(void)
{

  //-- Initialize the servo module
  servos_init();

  //-- Phase variable initialized
  phase=0;

  //-- Enable all the servos. However in this example only one servo is moved
  servos_enable_mask(SERVOS_ALL);

  //-- Enable the global interrupts. Now the PWM signals for controlling the
  //-- servos are being generated
  GIE=1;


  //----------------------
  //-- Main loop
  //----------------------
  while (1) {

     //-- Calculate the Oscillators values
     oscillator_calculate();
  }
}
