/**************************************************************************/
/** Cube6-2D example 6    Segment 1                                       */
/**-----------------------------------------------------------------------*/
/** Locomotion of the Cube6-2D configuration                              */
/** Example of a sequence of movements                                    */
/**                                                                       */
/** Summary of the gait's parameters                                      */
/**                                                                       */
/**    Gait           Av  Ah  Oh  PDv  PDh  PDvh                          */
/** -----------------------------------------------                       */
/**  Straight         20  0    0  120   0     0                           */
/**  Turning          20  0   30  120   0     0                           */
/**  Sidewinding      20  40   0  110  110   90                           */
/**  Rotating         20  40   0  180  90     0                           */
/**  Rolling          45  45   0   0   0     90                           */
/**                                                                       */
/**  The parameters for each virtual servo are given by:                  */
/** SEGMENT 1                                                             */
/**    Virtual servo 1:   A1=Av, O1=0,  P1=0                              */
/**    Virtual servo 2:   A2=Ah, O2=Oh, P2=PDvh                           */
/**    Virtual servo 3:   A3=Av, O3=0,  P3=PDv                            */
/** SEGMENT 2                                                             */
/**    Virtual servo 1:   A1=Ah, O1=Oh, P1=PDh+PDvh                       */
/**    Virtual servo 2:   A2=Av, O1=0,  P2=2*PDv-P1}                      */
/**    Virtual servo 3:   A3=Ah, O1=Oh, P3=PDh                            */
/**                                                                       */
/**               Straight   Turning   Sidewinding  Rotating  Rolling     */
/**  SEGMENT 1                                                            */
/**      A1:        20         20          20         20        45        */
/**      O1:        0           0          0          0          0        */
/**      P1:        0           0          0          0          0        */
/**                                                                       */
/**      A2:        0           0          40         40        45        */
/**      O2:        0          30          0          0          0        */
/**      P2:        0           0          90         0         90        */
/**                                                                       */
/**      A3:        20         20          20         20        45        */
/**      O4:        0           0          0          0         0         */
/**      P3:        120        120         110        180       0         */
/*                                                                        */
/**  SEGMENT 2                                                            */
/**      A1:        0          0           40         40        45        */
/**      O1:        0          30          0          0         0         */
/**      P1:        0          0           200        90        90        */
/**                                                                       */
/**      A2:        20         20          20         20        45        */
/**      O2:        0          0           0          0         0         */
/**      P2:        240        240         20         -90                 */
/**                                                                       */
/**      A3:        0          0           40         40        45        */
/**      O4:        0          30          0          0          0        */
/**      P3:        0          0           110        90        90        */
/**                                                                       */
/** --------------------------------------------------------------------- */
/** (c) Dr. Juan Gonzalez-Gomez.                                          */
/** Feb/2011                                                              */
/**-----------------------------------------------------------------------*/
/** GPL license                                                           */
/**************************************************************************/
//-- Specify the PIC micro-controller used in the Skycube board
#include <pic16f876a.h>

//-- Oscillator library
#include "oscillator.h"

//-- Location of the test button
#define TEST_BUTTON RA0

//-- Constant for determining the test button state
#define PRESSED  0
#define RELEASED 1

/**************************************/
/** LOCOMOTION PARAMETERS             */
/**************************************/
//-- This parameters determine the locomotion. They can be changed by the
//-- user. It is not necessary to change the code, just this parameters
#define I   3     //-- Phase Increment

/**********************************************/
/** GLOBAL VARIABLES                          */
/**********************************************/

//-- VIRTUAL SERVOS!
//-- For locomotion porposes we will think of robots as composed of virtual
//-- modules. Virtual module 1 is the tail, 2 is in the middle and 3 is the head
//-- The mapping between the virtual servos and the PWM channel to witch they are
//-- attached is determined by the v variable

//-- A value of '0' means module not used
//-- Virtual servos:              1  2  3
unsigned char v[]={0,             2, 6, 8};

//-- The meaning of the previous assigment is the following:
//-- Only the virtual servo 1 is used, that is connectec to the pwm channel 4

/** VIRTUAL SERVOS PARAMETERS **/

//-- Phase increment for all the servos
//-- Seq number     0    1    2     3    4,   5
__code int INC[] = {0,   8,   8,    3,   3,   3};

//-- Duration of each oscillation mode (in periods)
__code int NT[]  = {0,   6,   6,    6,   6,   2};

//-- Parameters for VIRTUAL MODULE 1
__code int A1[]  = {0,  20,   20,   20,  20,  45};
__code int O1[]  = {0,  0,     0,    0,  0,   0};
__code int P1[]  = {0,  0,     0,    0,  0,   0};

//-- Parameters for VIRTUAL MODULE 2
__code int A2[]  =  {0,  0,    0,   40,  40,  45};
__code int O2[]  =  {0,  0,    30,  0,    0,  0};
__code int P12[]  = {0,  0,    0,   90,   0, 90};

//-- Parameters for VIRTUAL MODULE 3
__code int A3[]  =  {0,  20,   20,  20,   20, 45};
__code int O3[]  =  {0,  0,    0,    0,   0,  0};
__code int P13[]  = {0, -120, -120, 110,  180,0};

/*******************************/
/** VARIABLES IMPORTED         */
/** From the OCILLATOR module  */
/*******************************/
extern int oscillator_periods;
extern int oscillator_new_period;


//----------------------------
//- MAIN PROGRAM
//----------------------------

void main(void)
{
  //-- The variable m is used to store the current oscillation
  unsigned char m=0;

  //-- Get the number of oscillations configured by the user
  unsigned char TAM=sizeof(A1)/sizeof(int);

  //-- Oscillator module initialization
  oscillator_init();

  //-- Set the parameters of the virtual servos
  oscillator_set_params(0);

  //-- Button configuration
  //-- The RA0 bit is configured as a digital input
  ADCON1=0x06;
  TRISA0 = 1;


  //----------------------
  //-- Main loop
  //----------------------
  while (1) {

     //-- Calculate module positions from the Oscillators
     oscillator_pos();

     //-- Check if the test button has been pressed
    if (TEST_BUTTON==PRESSED) {

      //-- Change to the following oscillation mode
      m = (m+1) % TAM;

      //-- Set the new oscillators parameters
      oscillator_set_params(m);

      //-- Wait until the test button is released
      while(TEST_BUTTON==PRESSED);
    }

    //-- Check if time has elapsed. If so, change the oscillation mode
    if (oscillator_new_period==1 && oscillator_periods>=NT[m]) {

      //-- Reset the variables
      oscillator_new_period=0;
      oscillator_periods=0;

      //-- Change to the following oscillation mode
      m = (m+1) % TAM;

      //-- Set the new oscillators parameters
      oscillator_set_params(m);
    }
  }

}
