/*****************************************************************************/
/* sg-oscillator-pic16f876-skypic.c  Enero-2010                              */
/*---------------------------------------------------------------------------*/
/* Servidor que implementa micro-osciladores para los servos                 */
/* Servidor: sg-oscillator                                                   */
/* Implementado para la tarjeta Skypic/Skycube 1.0 a 20MHz                   */
/*---------------------------------------------------------------------------*/
/* Implementados los servicios de Identificacion, PING                       */
/*---------------------------------------------------------------------------*/
/*  Juan Gonzalez <juan@iearobotics.com>                                     */
/*  Andres Prieto-Moreno <andres@iearobotics.com>                            */
/*---------------------------------------------------------------------------*/
/*  LICENCIA GPL                                                             */
/*****************************************************************************/

//-- Especificar el pic a emplear
#include <pic16f877.h>
#include "servos.h"
#include "libsci.h"
#include "sin.h"
#include "stargate.h"


/******************************************/
/*  PARAMETROS DE LAS OSCILACIONES        */
/******************************************/

//-- Incremento de fase (en grados) (1-180)
#define INC 3

//-- Offset (en grados) (-90 y 90)
#define OFFSET 0

//--Fase inicial de los servos (en grados) (0-360)
#define FASE0 0

/**********************************************/
/* VARIABLES GLOBALES                         */
/**********************************************/
static int fase;
static int A[8]={45,45,45,45,45,45,45,45};
static int FASE[8]={0,45,90,135,0,45,90,135};
static char pos_fix[8]={0,0,0,0,0,0,0,0};

//-- Mascara con los osciladores activos
static unsigned char on_mask=0x00;

//-- Variables para las tramas recibidas
static unsigned char trama_id;
static unsigned char nodo;
static unsigned char mask;
static unsigned char servo;
static signed char pos2;

static unsigned char temp;

//-- Acceso a las posiciones de los servos
extern volatile unsigned char pos_servos[8];
extern unsigned char indice;
extern unsigned char tic;
extern unsigned char servo_actual; 

/*******************************************************/
/* RUTINA DE ATENCION A LAS INTERRUPCIONES             */
/*******************************************************/
void intr(void) interrupt 0
{
    
  //---------------------------------------------------------------
  //-- Rutina de servicio del timer0 para actualizar las senales
  //-- de PWM
  //-----------------------------------------------------------
  if (T0IF==1)
    servos_intr();

  //-----------------------------------------------------------------
  //-- Rutina de servicio del puerto serie para procesar las tramas
  //------------------------------------------------------------------
  //-- LIMITACION: Actualmente, una vez recibida la cabecera,
  //-- se procesa la trama completa. Como se hace mediante interrupciones
  //-- mientras se procesa la trama, no se actualiza el PWM por lo que los
  //-- servos sufren "temblores" cuando hay comunicacion con el PC. Esto es
  //-- algo que hay que solucionar
  //------------------------------------------------------------------------
  if (RCIF==1) {
    //-- Leer dato (para que Flag se ponga a 0)
    trama_id = RCREG;
    //PORTA = trama_id;  //- DEBUG!

    //-- Procesar las tramas
    switch(trama_id) {
        
      case SPING:  // servicio SPING
        sci_write(RPING);
        break;
        
      case SID: // Servicio de identificacion
        sci_write(RSI);
        sci_write(IS);
        sci_write(IM);
        sci_write(IPV);
        break;

      case SENA: // Servicio de habilitacion mediante mascara
        nodo=sci_read();
        mask=sci_read();
       //-- Nodo maestro
       //if (nodo==0)
       servos_enable_mask(mask);
       //-- Resto de nodos
       //else;
       //-- Invocar servicio servos_mask por I2C
       //-- Por hacer
        break;

      case SENA2: // Servicio de habilitacion de servo
        nodo=sci_read();
        servo=sci_read();
        //-- Nodo maestro
        //if (nodo==0)
        servos_enable(servo);
        //-- Resto de nodos
        //else;
        //-- Invocar servicio servos_enable por I2C
        //-- Por hacer
        break;

      case SDIS: // Servicio de deshabilitacion de servo
        nodo=sci_read();
        servo=sci_read();
        //-- Nodo maestro
        //if (nodo==0)
        servos_disable(servo);
        //-- Resto de nodos
        //else;
        //-- Invocar servicio servos_disable por I2C
        //-- Por hacer
        break;

      case SOSCM: //-- Servicio de establecimiento de mascara de oscilacion
        nodo=sci_read();
        mask=sci_read();
        //-- Nodo maestro
        //if (nodo==0)
        on_mask=mask;
        tic=1;
        PORTA=2;
        //-- Resto de nodos
        //else;
        //-- Invocar servicio servos_mask por I2C
        //-- Por hacer
        break;

      case SON: // Servicio de activacion de oscilador (ON)
        nodo=sci_read();
        servo=sci_read();
        //-- Nodo maestro
        //if (nodo==0)
        on_mask = on_mask | (1<<servo);
        //-- Resto de nodos
        //else;
        //-- Invocar servicio servos_disable por I2C
        //-- Por hacer
        break;

      case SOFF: // Servicio de desactivacion oscilador (OFF)
        //PORTA=0x07;
        nodo=sci_read();
        servo=sci_read();
        //-- Nodo maestro
        //if (nodo==0)
        //-- Mini-bug en sdcc 2.8.0: hay que usar esta variable
        //-- temporal, sino la operacion no se realiza bien
        temp=(1<<servo);
        on_mask = on_mask & ~temp;
        //-- Resto de nodos
        //else;
        //-- Invocar servicio servos_disable por I2C
        //-- Por hacer
        break;

       case SPOS:  // Servicio de posicionamiento del servo
          nodo=sci_read();
          servo = sci_read();

          //-- Leer posicion en grados (byte con signo)
          pos2=sci_read();

          //-- Almacenar posicion del servo
          pos_fix[servo]=pos2;

          RA1^=1;  //-- test
          break;
    }
  }
}

//----------------------------
//- Comienzo del programa  
//----------------------------

void main(void)
{
  int pos_sin;
  int pos_grados;
  unsigned char upos;
  unsigned char i;
  unsigned char s;
  
  //-- Inicializacion
  sci_conf();          //-- Puerto serie
  servos_init();       //-- Servos

  //-- Inicializar variables
  fase=0;
  //-- establecer el estado inicial de las mascaras
  servos_enable_mask(0xff);
  on_mask=0x00;

  //-- Configurar puerto A como puerto Digital
  //-- Se usa para depurar con la Freeleds conectrada al puerto A
  ADCON1=0x06;
  TRISA=0x00;
  PORTA=0x00;

  //-- Habilitar interrupciones del puerto serie
  RCIE=1;
  PEIE=1;

  //-- Interrupciones globales
  GIE=1;

  //----------------------
  //-- Bucle principal
  //----------------------
  while (1) {

      RA0=1;  //-- test
      
    //-- Calculos de los osciladores: obtener siguiente posicion para los sevos
    //-- Se ejecuta cada 2.5ms.
    //-- La variable tic se pone a 1 cada 2.5ms (la activa la rutina de atencion
    //-- a las interrupciones del timer0, que se encarga de generar el pwm)
    if (tic==1) {

      
      //-- Reiniciar tic. Al cabe de 2.5ms se volvera a activar
      tic=0;

      //-- Obtener el numero de servo actual (variable indice) y su
      //-- mascara binaria. Ambas se obtienen del modulo servos.c
      //-- Esta operacion es muy importante, porque indice y sevo_actual
      //-- son variables globales que se modifican asincronamente mediante
      //-- interrupciones, por lo que es necesario crear una copia local
      //-- en las variables i, s para usarlas en los calculos
      i=indice;
      s=servo_actual;

      
        
      //-- Calcular la siguiente posicion del servo i, para el modo oscilador
      pos_sin=sin(A[i],fase+FASE[i]+FASE0)+OFFSET;

      //-- Establecer la siguiente posicion del servo, Si el oscilador
      //-- esta activo (ON) se usa el seno, sino se usa la posicion fija
      //-- disponible en pos_fix[]
        
      if ((on_mask & s)!=0) {  //-- Modo ON
          
        //-- La posicion proviene del oscilador sinusoidal
        pos_grados = pos_sin;

        //-- Actualizar la posicion fija para que al hacer OFF el servo
        //-- se quede en la ultima posicion
        pos_fix[i]=pos_sin;
          
      } else {
        //-- Modo OFF
        pos_grados = pos_fix[i];
      }
      
      //-- Realizar la conversion entre grados y tiempo
      upos = servos_grados2tiempo(pos_grados);
  
      //-- Refrescar la posicion del servo
      pos_servos[i]=upos;

      if (i==0) {
        //-- Incrementar la fase
        fase=fase + INC;

        //-- Las fases negativas se convierten a positivas
        //-- BUG: para fases negativas deveria ser: fase=360-fase
        if (fase<0) fase=359;

        //-- Mantener la fase en el rango 0 - 360 grados
        fase = fase % 360;

      }
    }

    //-- Realizar aqui otras tareas


  } //-- While (1)
}
