/*****************************************************************************/
/* oscillator-pyn.c  Enero-2010                                              */
/*---------------------------------------------------------------------------*/
/* Movimiento de configuraciones PYn                                         */
/*---------------------------------------------------------------------------*/
/*  Juan Gonzalez <juan@iearobotics.com>                                     */
/*---------------------------------------------------------------------------*/
/*  LICENCIA GPL                                                             */
/*****************************************************************************/

#include "servos.h"
#include "sin.h"

/*********************************/
/* VARIABLES DEL MODULO          */
/*********************************/
//-- Amplitudes de las oscilaciones
static int Am[8]={0,0,0,0,0,0,0,0};

//-- Fases de los osciladores
static int FASE[8]={0,0,0,0,0,0,0,0};

static char OFFSET[8]={0,0,0,0,0,0,0,0};

//-- Variable de fase
static int fase;

//-- Acceso a las variables del modulo SERVOS
extern volatile unsigned char pos_servos[8];
extern unsigned char indice;
extern unsigned char tic;
extern unsigned char servo_actual; 

//-- Acceso a variable de servos virtuales
extern __code unsigned char v[9];

/**************************************************************************/
/* Inicializar los osciladores. Se recorren todos los servos virtuales    */
/* asignando los valores de amplitud y fase correspondientes              */
/* Dada un servo virtual k, su oscilador correspondientes esta dado por : */
/*                                                                        */
/*  Angulo doblaje servo k = A*sin(fase + (k-1)*DF + FASE0)               */
/*                                                                        */
/**************************************************************************/
void oscillator_pyn_init(int Av, int Ah, int dfv, int dfh, int dfvh, 
                         int fase0, char oh)
{
  unsigned char j;
  unsigned char i;
  unsigned char k;
  unsigned char mask=0x00;

  //-- Recorrer todos los servos virtuales
  for (i=1; i<=8; i++) {

    //-- Obtener servo real j, a partir del virtual
    j=v[i];

    //-- Si ese servo se usa, establecer los parametros de su oscilador
    //-- y activar el servo
    if (j!=0) {

     //-- Calculos para los servos virtuales IMPARES: PITCH!
     if ((i&0x01) == 1) {  

       //-- k toma los valores 0,1,2... de los modulos verticales
       k=(i-1)>>1;
         
       //-- Establecer amplitud
       Am[j-1]=Av; 

       //-- Establecer fase
       FASE[j-1]=k*dfv + fase0;
     }
     //-- Calculos para los servos virtuales PARES: YAW!
     else {
       //-- k toma los valores 0,1,2... de los modulos horizontales
       k=(i>>1)-1;

       //-- Establecer amplitud
       Am[j-1]=Ah;

       //-- Establecer fase
       FASE[j-1]=k*dfh + dfvh + fase0;

       //-- Establecer offset
       OFFSET[j-1]=oh;
     }
        
     //-- Establecer mascara de habilitacion
     mask=mask | (1<<(j-1));
    }
  }

  //-- Activar los servos usados
  servos_enable_mask(mask);

  //-- Inicializar variables
  fase=0;
}


void oscillator_calculate(int inc)
{
  int pos_sin;
  unsigned char upos;
  unsigned char i;
  unsigned char s;
    
    //-- Calculos de los osciladores: obtener siguiente posicion para los sevos
    //-- Se ejecuta cada 2.5ms.
    //-- La variable tic se pone a 1 cada 2.5ms (la activa la rutina de atencion
    //-- a las interrupciones del timer0, que se encarga de generar el pwm)
    if (tic==1) {

      //-- Reiniciar tic. Al cabe de 2.5ms se volvera a activar
      tic=0;

      //-- Obtener el numero de servo actual (variable indice) y su
      //-- mascara binaria. Ambas se obtienen del modulo servos.c
      //-- Esta operacion es muy importante, porque indice y sevo_actual
      //-- son variables globales que se modifican asincronamente mediante
      //-- interrupciones, por lo que es necesario crear una copia local
      //-- en las variables i, s para usarlas en los calculos
      i=indice;
      s=servo_actual;

      //-- Calcular la siguiente posicion del servo i
      pos_sin=sin(Am[i],fase+FASE[i])+OFFSET[i];

      //-- Realizar la conversion entre grados y tiempo
      upos = servos_grados2tiempo(pos_sin);
  
      //-- Refrescar la posicion del servo
      pos_servos[i]=upos;

      //-- Cuando se llega al servo 0, se incrementa la fase
      if (i==0) {
        //-- Incrementar la fase
        fase=fase + inc;

        //-- Las fases negativas se convierten a positivas
        //-- BUG: para fases negativas deveria ser: fase=360-fase
        if (fase<0) fase=359;

        //-- Mantener la fase en el rango 0 - 360 grados
        fase = fase % 360;
      }
    } 
}
