//-- (c) Juan Gonzalez-Gomez (obijuan)  juan@iearobotics.com
//-- May-26th-2010.
//-- GPL license
//-- Oscillation of one servo

#define F_CPU 16000000

#include <avr/interrupt.h>
#include <avr/io.h>
#include <util/delay.h>
#include <math.h>


//-- Definitions
#define DELAY_0_3ms 19
#define DELAY_0_2ms 12
#define DELAY_1ms   62
#define DELAY_2ms   125
#define DELAY_2_5ms 156

#define CENTRO DELAY_1ms

//--- Global variables----
//-- States for generating the PWM
//-- State0: Fixed width of 0.3ms. Pwm signal to 1
//-- State1: Servo position. Pulse of width pos. Width between 0 and 2ms (pwm signal to 1)
//-- State2: Servo position complementary. Width = 2ms - State1 width (pwm signal to 0)
//-- State3: Fixed width of 0.2ms (Pwm signal to 0)
enum { STATE0, STATE1, STATE2, STATE3 } state;
unsigned char servo=0;  //-- Servo index: 0 to 7

//-- Servo position in ticks
volatile unsigned char pos[]={CENTRO,CENTRO,CENTRO,CENTRO,CENTRO,CENTRO,CENTRO,CENTRO};

//-- Signaling that a new Pwm cicle (20ms) start
volatile unsigned char pwm_cycle_start=0;

//-- Interrupt service routine
//-- It is executed everytime the comparator A is trigged
//-- It is in charge of generating the PWM signal for seting the servo position
ISR(TIMER0_COMPA_vect)
{
    //-- The interrupt flag is automatically cleared

    //-- Finite state machine implementation
    switch(state) {

        case STATE0:  //-- PWM signal to 1. Fixed witdh of 0.3ms
            PORTB=(1<<servo);
            OCR0A=DELAY_0_3ms;

            state = STATE1;  //-- Next state
            break;

        case STATE1:  //-- PWM signal to 1. Variable width of pos[servo] ticks
            OCR0A=pos[servo];

            state = STATE2;  //-- Next state
            break;

        case STATE2: //-- PWM signal to 0. Variable width of DELAY_2ms - pos[servo] ticks
            PORTB=0;
            OCR0A=DELAY_2ms - pos[servo];

            state = STATE3;  //-- Next state
            break;

        case STATE3: //-- PWM signal to 0. fixed witdh of 0.2ms
            OCR0A=DELAY_0_2ms;

            servo = (servo+1)%8;
            state = STATE0;  //-- Next state

             //-- A new pwm cicle starts!
            if (servo==0)
              pwm_cycle_start=1;
            break;
    }

}

//-- Conversion between degrees and ticks
//-- The PWM signal to set the servo position is calculated in ticks
//-- deg: angle in degrees in the range [-90,90]
//-- It returns the Pwm signal width in ticks (Range between 0 and 125)
unsigned char deg2time(char deg)
{
    return 25*(90-deg)/36;
}

void servo_pos(unsigned char s, char deg)
{
    pos[s]=deg2time(deg);
}

int main(void)
{

  //-- Configure the PORTB as output
  DDRB= 0xFF;

  //Clear Timer on Compare or CTC mode (WGM02:0 = 2)
  TCCR0A = _BV(WGM01);

  //-- Configurar timer0
  //-- Prescaler 256.  CS02:0=4. Reloj = 16Mhz (T=62.5ns)
  //-- Ticks con prescaler: 16us
  TCCR0B = _BV(CS02);

  //-- Dehay of 0.3ms --> OCR0A = 19 (18.750)
  //-- Delay of 0.2ms --> OCR0A = 12 (12.5)
  //-- delay of 1ms --> OCR0A = 62 (62.5)
  //-- delay of 2ms --> OCR0A = 125
  //-- Position range: 0 - 125 timer units --> [-90, 90] degrees

  //-- Enable Compare A (timer0) interupt
  TIMSK0= _BV(OCIE0A);


  //-- Enable interrupts
  sei();

  #define TS 20
  #define DEG2RAD(g) g*M_PI/180

  double ang;
  unsigned int n;
  unsigned int N;
  int T=1500;
  int A[8]={45,45,45,45,30,45,45,45};
  int offset[8]={0,0,0,0,0,0,0,0};
  double phase0[8] = {0,DEG2RAD(120),0,0,0,0,0,0,};
  double phase=0;

  N = T/TS;

  double inc = 2*M_PI/N;
  int i=5;


  for (;;) {

     for (n=0; n<N; n++) {
       //-- Calculate the samples for the 8 oscillators
       //-- It takes 6ms aproximatelly
       for (i=0; i<8; i++) {
         ang = round(A[i] * sin(phase + phase0[i]) + offset[i]);
         servo_pos(i,ang);
       }

       //-- Increment the phase
       phase = phase + inc;

       //-- Wait for a new PWM cycle
       while(pwm_cycle_start==0);
       pwm_cycle_start=0;
     }
  }

  return 0;
}

