#! /usr/bin/python
# -*- coding: iso-8859-15 -*-
#---------------------------------------------------------------------------
#--  Modulo Stargate para el movimiento de hasta 8 servos
#--
#--  LICENCIA GPL
#---------------------------------------------------------------------------

# Description: libStargate is a library for comunication to stargates
# Copyright: (C) 2007 by Juan González Gómez
#            (C) 2007 by Rafael Treviño Menéndez
# Authors: Juan González Gómez     <juan@iearobotics.com>
#          Rafael Treviño Menéndez <skasi.7@gmail.com>

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#------------------------------------------------------------------------------


import libStargate.Main


#--- Cadena identificativa del servidor
OSCILLATOR_IDENT = "OSCILLATOR"

#-----------------------------------
#------ Cabeceras de las tramas 
#----------------------------------- 
TRAMA_ENA_CAB    = 'E'    #-- Servicio Enable con mascara
TRAMA_ENA2_CAB   = 'F'    #-- Servicio Enable para servo
TRAMA_DIS_CAB   =  'D'    #-- Servicio disable para servo
TRAMA_OSC_MASK_CAB = 'M'  #-- Servicio de establecimiento de mascara oscilacion
TRAMA_ON_CAB = 'O'        #-- Servicio de activacion de oscilador (ON)
TRAMA_OFF_CAB = 'S'       #-- Servicio de desactivacion del oscilador (OFF)

TRAMA_POS_CAB    = 'W'    #-- Servicio de Posicionamiento

#-----------------------------------------------
#- Abrir la conexion con el servidor 
#-----------------------------------------------
def Open_session(serialName,logCallback=libStargate.Main.default_log,baud=57600):
  
  #-- Abrir puerto serie 
  try:
    s = libStargate.Oscillator.Open (serialName,logCallback,baud)
  except:
    logCallback("Error al abrir puerto serie")
    return None
    
  #---------------------------------------------
  #-- Comprobar la conexion con el servidor   
  #---------------------------------------------
  
  #--- Comprobar si hay conexion
  ok=s.check_connection();
  if (not ok):
    return None
  
  #-- Comprobar si el servidor al que se accede es el correcto
  ok=s.check_server_type();
  if (not ok):
    return None
    
  return s


#----------------------------------------------------------------------------
#-              Clase para la gestion de los errores
#----------------------------------------------------------------------------
class Error (Exception):
  """
  Excepciones producidas en el modulo Stargate
  """
  pass
  
#---------------------------------------------------------------------
#  CLASE STARGATE SERVOS8
#---------------------------------------------------------------------
class Open(libStargate.Main.Stargate):

  #-- Inicializar la clase
  def __init__(self, port, logCallback=libStargate.Main.default_log,baud=9600):
    libStargate.Main.Stargate.__init__(self,port,logCallback,baud)


  #----------------------------------------------------
  #-- Obtener si el servidor es el correcto (OSCILLATOR)
  #----------------------------------------------------
  def check_server_type(self):
  
    #-- servidor debe ser del tipo correcto
    ok=libStargate.Main.Stargate.check_server_type(self,OSCILLATOR_IDENT);
    
    if ok:
      self.log("Servidor correcto");
      self.log("\n");
    else:
      self.log("Servidor esperado: %s. \n" % OSCILLATOR_IDENT);
      self.log("---> SERVIDOR INCORRECTO\n");
      return False;

    return True

  #-------------------------------
  #-- Servicio ENABLE con mascara
  #-------------------------------
  def enable_mask(self, mask, nodo=0):
  
    #-- Si puerto serie no abierto retornar
    if self.serial==None: return 0
    
    #-- Enviar trama 
    self.serial.write(TRAMA_ENA_CAB)
    self.serial.write(chr(nodo&0xFF))
    self.serial.write(chr(mask&0xFF))
    
    return 1


  #-------------------------------
  #-- Servicio ENABLE para servo
  #-------------------------------
  def enable(self, servo, nodo=0):
  
    #-- Si puerto serie no abierto retornar
    if self.serial==None: return 0

    servo-=1

    #-- Enviar trama 
    self.serial.write(TRAMA_ENA2_CAB)
    self.serial.write(chr(nodo&0xFF))
    self.serial.write(chr(servo&0xFF))
    
    return 1

  #-------------------------------
  #-- Servicio DISABLE para servo
  #-------------------------------
  def disable(self, servo, nodo=0):
  
    #-- Si puerto serie no abierto retornar
    if self.serial==None: return 0

    servo-=1

    #-- Enviar trama 
    self.serial.write(TRAMA_DIS_CAB)
    self.serial.write(chr(nodo&0xFF))
    self.serial.write(chr(servo&0xFF))
    
    return 1

  #-------------------------------
  #-- Servicio ON con mascara
  #-- Activar la oscilacion de los servos indicados
  #-------------------------------
  def on_mask(self, mask, nodo=0):
  
    #-- Si puerto serie no abierto retornar
    if self.serial==None: return 0
    
    #-- Enviar trama 
    self.serial.write(TRAMA_OSC_MASK_CAB)
    self.serial.write(chr(nodo&0xFF))
    self.serial.write(chr(mask&0xFF))
    
    return 1

  #---------------------------------------
  #-- Servicio ON: activar un oscilador
  #---------------------------------------
  def on(self, servo, nodo=0):
  
    #-- Si puerto serie no abierto retornar
    if self.serial==None: return 0

    servo-=1

    #-- Enviar trama 
    self.serial.write(TRAMA_ON_CAB)
    self.serial.write(chr(nodo&0xFF))
    self.serial.write(chr(servo&0xFF))
    
    return 1

  #---------------------------------------
  #-- Servicio OFF: Parar un oscilador
  #---------------------------------------
  def off(self, servo, nodo=0):
  
    #-- Si puerto serie no abierto retornar
    if self.serial==None: return 0

    servo-=1

    #-- Enviar trama 
    self.serial.write(TRAMA_OFF_CAB)
    self.serial.write(chr(nodo&0xFF))
    self.serial.write(chr(servo&0xFF))
    
    return 1

  #------------------------------------------------------------------
  #-- Servicio POS
  #-- Se especifica el angulo del servo en grados, entre -90 y 90
  #------------------------------------------------------------------
  def pos(self, servo, pos, nodo=0):
  
    #-- Si puerto serie no abierto retornar
    if self.serial==None: return 0
 
    servo-=1
    pos=int(pos);

    #-- Enviar trama 
    self.serial.write(TRAMA_POS_CAB)
    self.serial.write(chr(nodo&0xFF))
    self.serial.write(chr(servo&0xFF))
    self.serial.write(chr(pos&0xFF))

    return 1

  #---------------------------------------
  #- Metodo GET. Equivalente a un POS
  #---------------------------------------
  def __setitem__(self, servo, pos): 
    return self.pos(servo,pos)  
