#! /usr/bin/python
# -*- coding: iso-8859-15 -*-
#---------------------------------------------------------------------------
#--  Modulo Stargate para la comunicacion con lo servidores "stargates"
#--  Este modulo implementa los servicios basicos y funciones comunes al
#--  resto de servidores
#--
#--  LICENCIA GPL
#---------------------------------------------------------------------------

# Description: libStargate is a library for comunication to stargates
# Copyright: (C) 2007 by Juan González Gómez
#            (C) 2007 by Rafael Treviño Menéndez
# Authors: Juan González Gómez     <juan@iearobotics.com>
#          Rafael Treviño Menéndez <skasi.7@gmail.com>

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.



import serial;
import time;
import sys;


#-----------------------------------
#------ Cabeceras de las tramas 
#-----------------------------------
TRAMA_PING_CAB   = 'P'   #-- PING
TRAMA_PONG_CAB   = 'O'   #-- PONG
TRAMA_ID_CAB     = 'I'   #-- IDENTIFICACION
TRAMA_RID_CAB    = 'I'   #-- Respuesta IDENTIFICACION


#-----------------------------------------
#---- Identificacion de los servidores
#-----------------------------------------
SERVIDOR = {0x00:"XXXX",
            0x10:"NULL", 
            0x20:"GENERIC",
            0x30:"SERVOS8",
            0x40:"PICP",
            0x50:"OSCILLATOR",
           }

#------------------------------------------------
#--- Identificacion de los microcontroladores
#------------------------------------------------
MICRO = {0x00: "XXXX",
         0x10: "68HC11E2",
         0x20: "68HC08",
         0x30: "PIC16F876",
        }

#--------------------------------------
#-- Identificacion de las placas
#--------------------------------------
PLACA = {0x00 : "USER",
         0x101: "CT6811",
         0x201: "GPBOT",
         0x300: "USER",
         0X301: "SKYPIC",
        } 


#-------------------------------------------------------------
#-- Funcion de Log por defecto. Se escribe en la consola
#-------------------------------------------------------------
def default_log(msg):
  sys.stdout.write(msg)
  sys.stdout.flush()

#--------------------------
#-- Funcion de log "nula"
#--------------------------
def null_log(msg):
  None
 
#----------------------------------------------------------------------------
#-              Clase para la gestion de los errores
#----------------------------------------------------------------------------
class Error (Exception):
  """
  Excepciones producidas en el modulo Stargate
  """
  pass


#-----------------------------------------------------------------------------
#                          CLASE STARGATE
#-----------------------------------------------------------------------------
class Stargate:
  def __init__(self, serialName, logCallback=default_log,baud=9600):
   
    print "hola!"

    #-- Abrir puerto serie
    self.Puerto = serialName
    
    try:
      self.serial = serial.Serial(serialName, baud)
      
    except serial.SerialException:
      self.serial=None
      raise Error,'Error al abrir puerto serie %s.' % serialName
    
    #-- Establecer funcion de retrollamada del log
    if logCallback==None:
      self.log=null_log;
    else:
      self.log=logCallback

    #-- Timeout: 1 seg
    self.serial.timeout=0.4;
      
    #-- Vaciar los buffers de entrada y salida
    self.serial.flushInput()
    self.serial.flushOutput()
    
    #-- Activar el DTR para no resetear la Skypic
    self.serial.setDTR(1)
    time.sleep (0.3)
    
  #---------------------
  #- Destructor 
  #---------------------
  def __del__(self):
  
    #-- Cerrar el pueto serie
    if self.serial:
      #self.log("Debug: cerrando puerto serie: %s\n" % (self.serial.portstr))
      self.serial.setDTR(0)
      self.serial.close()

  #-----------------------
  #-- Cerrar el Stargate
  #-----------------------
  def close(self):
    if self.serial:
      self.serial.setDTR(0)
      self.serial.close()
      self.serial=None
  
  #-------------------------------------
  #-- Hacer un reset de la Skypic
  #-------------------------------------
  def Reset(self):
    #-- Desactivar la senal DTR durante 0.5 segundos
    self.serial.setDTR (0)
    time.sleep (0.4)
    
    #-- Volver a activarla. Reset hecho
    self.serial.setDTR (1)
    time.sleep(0.4)
  
  #------------------
  #- Servicio PING 
  #------------------
  def ping(self):
  
    #-- Si puerto serie no abierto retornar
    if self.serial==None: return 0
    
    #-- Enviar trama PING
    self.serial.write(TRAMA_PING_CAB);
    
    #-- Esperar respuesta
    pong = self.serial.read();
    
    #-- Timeout o dato incorrecto
    if len(pong)==0 or pong!=TRAMA_PONG_CAB: 
    
      #-- Vaciar los buffers de entrada y salida
      self.serial.flushInput()
      self.serial.flushOutput()
      return 0
    
    #-- Ping ok
    return 1
    
  #-----------------------------------------------------
  #- Servicio de identificacion. Devolver los bytes
  #- Devuelve la tupla (is, im, id):
  # -    is : identificador de servicio
  #-     im : Identificador de micro
  #-     id : Identificador de placa y version
  #-  Todos son valores enteros
  #-----------------------------------------------------
  def id_bytes(self):
  
    #-- Si puerto serie no abierto retornar
    if self.serial==None: return (0,0,0)
    
    #-- Enviar trama de IDENTIFICACION
    self.serial.write(TRAMA_ID_CAB)
    
    #-- Esperar respuesta
    id = self.serial.read(4)
    
    #-- Timeout o trama incorrecta
    if len(id)!=4 or id[0]!=TRAMA_RID_CAB:
    
      #-- Vaciar los buffers de entrada y salida
      self.serial.flushInput()
      self.serial.flushOutput()
      return (0,0,0)
    
    #-- Trama correcta, devolver valores
    return (ord(id[1]), ord(id[2]), ord(id[3]))
    
  #-----------------------------------------------------
  #- Servicio de identificacion. Devolver la cadena
  #-----------------------------------------------------
  def id(self):
    #-- Leer la identificcion
    (iserv, im, id) = self.id_bytes()
    
    #-- Si hay error sacar la cadena "UNKNOW"
    if (iserv==0 and im==0 and id==0): return "UNKNOW"
    
    #-- devolver la cadena
    return "SG-%s-%s-%s-%d" % (SERVIDOR[iserv],
                               MICRO[im],
                               PLACA[im<<4 | id>>4],
                               id&0x0F)
                               
  #---------------------------------------------------------------------------
  #-- Comprobar la conexion con el servidor. 
  #-- Esta funcion realiza tres pings. Si el servidor no responde a la tercera,
  #-- devolvera un 0, indicando que no hay conexiond
  #-- En caso de que haya respuesta, se imprime el tipo de servidor y se
  #-- devuelve 1
  #
  #  DEVUELVE:
  #     False : No hay conexion con el servidor
  #     True  : Conexion establecida
  #
  #----------------------------------------------------------------------------
  def check_connection(self):
    
    #-- Mostrar el dispositivo serie empleado
    serial_disp = self.serial.portstr;
    self.log("\n");
    self.log("Dispositivo serie: %s\n" % serial_disp);
    
    #-- Realizar como maximo tres iteraciones comprobando si hay ping
    for i in range(3):
    
      self.log("Conexion...");
        
      #-- Hacer ping  
      if (self.ping()):
        self.log("OK\n");
        connection=True;
        break;
      else:
        self.log("TIMEOUT\n");
        connection=False;
  
    #-- Si hay conexion obtener la identificacion
    if connection:
      
      self.log("Servidor: %s\n" % self.id());
      return True
      
    else:
      self.log("Conexion NO establecida\n");#-- No hay conexion
      return False

  #-------------------------------------------------------------------
  #- Comprobar el tipo de servidor para saber si es el 
  #- esperado o no
  #- ENTRADAS:
  #-    -tipo: Cadena identificaiva del servidor. Definidas en el 
  #            diccionario SERVIDOR
  #  DEVUELVE:
  #     -True : Es el servidor esperado
  #     -False: Otro servidor diferente
  #------------------------------------------------------------------
  def check_server_type(self,tipo):
  
    tipo=tipo.upper();
      
    #-- Solicitar servico de identificacion  
    (iserv,im,ip) = self.id_bytes();
    
    #-- Servidor localizadao
    server_type = SERVIDOR[iserv];
    
    
    if (tipo==server_type):
      return True
    else:
      return False
  
  #-------------------------------------------
  #-- Vaciar los buffers de entrada y salida
  #-------------------------------------------
  def flush(self):
  
    #-- Vaciar los buffers de entrada y salida
    self.serial.flushInput()
    self.serial.flushOutput()
