//-----------------------------------------------------------------------------
//--  Tensile specimen. ISO 527 A.   Specimens 5A/5B
//--  Probeta para ensayos de tracción. Segun ISO 527 A. Probetas 5A/5B
//-----------------------------------------------------------------------------
//-- (c) Juan Gonzalez-Gomez (obijuan).  Jan-2013
//-- License: GPL
//------------------------------------------------------------------------------
//-- See the document specimen-schematic.dxf for the meaning of the parameters
//------------------------------------------------------------------------------
use <cccurve.scad>

//-- Constants
X = 0;
Y = 1;
Z = 2;

module specimen_5A()
{
  specimen();
}

module specimen_5B()
{

  specimen(
    l2 = 35,
    b2 = 6,
    l1 = 12,
    b1 = 2,
    r1 = 3,
    r2 = 3,
    L = 20.95,
    h = 1
    ); 

}


//-- Parametric Specimen  /  Probeta parametrica
//-- Default parameters correspond to ISO 527 A  (specimen 5A)
module specimen(
  l2 = 75,    //--  Overall length (minimum) / Longitud total, minima
  b2 = 12.5,  //--  Width of grip section / Anchura de los extremos
  l1 = 25,    //--  Narrow zone length / Longitud de la zona estrecha
  b1 =  4,    //--  Narrow zone width / Anchura de la zona estrecha
  r1 = 8,     //--  Minor radius / Radio menor
  r2 = 12.5,  //--  Mayor raidus / Radio mayor
  L  = 50,    //--  Distance between shoulders / Distancia entre mordazas
  h  = 2,     //--  Thickness / Espesor
  )
{
  
  //-- Specimen, only the 1st cuadrant (because of its simetry)
  module specimen_q1()
  {
  
    //-- Grip section size
    hsize = [l2/2 - L/2, b2/2, h];
    
    //-- Narrow zone size
    msize = [l2/2, b1/2, h];

    
    //-- Build the Grip
    translate([l2/2, 0, 0])
    translate([-hsize[X]/2, b2/4, 0])
    cube(hsize,center=true);

    //-- Build the narrow zone
    translate([msize[X]/2, msize[Y]/2,0])
    cube(msize, center=true);

    //-- Build the Transition between the Grip and narrow zones
    translate([l1/2, 0, 0])
      cc_curve(r1 = r1, r2 = r2, h1 = b1/2, dx = L/2 - l1/2, th = h);
  }
  
  //-- Right part of the specimen
  module specimen_right()
  {
    union() {
      //-- right part
      mirror([0, 1, 0])
        translate([0, -0.01, 0])
          specimen_q1();

      specimen_q1();
    } 
  }
  
  union() {
  
    //-- Right part
    specimen_right();  
    
    //-- Left part
    mirror([1, 0, 0])
      specimen_right();
  }  
}


//--  Test examples

//specimen_5B(h=1.4);

translate([0, 20, 0])
  specimen_5A();


