//------------------------------------------------------------------------------
//  Bevel edge cube.
//  (c) Juan Gonzalez-Gomez (Obijuan), Sep-2012
//------------------------------------------------------------------------------

//-------------- IMPLEMENTATION USING THE HULL OPERATOR ----------------------

//------------------------------------------------------------------------------
//-- Bevel Cube main function
//-- Parameters:
//--   * Size:  Cube size
//--   * cr : Corner radius (if cr==0, a standar cube is built)
//--   * cres:  Corner resolution (in points). cres=0 means flat corners
//------------------------------------------------------------------------------
//-- NOTE!
//--  I have detected a "Bug" on OpenSCAD 2012.02.22. The cube
//-- is done with the hull() parameter.. When 2 cylinder of adjacent
//-- corners are tangents, the hull() parameter just crash
//--
//--  For avoiding that, I have create the bcube() function, that is a 
//--  Safety module that check for this special condition before 
//--  creating the cube.  If it is met then the corner radius is increased
//--  by 0.001, avoiding the hull parameter to crash
//--
//--  Once this bug is fixed (in future releases of openscad) the
//--  bcube_raw can be renamed to bcube and the safety bcube module
//--  removed
//------------------------------------------------------------------------------
module bcube_raw(size,cr=0,cres=0)
{
  //-- Signs for the elements on the 4 cartesian-cuadrants, starting from 0
  //-- s[i][0] is the sign of the x-coordinate cuadrant i
  //-- s[i][1] is the sign of the y-coordinate in cuadrant i
  //--  Cuadrant:       0        1         2        3
  s =                  [ [1,1], [-1,1], [-1,-1], [1,-1] ];

  //-- Get the (x,y) coorner coordinates in the 1st cuadrant
  x = size[0]/2;
  y = size[1]/2;

  
  //-- A corner radius of 0 means a standar cube!
  if (cr==0)
    cube(size,center=true);
  else {

       //-- Put the cylinders in the corners and create the cube
       //-- using the hull operator
 	 hull()
	    for (i=[0:3]) {
	      translate([s[i][0]*(x-cr),s[i][1]*(y-cr), 0])
	          cylinder(r=cr, h=size[2], center=true, $fn=4*(cres+1));
	    }
  }

}

//------------------------------------------------------------------------------
//-- Safectly bcube module
//-- It just checks for the condition that makes the hull() 
//-- operator to crash
//------------------------------------------------------------------------------
module bcube(size,cr=0,cres=0)
{
  //-- Get the (x,y) coorner coordinates in the 1st cuadrant
  x = size[0]/2;
  y = size[1]/2;

  if (y==2*cr || x==2*cr) {
    echo("Warning",cr+0.01);
    bcube_raw(size,cr+0.01,cres);
  }
  else {
    bcube_raw(size,cr,cres);
  }
}

//-- Examples of use of the bcube() module

//-- Standar cube
translate([-15,15,0])
bcube([20,20,10]);

//-- Beveled cube (0 point resolution)
translate([15,15,0])
  bcube([20,20,10],cr=4);

//-- Beveled cube (1 point resolution)
translate([-15,-15,0])
  bcube([20,20,10],cr=4, cres=1);

//-- Beveled cube (4 points resolution)
translate([15,-15,0])
  bcube([20,20,10],cr=4, cres=4);


