package robot.gui;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;

import javax.swing.GrayFilter;
import javax.swing.JComponent;

import robot.EstadoRobot;
import utils.ImageUtils;

/**
 * Clase para monitorizar gráficamente el robot.
 */
@SuppressWarnings( "serial" )
final class Monitor extends JComponent{
	
	/**
	 * Clase que almacena las imágenes empleadas.
	 */
	private static class Images{
		
		final Image fondo;
		
		final Image bump_on;
		final Image bump_off;
		
		final Image led_on;
		final Image led_off;
		
		final Image casa;
		final Image automatico;
		final Image manual;
		
		final Image emergencia;
		
		final Image flecha;
		final Image atras;
		final Image giro;
		final Image paro;
		
		final Image cargado;
		final Image descargado;
		
		Images(){
			fondo         = ImageUtils.getImage( "resources/robot/chasis.png");
			
			bump_on       = ImageUtils.getImage( "resources/robot/bumper-on.png");
			bump_off      = ImageUtils.getImage( "resources/robot/bumper-off.png");
			
			led_on        = ImageUtils.getImage( "resources/robot/led-on.png");
			led_off       = ImageUtils.getImage( "resources/robot/led-off.png");
			
			casa          = ImageUtils.getImage( "resources/iconos/casa.png");
			automatico    = ImageUtils.getImage( "resources/iconos/auto.png");
			manual        = ImageUtils.getImage( "resources/iconos/manual.png");
			
			emergencia    = ImageUtils.getImage( "resources/iconos/emergencia.png");
			
			flecha        = ImageUtils.getImage( "resources/iconos/adelante.png");
			atras         = ImageUtils.getImage( "resources/iconos/atras.png");
			giro          = ImageUtils.getImage( "resources/iconos/giro.png");
			paro          = ImageUtils.getImage( "resources/iconos/paro.png");
			
			cargado       = ImageUtils.getImage( "resources/iconos/cargado.png");
			descargado    = ImageUtils.getImage( "resources/iconos/descargado.png");
		}
	}
	
	private static final Images IMGS           = new Images();
	private static final Color  DISPLAY_COLOR  = new Color( 0, 32, 0, 192 );
	private static final Font   FONT           = new Font( "Monospaced", Font.BOLD, 100 );

	private final Image disabledImage;
	private EstadoRobot estado;
	
	/**
	 * Constructor que crea el Monitor.
	 */
	Monitor(){
		
		estado = new EstadoRobot( (byte)0x00, (byte)0x00, (byte)0x00 );
		
		BufferedImage bi = new BufferedImage(512,512, BufferedImage.TYPE_INT_ARGB );
		paintRobot( (Graphics2D)bi.getGraphics(), estado, 512, 0, 0 );
		disabledImage = GrayFilter.createDisabledImage( bi );
		
		this.setMinimumSize( new Dimension( 256, 256 ) );
		//FIXME
		this.setPreferredSize( new Dimension( 384, 384 )  );
		this.setEnabled( false );
	}
	
	/**
	 * Método que representa mediante circulos de colores, los datos leidos por un sensor.
	 * 
	 * Nota: Las coordenadas y dimensiones absolutas se toman en un espacio de 512 x 512,
	 * que son las dimensiones de la imagen de fondo. Por tanto multiplicando por el side,
	 * dividiendo por 512 ( igual a desplazar 9 bits ), y sumando el offset correspondiente,
	 * se obtienen las nuevas coordenadas ajustadas a las dimensiones reales del componente.
	 * 
	 * @param g2 Contexto gráfico empleado para dibujar el componente.
	 * @param sensor Elemento a representar.
	 * @param x Posición X en coordenadas absolutas.
	 * @param y Posición Y en coordenadas absolutas.
	 * @param r Radio en valores absolutos.
	 * @param side Mínimo entre la anchura y altura del componente en pixels.
	 * @param xOff Despazamiento de la coordenada X en pixels.
	 * @param yOff Despazamiento de la coordenada Y en pixels.
	 */
	private void paintSensor( Graphics2D g2, EstadoRobot.SensorIR sensor, int x, int y, int r, int side, int xOff, int yOff ){
		Color fillColor, drawColor;
		if( sensor == EstadoRobot.SensorIR.BLANCO ){
			fillColor = Color.WHITE;
			drawColor = Color.DARK_GRAY;
		}else{
			fillColor = Color.BLACK;
			drawColor = Color.LIGHT_GRAY;
		}
		
		x = ( x * side >> 9 ) + xOff; 
		y = ( y * side >> 9 ) + yOff;
		r = r * side >> 9;
		
		g2.setPaint( fillColor );
		g2.fillOval( x, y, r, r );
		g2.setPaint( drawColor );
		g2.drawOval( x, y, r, r );
	}
	
	/**
	 * Método muestra mediante imágenes un bumper, empleando la imagen apropiada dependiendo del estado.
	 * 
	 * Nota: Las coordenadas y dimensiones absolutas se toman en un espacio de 512 x 512,
	 * que son las dimensiones de la imagen de fondo. Por tanto multiplicando por el side,
	 * dividiendo por 512 ( igual a desplazar 9 bits ), y sumando el offset correspondiente,
	 * se obtienen las nuevas coordenadas ajustadas a las dimensiones reales del componente.
	 * 
	 * @param g2 Contexto gráfico empleado para dibujar el componente.
	 * @param bumper Elemento a representar.
	 * @param x Posición X en coordenadas absolutas.
	 * @param y Posición Y en coordenadas absolutas.
	 * @param w Anchura en valores absolutos.
	 * @param h Altura en valores absolutos.
	 * @param side Mínimo entre la anchura y altura del componente en pixels.
	 * @param xOff Despazamiento de la coordenada X en pixels.
	 * @param yOff Despazamiento de la coordenada Y en pixels.
	 */
	private void paintBumper( Graphics2D g2, EstadoRobot.Bumper bumper, int x, int y, int w, int h, int side, int xOff, int yOff ){
		g2.drawImage(
				bumper == EstadoRobot.Bumper.ESTIRADO ? IMGS.bump_off: IMGS.bump_on,
				( x * side >> 9 ) + xOff, ( y * side >> 9 ) + yOff, w * side >> 9, h * side >> 9,
				this
		);
	}
	
	/**
	 * Método dibuja las flechas que respresenta el sentido de giro de una rueda.
	 * 
	 * Nota: Las coordenadas y dimensiones absolutas se toman en un espacio de 512 x 512,
	 * que son las dimensiones de la imagen de fondo. Por tanto multiplicando por el side,
	 * dividiendo por 512 ( igual a desplazar 9 bits ), y sumando el offset correspondiente,
	 * se obtienen las nuevas coordenadas ajustadas a las dimensiones reales del componente.
	 *  
	 * @param g2 Contexto gráfico empleado para dibujar el componente.
	 * @param rueda Elemento a representar.
	 * @param x Posición X en coordenadas absolutas.
	 * @param y Posición Y en coordenadas absolutas.
	 * @param w Anchura en valores absolutos.
	 * @param h Altura en valores absolutos.
	 * @param sOff Desplazamiento vertical, en valores absolutos, que se aplicará la imagen según el sentido.
	 * @param side Mínimo entre la anchura y altura del componente en pixels.
	 * @param xOff Despazamiento de la coordenada X en pixels.
	 * @param yOff Despazamiento de la coordenada Y en pixels.
	 */
	private void paintRueda( Graphics2D g2, EstadoRobot.Rueda rueda, int x, int y, int w, int h, int sOff, int side, int xOff, int yOff ){
		if( rueda != EstadoRobot.Rueda.PARADA ){
			if( rueda == EstadoRobot.Rueda.ADELANTE )
				g2.drawImage(
						IMGS.flecha,
						( x * side >> 9 ) + xOff, ( ( y - sOff ) * side >> 9 ) + yOff,
						w * side >> 9, h * side >> 9, this
				);
			else
				g2.drawImage(
						IMGS.atras,
						( x * side >> 9 ) + xOff, ( ( y + sOff ) * side >> 9 ) + yOff,
						w * side >> 9, h * side >> 9, this
				);
		}
	}
	
	/**
	 * Método muestra mediante imágenes un led, empleando la imagen apropiada dependiendo del estado.
	 * 
	 * Nota: Las coordenadas y dimensiones absolutas se toman en un espacio de 512 x 512,
	 * que son las dimensiones de la imagen de fondo. Por tanto multiplicando por el side,
	 * dividiendo por 512 ( igual a desplazar 9 bits ), y sumando el offset correspondiente,
	 * se obtienen las nuevas coordenadas ajustadas a las dimensiones reales del componente.
	 * 
	 * @param g2 Contexto gráfico empleado para dibujar el componente.
	 * @param led Elemento a representar.
	 * @param x Posición X en coordenadas absolutas.
	 * @param y Posición Y en coordenadas absolutas.
	 * @param w Anchura en valores absolutos.
	 * @param h Altura en valores absolutos.
	 * @param side Mínimo entre la anchura y altura del componente en pixels.
	 * @param xOff Despazamiento de la coordenada X en pixels.
	 * @param yOff Despazamiento de la coordenada Y en pixels.
	 */
	private void paintLed( Graphics2D g2, EstadoRobot.Led led, int x, int y, int w, int h, int side, int xOff, int yOff ){
		g2.drawImage(
				( led == EstadoRobot.Led.APAGADO ) ? IMGS.led_off: IMGS.led_on,
				( x * side >> 9 ) + xOff, ( y * side >> 9 ) + yOff, w * side >> 9, h * side >> 9,
				this
		);
	}
	
	/** Variable temporal para no recalcular las tipografía en cada dibujado. */
	private transient int previousWidth = -1;
	
	/** Variable temporal que almacena la tipografía ajustada. */
	private transient Font currentFont1 = null;
	
	/** Factor para ajustar la altura de la fuente 2 con más precisión. */
	private static final double ajusteFont2 = 1.2;
	
	/** Variable temporal que almacena la tipografía ajustada. */
	private transient Font currentFont2 = null;
	
	/**
	 * Método que genera las tipografías con el tamaño adecuado para mostrar
	 * los datos necesarios. Sólo se regenerarán las tipografías cuando cambie
	 * el tamaño del componente.
	 * 
	 * Nota: Las dimensiones absolutas se toman en un espacio de 512 x 512,
	 * que son las dimensiones de la imagen de fondo. Por tanto multiplicando por el side,
	 * dividiendo por 512 ( igual a desplazar 9 bits ), se obtienen las dimensiones reales
	 * del componente.
	 * 
	 * @param g2 Contexto gráfico empleado para dibujar el componente.
	 * @param w1 Anchura, en valores absolutos, del texto mostrado con la tipografía 1.
	 * @param h1 Altura, en valores absolutos, del texto mostrado con la tipografía 1.
	 * @param w2 Anchura, en valores absolutos, del texto mostrado con la tipografía 2.
	 * @param h2 Altura, en valores absolutos, del texto mostrado con la tipografía 2.
	 * @param side Mínimo entre la anchura y altura del componente en pixels.
	 */
	private void regenerateFonts( Graphics2D g2, int w1, int h1, int w2, int h2, int side ){
		w1 = w1 * side >> 9;
		if( previousWidth != w1 ){
			previousWidth = w1;
			h1 = h1 * side >> 9;
			w2 = w2 * side >> 9;
			h2 = h2 * side >> 9;
			
			FontMetrics fontMetrics = g2.getFontMetrics( FONT );
			Rectangle2D bounds;
			AffineTransform at;
			
			bounds = fontMetrics.getStringBounds( "0x00: 00000000", g2 );
			at = new AffineTransform();
			at.scale( w1 / bounds.getWidth(), h1 / bounds.getHeight() );
			currentFont1 = FONT.deriveFont( at );
			
			bounds = fontMetrics.getStringBounds( "0", g2 );
			at = new AffineTransform();
			at.scale( w2 / bounds.getWidth(), h2 * ajusteFont2 / fontMetrics.getAscent()  );
			currentFont2 = FONT.deriveFont( at );
		}
	}
	
	/**
	 * Función que transforma un entero en una cadena con su representación en
	 * binario. Mostrando el número de bits que se indican como parámetro.
	 * 
	 * Si valor tiene un tamaño mayor que el número de bits indacado:
	 *    -Se devolverán los bits de menor peso.
	 * Si valor tiene un tamaño menor que el número de bits indacado:
	 *    -Se añadirá delante el número de ceros necesarios.
	 *    
	 * @param data Entero a transformar.
	 * @param nBits Número de bits a mostrar.
	 * @return La cadena generada.
	 */
	private static String toBinary( int data, int nBits ){
		String str = Integer.toBinaryString( data );
		int len = str.length();
		
		if( len == nBits )
			return str;
		if( len > nBits )
			return str.substring( len - nBits );
		
		StringBuffer buff = new StringBuffer();
		while( len < nBits ){
			buff.append( '0' );
			len++;
		}
		buff.append( str );
		return buff.toString();		
	}
	
	/**
	 * Método que dibuja los valores binarios del portA y el portB.
	 * 
	 * Nota: Las coordenadas y dimensiones absolutas se toman en un espacio de 512 x 512,
	 * que son las dimensiones de la imagen de fondo. Por tanto multiplicando por el side,
	 * dividiendo por 512 ( igual a desplazar 9 bits ), y sumando el offset correspondiente,
	 * se obtienen las nuevas coordenadas ajustadas a las dimensiones reales del componente.
	 * 
	 * @param g2 Contexto gráfico empleado para dibujar el componente.
	 * @param estado Estado con los datos a representar.
	 * @param x Posición X en coordenadas absolutas.
	 * @param y Posición Y en coordenadas absolutas.
	 * @param w Anchura en valores absolutos.
	 * @param h Altura en valores absolutos.
	 * @param b Anchura, en valores absolutos, del borde exterior.
	 * @param side Mínimo entre la anchura y altura del componente en pixels.
	 * @param xOff Despazamiento de la coordenada X en pixels.
	 * @param yOff Despazamiento de la coordenada Y en pixels.
	 */
	private void paintPorts( Graphics2D g2, EstadoRobot estado, int x, int y, int w, int h, int b, int side, int xOff, int yOff ){
		x = ( x * side >> 9 ) + xOff; 
		y = ( y * side >> 9 ) + yOff;
		w = w * side >> 9;
		h = h * side >> 9;
		int b2 = b * side >> 8; //borde * 2
		b = b2 >> 1;
	
		String strA = String.format( "0x%02X: %s", estado.getPortA() & 0xFF, toBinary( estado.getPortA(), 8 ) );
		String strB = String.format( "0x%02X: %s", estado.getPortB() & 0xFF, toBinary( estado.getPortB(), 8 ) );
		
		g2.setPaint( DISPLAY_COLOR );
		g2.fillRoundRect( x - b, y - b, w + b2, h + b2, b, b );
		
		g2.setFont( currentFont1 );
		g2.setPaint( Color.YELLOW );

		y += g2.getFontMetrics().getHeight();
		y -= g2.getFontMetrics().getDescent();
		int newline = h / 4;
		g2.drawString( "PORT A", x, y );
		y += newline;
		g2.drawString( strA,     x, y );
		y += newline;
		g2.drawString( "PORT B", x, y );
		y += newline;
		g2.drawString( strB,     x, y );
	}
	
	/**
	 * Método que muestra mediante una serie de iconos el estado general del robot.
	 * 
	 * Nota: Las coordenadas y dimensiones absolutas se toman en un espacio de 512 x 512,
	 * que son las dimensiones de la imagen de fondo. Por tanto multiplicando por el side,
	 * dividiendo por 512 ( igual a desplazar 9 bits ), y sumando el offset correspondiente,
	 * se obtienen las nuevas coordenadas ajustadas a las dimensiones reales del componente.
	 * 
	 * @param g2 Contexto gráfico empleado para dibujar el componente.
	 * @param estado Estado con los datos a representar.
	 * @param x Posición X en coordenadas absolutas.
	 * @param y Posición Y en coordenadas absolutas.
	 * @param w Anchura en valores absolutos.
	 * @param h Altura en valores absolutos.
	 * @param p Anchura, en valores absolutos, del relleno interior.
	 * @param b Anchura, en valores absolutos, del borde exterior.
	 * @param side Mínimo entre la anchura y altura del componente en pixels.
	 * @param xOff Despazamiento de la coordenada X en pixels.
	 * @param yOff Despazamiento de la coordenada Y en pixels.
	 */
	private void paintEstadoGeneral( Graphics2D g2, EstadoRobot estado, int x, int y, int w, int h, int p, int b, int side, int xOff, int yOff ){		
		x = ( x * side >> 9 ) + xOff; 
		y = ( y * side >> 9 ) + yOff;
		int eW = ( w - 2 * p ) / 3 * side >> 9;
		w = w * side >> 9;
		h = h * side >> 9;
		p = p * side >> 9;
		int b2 = b * side >> 8; //borde * 2
		b = b2 >> 1;
	
		g2.setPaint( DISPLAY_COLOR );
		g2.fillRoundRect( x - b, y - b, w + b2, h + b2, b, b );
		
		switch( estado.getControl() ){
		case HOME:
			g2.drawImage( IMGS.casa, x, y, eW, h, this );
			if( estado.isParadaEmergencia() )
				g2.drawImage( IMGS.emergencia, x + eW + p, y, eW, h, this );
			break;
		case AUTO:
			g2.drawImage( IMGS.automatico, x, y, eW, h, this );
			if( estado.isParadaEmergencia() )
				g2.drawImage( IMGS.emergencia, x + eW + p, y, eW, h, this );
			else{
				g2.setFont( currentFont2 );
				g2.setPaint( Color.YELLOW );
				g2.drawString(
						Integer.toString( estado.getRuta() ),
						x + eW + p, (int)( y + g2.getFontMetrics().getAscent() / ajusteFont2 )
				);
				int posX = x + 13 * eW / 8 + p;
				switch( estado.getSentido() ){
				case IDA:
					g2.drawImage( IMGS.flecha, posX, y, eW/2, h/2, this );
					break;
				case VUELTA:
					g2.drawImage( IMGS.flecha, posX, y + h, eW/2, -h/2, this );
					break;
				default:
					g2.drawImage( IMGS.flecha, posX, y +h/2, eW/2, -h/2, this );
					g2.drawImage( IMGS.flecha, posX, y +h/2, eW/2, h/2, this );
					break;
				}
			}
			break;
		case MANUAL:
			g2.drawImage( IMGS.manual, x, y, eW, h, this );
			if( estado.isParadaEmergencia() )
				g2.drawImage( IMGS.emergencia, x + eW + p, y, eW, h, this );
			else
				switch( estado.getOrdenManual() ){
				case AVANZAR:
					g2.drawImage( IMGS.flecha, x + eW + p, y, eW, h, this );
					break;
				case RETROCEDER:
					g2.drawImage( IMGS.flecha, x + eW + p, y + h, eW, -h, this );
					break;
				case GIRAR_IZQ:
					g2.drawImage( IMGS.giro, x + eW + p, y, eW, h, this );
					break;
				case GIRAR_DER:
					g2.drawImage( IMGS.giro, x + 2 * eW + p, y, -eW, h, this );
					break;
				case PARAR:
					g2.drawImage( IMGS.paro, x + eW + p, y, eW, h, this );
					break;
				}
			break;
		}

		g2.drawImage( 
				( estado.getEstadoCarga() == EstadoRobot.Carga.CARGADO ) ? IMGS.cargado: IMGS.descargado,
				x + 2 * ( eW + p ), y, eW, h, this
		);
	}
	
	/**
	 * Método que dibuja todas las partes que resprensetan al robot.
	 * 
	 * @param g2D Contexto gráfico empleado para dibujar el componente.
	 * @param estado Estado con los datos a representar.
	 * @param side Mínimo entre la anchura y altura del componente en pixels.
	 * @param xOff Despazamiento de la coordenada X en pixels.
	 * @param yOff Despazamiento de la coordenada Y en pixels.
	 */
	private void paintRobot( Graphics2D g2D, EstadoRobot estado, int side, int xOff, int yOff ){
		
		g2D.setRenderingHint( RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON );
		g2D.setRenderingHint( RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_ON );
		g2D.setRenderingHint( RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON );
		g2D.setRenderingHint( RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY );
		
		g2D.setStroke( new BasicStroke( 3.0f * side / 512 ) );
		
		paintSensor( g2D, estado.getSensorDelIzq(),  66,  5, 60, side, xOff, yOff );
		paintSensor( g2D, estado.getSensorDelDer(), 387,  5, 60, side, xOff, yOff );
		paintSensor( g2D, estado.getSensorCenIzq(), 196, 53, 60, side, xOff, yOff );
		paintSensor( g2D, estado.getSensorCenDer(), 256, 53, 60, side, xOff, yOff );

		g2D.drawImage( IMGS.fondo, xOff, yOff, side, side, this );
		
		paintBumper( g2D, estado.getBumperIzq(),   0, 53,  160, 80, side, xOff, yOff );
		paintBumper( g2D, estado.getBumperDer(), 512, 53, -160, 80, side, xOff, yOff );
		
		paintRueda( g2D, estado.getRuedaIzq(),  40, 210, 64, 128, 32, side, xOff, yOff );
		paintRueda( g2D, estado.getRuedaDer(), 412, 210, 64, 128, 32, side, xOff, yOff );
		
		paintLed( g2D, estado.getLed(), 240, 420, 32, 32, side, xOff, yOff );
		
		regenerateFonts( g2D, 220, 30, 37, 50, side );
		paintPorts( g2D, estado, 146, 150, 220, 120, 13, side, xOff, yOff );
		paintEstadoGeneral( g2D, estado, 168, 320, 176, 50, 13, 13, side, xOff, yOff );
	}
	
	/* (non-Javadoc)
	 * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
	 */
	public void paintComponent( Graphics g ){
		
		int min = Math.min( this.getWidth(), this.getHeight() );
		int xOff = this.getWidth()  - min >> 1;
		int yOff = this.getHeight() - min >> 1;
		
		Graphics2D g2D = (Graphics2D)g;
		
		if( this.isEnabled() )
			paintRobot( g2D, estado, min, xOff, yOff );
		else{
			g2D.setRenderingHint( RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY );
			g2D.drawImage( disabledImage, xOff, yOff, min, min, this );
		}
	}
	
	/**
	 * Método que asigna el estado del robot, y repinta el componente.
	 * 
	 * @param estado Estado asignado.
	 */
	void setEstado( EstadoRobot estado ){
		this.estado = estado;
		this.repaint();
	}
}