package robot.gui;

import giovynet.serial.Com;

import java.awt.event.ActionEvent;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.Icon;

import robot.Definitions;
import robot.EstadoRobot;
import utils.ImageUtils;

/**
 * Clase que contiene y sincroniza, en función del estado del robot, las distintas
 * acciones que se le pueden enviar.
 */
final class ButtonActions{
	
	/**
	 * Clase que envia un dato a tavés de un puerto serie.
	 */
	@SuppressWarnings( "serial" )
	private static class SendSingleDataAction extends AbstractAction{
		
		/** Puerto por el que se enviará el dato */
		private Com com;
		/** Dato que será enviado */
		private int data;
		
		/**
		 * Constructor por defecto.
		 */
		SendSingleDataAction(){
		}
		
		/**
		 * Constructor que crea la acción y le asigna un nombre, una descripción
		 * y el dato a eviar.
		 * @param name Nombre asignado.
		 * @param description Descripción asignada.
		 * @param data Dato asignado.
		 */
		SendSingleDataAction( String name, String description, int data ){
			this.putValue( Action.NAME, name );
			this.putValue( Action.SHORT_DESCRIPTION, description );
			this.data = data;
		}

		/**
		 * Constructor que crea la acción y le asigna un icono, una descripción
		 * y el dato a eviar.
		 * 
		 * @param icon Icono asignado.
		 * @param description Descripción asignada.
		 * @param data Dato asignado.
		 */
		SendSingleDataAction( Icon icon, String description, int data ){
			this.putValue( Action.SMALL_ICON, icon );
			this.putValue( Action.SHORT_DESCRIPTION, description );
			this.data = data;
		}

		/**
		 * Método que asigna el puerto por el que enviar el dato.
		 * @param com Puerto asignado.
		 */
		final void setCom( Com com ){
			this.com = com;
		}

		/**
		 * Método que asigna el dato que será enviado.
		 * @param data Dato asignado.
		 */
		final void setData( int data ){
			this.data = data;
		}
		
	    /* (non-Javadoc)
	     * @see javax.swing.AbstractAction#setEnabled(boolean)
	     */
		@Override
	    public void setEnabled(boolean newValue) {
	    	super.setEnabled( newValue && this.com != null );
        }

		/* (non-Javadoc)
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		@Override
		public void actionPerformed( ActionEvent ae ){
    		try{
				com.sendSingleData( data );
			}catch( Exception e ){
				e.printStackTrace();
			}
		}
	}
	
	/**
	 * Clase, derivada de SendSingleDataAction, que almacena dos estados distintos,
	 * y envia, en fución de su estado, un detrminado dato a tavés de un puerto serie.
	 */
	@SuppressWarnings( "serial" )
	private static class SendToggleDataAction extends SendSingleDataAction{
		
		/** Icono cuando NO está seleccionado. */
		private final Icon icon1;
		/** Descripción cuando NO está seleccionado. */
		private final String description1;
		/** Dato que será enviado cuando NO está seleccionado. */
		private final int data1;
		
		/** Icono cuando está seleccionado. */
		private final Icon icon2;
		/** Descripción cuando está seleccionado. */
		private final String description2;
		/** Dato que será enviado cuando está seleccionado. */
		private final int data2;
	
		/**
		 * Constructor que crea la acción, con el estado a NO seleccionado.
		 * 
		 * @param icon1 Icono cuando NO está seleccionado.
		 * @param description1 Descripción cuando NO está seleccionado.
		 * @param data1 Dato que será enviado cuando NO está seleccionado.
		 * @param icon2 Icono cuando está seleccionado.
		 * @param description2 Descripción cuando está seleccionado.
		 * @param data2 Dato que será enviado cuando está seleccionado.
		 */
		SendToggleDataAction(
				Icon icon1, String description1, int data1,
				Icon icon2, String description2, int data2
		){
			this.icon1 = icon1;
			this.description1 = description1;
			this.data1 = data1;

			this.icon2 = icon2;
			this.description2 = description2;
			this.data2 = data2;
			
			this.setSelected( false );
		}

		/**
		 * Método que establece el estado de la acción.
		 * 
		 * @param selected Estado asignado.
		 */
		public void setSelected( boolean selected ){
			if( !selected ){
				this.putValue( Action.SMALL_ICON, icon1 );
				this.putValue( Action.SHORT_DESCRIPTION, description1 );
				this.setData( data1 );
			}else{
				this.putValue( Action.SMALL_ICON, icon2 );
				this.putValue( Action.SHORT_DESCRIPTION, description2 );
				this.setData( data2 );
			}
		}		
	}

	final SendSingleDataAction ruta1;
	final SendSingleDataAction ruta2;
	final SendSingleDataAction ruta3;
	
	final SendSingleDataAction home;
	
	final SendToggleDataAction detener_reanudar;
	
	final SendSingleDataAction avanzar;
	final SendSingleDataAction girarIzq;
	final SendSingleDataAction parar;
	final SendSingleDataAction girarDer;
	final SendSingleDataAction retroceder;
	
	final SendToggleDataAction cargar_descargar;
	
	private EstadoRobot estado;
	
	/**
	 * Constructor que crea las distintas acciones contenidas y almacena
	 * la referencia al estado del robot, empleada para sincronizar dichas
	 * acciones.
	 * 
	 * @param estado Referencia al estado del robot asignada.
	 */
	ButtonActions( ){
		
		this.estado = new EstadoRobot( (byte)0, (byte)0, (byte)0 );
		
		ruta1 = new SendSingleDataAction( "Ruta 1", "Ruta Izquierda", Definitions.ORD_CAMINO_IZQ );
		ruta2 = new SendSingleDataAction( "Ruta 2", "Ruta Central", Definitions.ORD_CAMINO_RECTO );
		ruta3 = new SendSingleDataAction( "Ruta 3", "Ruta Derecha", Definitions.ORD_CAMINO_DER );
		
		home = new SendSingleDataAction(
			ImageUtils.getImageIcon( "resources/botones/casa.png", 48, 48 ),
			"Home",
			Definitions.ORD_HOME
		);
		
		detener_reanudar = new SendToggleDataAction(
			ImageUtils.getImageIcon( "resources/botones/stop_off.png" ),
			"Parada de Emergencia",
			Definitions.ORD_EMERGENCIA,
			ImageUtils.getImageIcon( "resources/botones/stop_on.png" ),
			"Reanudar",
			Definitions.ORD_EMERGENCIA
		);
		
		avanzar = new SendSingleDataAction(
			ImageUtils.getImageIcon( "resources/botones/adelante.png", 32, 32 ),
			"Avanzar",
			Definitions.ORD_ADELANTE
		);
		girarIzq = new SendSingleDataAction(
			ImageUtils.getImageIcon( "resources/botones/izquierda.png", 32, 32 ),
			"Girar a la Izquierda",
			Definitions.ORD_GIRO_IZQ
		);
		parar = new SendSingleDataAction(
			ImageUtils.getImageIcon( "resources/botones/paro.png", 32, 32 ),
			"Parar",
			Definitions.ORD_PARAR
		);
		girarDer = new SendSingleDataAction(
			ImageUtils.getImageIcon( "resources/botones/derecha.png", 32, 32 ),
			"Girar a la Derecha",
			Definitions.ORD_GIRO_DER
		);
		retroceder = new SendSingleDataAction(
			ImageUtils.getImageIcon( "resources/botones/atras.png", 32, 32 ),
			"Retroceder",
			Definitions.ORD_ATRAS
		);
		
		cargar_descargar = new SendToggleDataAction(
			ImageUtils.getImageIcon( "resources/botones/descargado.png", 48, 48 ),
			"Cargar",
			Definitions.ORD_CARGA_DESCARGA,
			ImageUtils.getImageIcon( "resources/botones/cargado.png", 48, 48 ),
			"Descargar",
			Definitions.ORD_CARGA_DESCARGA
		);
		
		update();
	}
	
	/**
	 * Método que asigna el puerto, por el que enviar los datos, a las distintas acciones.
	 * @param com Puerto asignado.
	 */
	void setCom( Com com ){
		ruta1.setCom( com );
		ruta2.setCom( com );
		ruta3.setCom( com );
		
		home.setCom( com );
		
		detener_reanudar.setCom( com );
		
		avanzar.setCom( com );
		girarIzq.setCom( com );
		parar.setCom( com );
		girarDer.setCom( com );
		retroceder.setCom( com );
		
		cargar_descargar.setCom( com );
		
		update();
	}
	
	/**
	 * Método que asigna el estado del robot, y actualiza las acciones.
	 * 
	 * @param estado Estado asignado.
	 */
	void setEstado( EstadoRobot estado ){
		this.estado = estado;
		this.update();
	}
	
	/**
	 * Método que actualiza el estado de las acciones en función del
	 * puerto serie y del estado del robot.
	 */
	private void update(){
		
		detener_reanudar.setSelected( estado.isParadaEmergencia() );
		cargar_descargar.setSelected( estado.getEstadoCarga() == EstadoRobot.Carga.CARGADO );
		
		detener_reanudar.setEnabled( true );
		
		if( estado.isParadaEmergencia() ){
			ruta1.setEnabled( false );
			ruta2.setEnabled( false );
			ruta3.setEnabled( false );
			
			home.setEnabled( false );
			
			avanzar.setEnabled( false );
			girarIzq.setEnabled( false );
			parar.setEnabled( false );
			girarDer.setEnabled( false );
			retroceder.setEnabled( false );
			
			cargar_descargar.setEnabled( false );
		}else{
			ruta1.setEnabled( estado.getControl() == EstadoRobot.Control.HOME );
			ruta2.setEnabled( estado.getControl() == EstadoRobot.Control.HOME );
			ruta3.setEnabled( estado.getControl() == EstadoRobot.Control.HOME );
			
			home.setEnabled( estado.getControl() == EstadoRobot.Control.MANUAL );
			
			avanzar.setEnabled( estado.getControl() == EstadoRobot.Control.MANUAL );
			girarIzq.setEnabled( estado.getControl() == EstadoRobot.Control.MANUAL );
			parar.setEnabled( true );
			girarDer.setEnabled( estado.getControl() == EstadoRobot.Control.MANUAL );
			retroceder.setEnabled( estado.getControl() == EstadoRobot.Control.MANUAL );
			
			cargar_descargar.setEnabled( estado.getControl() != EstadoRobot.Control.AUTO );
		}
	}
}
