package robot.comunication;

import giovynet.serial.Com;

import java.util.Collection;
import java.util.LinkedList;

import robot.Definitions;
import robot.EstadoRobot;

/**
 * Hilo encargado de leer es estado del robot a través del puerto serie y
 * notificar los cambios de estado leido a los observadores registrados.
 */
public class Lector extends Thread{
	
	/** Objecto usado para evitar accesos concurrentes al puerto serie.*/
	private final Object lock = new Object();
	
	/** Puerto serie de donde se leerán los datos.*/
	private Com com;
	
	/** Lista de de observadores. */
	private final Collection<RobotObserver> observers;
	
	/**
	 * Constructor que crea el hilo y asigna la referencia donde
	 * almacenar el estado del robot.
	 * 
	 * @param estado Referencia asignada.
	 */
	public Lector(){
		this.com = null;
		this.observers = new LinkedList<RobotObserver>();
	}
	
	/**
	 * Método que asigna el puerto de donde se leerán los datos.
	 * 
	 * @param com Puerto asignado.
	 */
	public final void setCom( Com com ){
		this.com = com;
		if( com != null )
			synchronized( lock ){
				lock.notifyAll();
			}
	}
	
	/**
	 * Método que lee es estado del robot a través del puerto serie.
	 * 
	 * Nota: Se usa un método con el puerto como parámetro, para evitar
	 * los errores que se pudieran producir si se cambia el puerto a 
	 * mitad de lectura.
	 * 
	 * @param com Puerto del que se leerán los datos.
	 * @throws Exception Excepción lanzada si hay algún error de lectura.
	 */
	private void read( Com com ) throws Exception{
		
		int head = com.receiveSingleDataInt() & 0xFF;
		System.out.printf( "%h\n", head );
		if( (byte)head == Definitions.CABECERA ){
			
			int portA;
			portA = com.receiveSingleDataInt();
			portA &= 0xFF;
			
			int portB;
			portB = com.receiveSingleDataInt();
			portB &= 0xFF;
			
			int state;
			state = com.receiveSingleDataInt();
			state &= 0xFF;
			
			int check;
			check = com.receiveSingleDataInt();
			check &= 0xFF;
			
			System.out.printf( "%h %h %h %h\n", portA, portB, state, check );
			if( check  == 0x0F )
				notifyObservers( new EstadoRobot( (byte)portA, (byte)portB, (byte)state ) );
		}
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Runnable#run()
	 */
	@Override
	public void run(){
		while( true ){
			try{
				if( com == null )
					synchronized( lock ){
						lock.wait();
					}
				else
					read( com );
			}catch( InterruptedException ignorada ){
				ignorada.printStackTrace();
			}catch( Exception ignorada ){
				ignorada.printStackTrace();
			}
		}
	}
	
	/**
	 * Método que registra un nuevo observador.
	 * 
	 * @param observer Observador registrado.
	 */
	public void registerObserver( RobotObserver observer ){
		observers.add( observer );
	}

	/**
	 * Método que notifica a los observadores que ha cambiado el
	 * estado del robot.
	 * 
	 * @param estado Nuevo estado del robot.
	 */
	public void notifyObservers( EstadoRobot estado ){
		for( RobotObserver observer: observers )
			observer.notifyRobotStateChange( estado );
	}
}
