package robot;

/**
 * Clase que almacena e interpreta el estado del robot.
 */
public final class EstadoRobot{
	
	/**
	 * Enumeración que describe los distintos estados del sensor IR.
	 */
	public static enum SensorIR{
		NEGRO, BLANCO
	}
	
	/**
	 * Enumeración que describe los distintos estados del bumper.
	 */
	public static enum Bumper{
		ESTIRADO, COMPRIMIDO
	}
	
	/**
	 * Enumeración que describe los distintos estados del giro de una rueda.
	 */
	public static enum Rueda{
		ATRAS, PARADA, ADELANTE
	}
	
	/**
	 * Enumeración que describe los distintos estados de un led.
	 */
	public static enum Led{
		APAGADO, ENCENDIDO
	}
	
	/**
	 * Enumeración que describe los distintos estados en que puede estar controlado el robot.
	 */
	public static enum Control{
		HOME, AUTO, MANUAL
	}
	
	/**
	 * Enumeración que describe los distintos estados de la ruta automática.
	 */
	public static enum Sentido{
		IDA, VUELTA, INDEFINIDO
	}
	
	/**
	 * Enumeración que describe las distintas ordenes de moviento.
	 */
	public static enum Movimiento{
		AVANZAR, RETROCEDER, GIRAR_IZQ, GIRAR_DER, PARAR
	}
	
	/**
	 * Enumeración que describe los distintos estados de la carga.
	 */
	public static enum Carga{
		DESCARGADO, CARGADO
	}
	
	private final byte portA;
	private final byte portB;
	private final byte state;
	
	public EstadoRobot( byte portA, byte portB, byte state ){
		this.portA = portA;
		this.portB = portB;
		this.state = state;
	}
	
	/**
	 * @return El valor del PortA.
	 */
	public byte getPortA(){
		return portA;
	}
	
	/**
	 * @return El estado del bumper izquierdo.
	 */
	public Bumper getBumperIzq(){
		return ( portA & Definitions.BUMPER_IZQ_MASK ) == 0 ? Bumper.ESTIRADO: Bumper.COMPRIMIDO;
	}

	/**
	 * @return El estado del bumper derecho.
	 */
	public Bumper getBumperDer(){
		return ( portA & Definitions.BUMPER_DER_MASK ) == 0 ? Bumper.ESTIRADO: Bumper.COMPRIMIDO;
	}
	
	/**
	 * @return El valor del PortB.
	 */
	public byte getPortB(){
		return portB;
	}
	
	/**
	 * @return El estado del sensor delantero izquierdo.
	 */
	public SensorIR getSensorDelIzq(){
		return ( portB & Definitions.SENSOR_DEL_IZQ_MASK ) == 0 ? SensorIR.BLANCO: SensorIR.NEGRO;
	}

	/**
	 * @return El estado del sensor delantero derecho.
	 */
	public SensorIR getSensorDelDer(){
		return ( portB & Definitions.SENSOR_DEL_DER_MASK ) == 0 ? SensorIR.BLANCO: SensorIR.NEGRO;
	}

	/**
	 * @return El estado del sensor central izquierdo.
	 */
	public SensorIR getSensorCenIzq(){
		return ( portB & Definitions.SENSOR_CEN_IZQ_MASK ) == 0 ? SensorIR.BLANCO: SensorIR.NEGRO;
	}

	/**
	 * @return El estado del sensor central derecho.
	 */
	public SensorIR getSensorCenDer(){
		return ( portB & Definitions.SENSOR_CEN_DER_MASK ) == 0 ? SensorIR.BLANCO: SensorIR.NEGRO;
	}

	/**
	 * @return El estado de la rueda izquierda.
	 */
	public Rueda getRuedaIzq(){
		switch( portB & Definitions.RUDEDA_IZQ_MASK ){
		case Definitions.RUDEDA_IZQ_ADELANTE:
			return Rueda.ADELANTE;
		case Definitions.RUDEDA_IZQ_ATRAS:
			return Rueda.ATRAS;
		default:
			return Rueda.PARADA;
		}
	}

	/**
	 * @return El estado de la rueda derecha.
	 */
	public Rueda getRuedaDer(){
		switch( portB & Definitions.RUDEDA_DER_MASK ){
		case Definitions.RUDEDA_DER_ADELANTE:
			return Rueda.ADELANTE;
		case Definitions.RUDEDA_DER_ATRAS:
			return Rueda.ATRAS;
		default:
			return Rueda.PARADA;
		}
	}

	/**
	 * @return El estado del led.
	 */
	public Led getLed(){
		return ( portB & Definitions.LED_MASK ) == 0 ? Led.APAGADO: Led.ENCENDIDO;
	}
	
	/**
	 * @return CIERTO si el robot está en parada de emergencia, FALSO en caso contrario.
	 */
	public boolean isParadaEmergencia(){
		return ( state & Definitions.EMERGENCIA_MASK ) != 0;
	}
	
	/**
	 * @return El modo de control del robot.
	 */
	public Control getControl(){
		switch( state & Definitions.CONTROL_MASK ){
		case Definitions.CONTROL_AUTO:
			return Control.AUTO;
		case Definitions.CONTROL_MANUAL:
			return Control.MANUAL;
		default:
			return Control.HOME;
		}
	}
	
	/**
	 * @return El número de la ruta automática que está recorriendo el robot.
	 */
	public int getRuta(){
		return state & Definitions.RUTA_MASK;
	}
	
	/**
	 * @return El sentido en la ruta automática que está recorriendo el robot.
	 */
	public Sentido getSentido(){
		switch( state & Definitions.SENTIDO_MASK ){
		case Definitions.SENTIDO_IDA:
			return Sentido.IDA;
		case Definitions.SENTIDO_VUELTA:
			return Sentido.VUELTA;
		default:
			return Sentido.INDEFINIDO;
		}
	}
	
	/**
	 * @return La orden manual que está ejecutando el robot.
	 */
	public Movimiento getOrdenManual(){
		switch( state & Definitions.MOVIMIENTO_MASK ){
		case Definitions.MOVIMIENTO_ADELANTE:
			return Movimiento.AVANZAR;
		case Definitions.MOVIMIENTO_ATRAS:
			return Movimiento.RETROCEDER;
		case Definitions.MOVIMIENTO_GIRO_IZQ:
			return Movimiento.GIRAR_IZQ;
		case Definitions.MOVIMIENTO_GIRO_DER:
			return Movimiento.GIRAR_DER;
		default:
			return Movimiento.PARAR;
		}
	}
	
	/**
	 * @return El estado de l carga del robot.
	 */
	public Carga getEstadoCarga(){
		return ( state & Definitions.CARGA_MASK ) == 0 ? Carga.DESCARGADO: Carga.CARGADO;
	}

}
