/*****************************************************************************/
/* test-oscillator8.c  Enero-2010                                            */
/*---------------------------------------------------------------------------*/
/* Ejemplo de oscilacion de 8 servos                                         */
/*---------------------------------------------------------------------------*/
/*  Juan Gonzalez <juan@iearobotics.com>                                     */
/*  Andres Prieto-Moreno <andres@iearobotics.com>                            */
/*---------------------------------------------------------------------------*/
/*  LICENCIA GPL                                                             */
/*****************************************************************************/

//-- Especificar el pic a emplear
#include <pic16f876a.h>
#include "servos.h"
#include "sin.h"


#define PULSADOR RA0  //-- Skycube 

//-- Constantes para determinar si pulsador esta apretado o no
#define ON       0
#define OFF      1

/******************************************/
/*  PARAMETROS DE LAS OSCILACIONES        */
/******************************************/

//-- Velocidad de fase (en grados) (1-180)
//-- Este parametro permite cambiar la frecuencia de oscilacion, asi como
//-- el sentido. Es un numero entero
//-- Valores negativos hacen que oscile en sentido contrario
//-- El periodo de oscilacion se calcula con la formula:
//--  T=360/W, con W expresado en grados/seg. Para los servos futaba,
//-- la velocidad angular es aproximadamente de 260 grados/seg, por lo que
//-- la formula queda:
//--
//--  T=7.2/INC
//--
//--  Tabla de valores de T en funcion de INC:
//--
//--      INC      T (seg)   f(Hz)
//--    -----------------------------
//--       0       no oscila   0
//--       1       7.2         0.14
//--       2       3.7         0.28
//--       3       2,4         0.42
//--       4       1.8         0.55
//--       5       1.4         0.71
//--       6       1.2         0.83
//--       7       1.03        0.97
//--       8       0.9         1.11
//--       9       0.8         1.25
//--      10       0.72        1.39
//--      ..       ...         ...
#define INC 5

//--Fase inicial de los servos (en grados) (0-360)
#define FASE0 0

/**********************************************/
/* VARIABLES GLOBALES                         */
/**********************************************/
//-- Amplitudes de las oscilaciones
static int A[8]={50,50,50,50,50,50,50,50};

//-- Fases iniciales
static int FASE[8]={0,0,90,90,180,225,270,315};

//-- Offset (entre -90 y 90)
static int OFFSET=0;

//-- Numero de periodos
static int NT=0;
static int new_period=0;

//-- Secuencia de offsets
//__code int off_sec[]  =  {40, -40 };
//__code int fase1_sec[] = {0,  0};
//__code int fase2_sec[] = {90, 90};

__code int off_sec[]  =  {0, -40,  0, 40 };
__code int fase1_sec[] = {0,  0,  0,  0};
__code int fase2_sec[] = {90, 90, 90,  90};

//__code int off_sec[]  =  {-60};
//__code int fase1_sec[] = {0 };
//__code int fase2_sec[] = {90};


//-- Variable de fase
static int fase;

//-- Acceso a las variables del modulo SERVOS
extern volatile unsigned char pos_servos[8];
extern unsigned char indice;
extern unsigned char tic;
extern unsigned char servo_actual; 

/*******************************************************/
/* RUTINA DE ATENCION A LAS INTERRUPCIONES             */
/*******************************************************/
void intr(void) interrupt 0
{
  //---------------------------------------------------------------
  //-- Rutina de servicio del timer0 para actualizar las senales
  //-- de PWM
  //-----------------------------------------------------------
  if (T0IF==1)
    servos_intr();
}



void oscillator_calculate(int m)
{
  int pos_sin;
  unsigned char upos;
  unsigned char i;
  unsigned char s;
    
     //-- Calculos de los osciladores: obtener siguiente posicion para los sevos
    //-- Se ejecuta cada 2.5ms.
    //-- La variable tic se pone a 1 cada 2.5ms (la activa la rutina de atencion
    //-- a las interrupciones del timer0, que se encarga de generar el pwm)
    if (tic==1) {

      //-- Reiniciar tic. Al cabe de 2.5ms se volvera a activar
      tic=0;

      //-- Obtener el numero de servo actual (variable indice) y su
      //-- mascara binaria. Ambas se obtienen del modulo servos.c
      //-- Esta operacion es muy importante, porque indice y sevo_actual
      //-- son variables globales que se modifican asincronamente mediante
      //-- interrupciones, por lo que es necesario crear una copia local
      //-- en las variables i, s para usarlas en los calculos
      i=indice;
      s=servo_actual;

      //-- Calcular la siguiente posicion del servo i
      pos_sin=sin(A[i],fase+FASE[i]+FASE0)+off_sec[m];

      //-- Realizar la conversion entre grados y tiempo
      upos = servos_grados2tiempo(pos_sin);
  
      //-- Refrescar la posicion del servo
      pos_servos[i]=upos;

      //-- Cuando se llega al servo 0, se incrementa la fase
      if (i==0) {

        //-- Incrementar la fase
        fase=fase + INC;

        if (fase>=360) {
          new_period=1;
          NT++;
        }

        //-- Las fases negativas se convierten a positivas
        //-- BUG: para fases negativas deveria ser: fase=360-fase
        //if (fase<0) fase=359;

        //-- Mantener la fase en el rango 0 - 360 grados
        fase = fase % 360;

      }
    }

}

//----------------------------
//- Comienzo del programa  
//----------------------------

void main(void)
{
  unsigned char m=0;
  unsigned char TAM=sizeof(off_sec)/sizeof(int); 


  //-- recalcular las fases
  FASE[1]=fase1_sec[m];
  FASE[3]=fase2_sec[m];

  //-- Inicializacion del modulo de servos
  servos_init();

  //-- Configurar el pulsador de test
  ADCON1=0x06; //-- Configurar RA0 como entrada digital
  TRISA0 = 1;  //-- RA0 como entrada

  //-- Inicializar variables
  fase=0;
    
  //-- Todos los servos activados
  servos_enable_mask(0xff);

  //-- Habilitar Interrupciones
  GIE=1;

  //----------------------
  //-- Bucle principal
  //----------------------
  while (1) {
      
     //-- Refrescar osciladores
     oscillator_calculate(m);

     //-- Si el pulsador esta apretado, cambiar el incremento de fase
     //if (PULSADOR==ON) {
       if (new_period==1 && NT==6) {

         new_period=0;
         NT=0;

        //-- Siguiente movimiento
        m = (m+1) % TAM;

        //-- recalcular las fases
        FASE[1]=fase1_sec[m];
        FASE[3]=fase2_sec[m];

        //-- Esperar hasta que el boton se suelte
        //while(PULSADOR==ON);
    }
  }
}
