/*****************************************************************************/
/* oscillator-pn.c  Enero-2010                                               */
/*---------------------------------------------------------------------------*/
/* Movimiento de configuraciones Pn                                          */
/*---------------------------------------------------------------------------*/
/*  Juan Gonzalez <juan@iearobotics.com>                                     */
/*---------------------------------------------------------------------------*/
/*  LICENCIA GPL                                                             */
/*****************************************************************************/

#include "servos.h"
#include "sin.h"

/*********************************/
/* VARIABLES DEL MODULO          */
/*********************************/
//-- Amplitudes de las oscilaciones
static int Am[8]={0,0,0,0,0,0,0,0};

//-- Fases de los osciladores
static int FASE[8]={0,0,0,0,0,0,0,0};

//-- Variable de fase
static int fase=0;

//-- Acceso a las variables del modulo SERVOS
extern volatile unsigned char pos_servos[8];
extern unsigned char indice;
extern unsigned char tic;
extern unsigned char servo_actual; 

//-- Acceso a variable de servos virtuales
extern unsigned char v[9];

/**************************************************************************/
/* Inicializar los osciladores. Se recorren todos los servos virtuales    */
/* asignando los valores de amplitud y fase correspondientes              */
/* Dada un servo virtual k, su oscilador correspondientes esta dado por : */
/*                                                                        */
/*  Angulo doblaje servo k = A*sin(fase + (k-1)*DF + FASE0)               */
/*                                                                        */
/**************************************************************************/
void oscillator_init(int amplitud, int df, int fase0)
{
  unsigned char j;
  unsigned char i;
  unsigned char mask=0x00;

  //-- Tomar como fase inicial la actual o 0
  if (fase0==0) fase=0;

  //-- Recorrer todos los servos virtuales
  for (i=1; i<=8; i++) {

    //-- Obtener servo real j, a partir del virtual
    j=v[i];

    //-- Si ese servo se usa, establecer los parametros de su oscilador
    //-- y activar el servo
    if (j!=0) {

     //-- Establecer amplitud
     Am[j-1]=amplitud; 

     //-- Establecer fase
     FASE[j-1]=(i-1)*df + 0;

     //-- Establecer mascara de habilitacion
     mask=mask | (1<<(j-1));
    }
  }

  //-- Activar los servos usados
  servos_enable_mask(mask);
}


void oscillator_calculate(int inc)
{
  int pos_sin;
  unsigned char upos;
  unsigned char i;
  unsigned char s;
    
    //-- Calculos de los osciladores: obtener siguiente posicion para los sevos
    //-- Se ejecuta cada 2.5ms.
    //-- La variable tic se pone a 1 cada 2.5ms (la activa la rutina de atencion
    //-- a las interrupciones del timer0, que se encarga de generar el pwm)
    if (tic==1) {

      //-- Reiniciar tic. Al cabe de 2.5ms se volvera a activar
      tic=0;

      //-- Obtener el numero de servo actual (variable indice) y su
      //-- mascara binaria. Ambas se obtienen del modulo servos.c
      //-- Esta operacion es muy importante, porque indice y sevo_actual
      //-- son variables globales que se modifican asincronamente mediante
      //-- interrupciones, por lo que es necesario crear una copia local
      //-- en las variables i, s para usarlas en los calculos
      i=indice;
      s=servo_actual;

      //-- Calcular la siguiente posicion del servo i
      pos_sin=sin(Am[i],fase+FASE[i]);

      //-- Realizar la conversion entre grados y tiempo
      upos = servos_grados2tiempo(pos_sin);
  
      //-- Refrescar la posicion del servo
      pos_servos[i]=upos;

      //-- Cuando se llega al servo 0, se incrementa la fase
      if (i==0) {
        //-- Incrementar la fase
        fase=fase + inc;

        //-- Las fases negativas se convierten a positivas
        //-- BUG: para fases negativas deveria ser: fase=360-fase
        if (fase<0) fase=359;

        //-- Mantener la fase en el rango 0 - 360 grados
        fase = fase % 360;
      }
    } 
}
