/*****************************************************************************/
/* test-oscillator8.c  Enero-2010                                            */
/*---------------------------------------------------------------------------*/
/* Ejemplo de oscilacion de 8 servos                                         */
/*---------------------------------------------------------------------------*/
/*  Juan Gonzalez <juan@iearobotics.com>                                     */
/*  Andres Prieto-Moreno <andres@iearobotics.com>                            */
/*---------------------------------------------------------------------------*/
/*  LICENCIA GPL                                                             */
/*****************************************************************************/

//-- Especificar el pic a emplear
#include <pic16f877.h>
#include "servos.h"
#include "sin.h"


/******************************************/
/*  PARAMETROS DE LAS OSCILACIONES        */
/******************************************/

//-- Velocidad de fase (en grados) (1-180)
//-- Este parametro permite cambiar la frecuencia de oscilacion, asi como
//-- el sentido. Es un numero entero
//-- Valores negativos hacen que oscile en sentido contrario
//-- El periodo de oscilacion se calcula con la formula:
//--  T=360/W, con W expresado en grados/seg. Para los servos futaba,
//-- la velocidad angular es aproximadamente de 260 grados/seg, por lo que
//-- la formula queda:
//--
//--  T=7.2/INC
//--
//--  Tabla de valores de T en funcion de INC:
//--
//--      INC      T (seg)   f(Hz)
//--    -----------------------------
//--       0       no oscila   0
//--       1       7.2         0.14
//--       2       3.7         0.28
//--       3       2,4         0.42
//--       4       1.8         0.55
//--       5       1.4         0.71
//--       6       1.2         0.83
//--       7       1.03        0.97
//--       8       0.9         1.11
//--       9       0.8         1.25
//--      10       0.72        1.39
//--      ..       ...         ...
#define INC 4

//-- Offset (en grados) (entre -90 y 90)
#define OFFSET 0

//--Fase inicial de los servos (en grados) (0-360)
#define FASE0 0

/**********************************************/
/* VARIABLES GLOBALES                         */
/**********************************************/

//-------------------------------------------------
//-- PARAMETROS A MODIFICAR PARA EL EXPERIMENTO
//---
//-- Fase iniciales de las oscilaciones
//-- Grupo 1:    0, -30,  -60
//-- Grupo 2:  -90, -120, -150
//-- Grupo 3: -180, -210, -240


//-- Servos:        1  2  3   4   5   6   7  8
static int FASE[8]={0, 180, 0, 0, 0, 180, 0, 270};

//--------------------------------------------------
//-- Amplitudes de las oscilaciones
static int A[8]=   {0,40,0,0,0,20,0,40};

//-- Variable de fase
static int fase;

//-- Acceso a las variables del modulo SERVOS
extern volatile unsigned char pos_servos[8];
extern unsigned char indice;
extern unsigned char tic;
extern unsigned char servo_actual; 

/*******************************************************/
/* RUTINA DE ATENCION A LAS INTERRUPCIONES             */
/*******************************************************/
void intr(void) interrupt 0
{
  //---------------------------------------------------------------
  //-- Rutina de servicio del timer0 para actualizar las senales
  //-- de PWM
  //-----------------------------------------------------------
  if (T0IF==1)
    servos_intr();
}



void oscillator_calculate()
{
  int pos_sin;
  unsigned char upos;
  unsigned char i;
  unsigned char s;
    
     //-- Calculos de los osciladores: obtener siguiente posicion para los sevos
    //-- Se ejecuta cada 2.5ms.
    //-- La variable tic se pone a 1 cada 2.5ms (la activa la rutina de atencion
    //-- a las interrupciones del timer0, que se encarga de generar el pwm)
    if (tic==1) {

      //-- Reiniciar tic. Al cabe de 2.5ms se volvera a activar
      tic=0;

      //-- Obtener el numero de servo actual (variable indice) y su
      //-- mascara binaria. Ambas se obtienen del modulo servos.c
      //-- Esta operacion es muy importante, porque indice y sevo_actual
      //-- son variables globales que se modifican asincronamente mediante
      //-- interrupciones, por lo que es necesario crear una copia local
      //-- en las variables i, s para usarlas en los calculos
      i=indice;
      s=servo_actual;

      //-- Calcular la siguiente posicion del servo i
      pos_sin=sin(A[i],fase+FASE[i]+FASE0)+OFFSET;

      //-- Realizar la conversion entre grados y tiempo
      upos = servos_grados2tiempo(pos_sin);
  
      //-- Refrescar la posicion del servo
      pos_servos[i]=upos;

      //-- Cuando se llega al servo 0, se incrementa la fase
      if (i==0) {
        //-- Incrementar la fase
        fase=fase + INC;

        //-- Las fases negativas se convierten a positivas
        //-- BUG: para fases negativas deveria ser: fase=360-fase
        if (fase<0) fase=359;

        //-- Mantener la fase en el rango 0 - 360 grados
        fase = fase % 360;
      }
    }

}

//----------------------------
//- Comienzo del programa  
//----------------------------

void main(void)
{
  //-- Inicializacion del modulo de servos
  servos_init();

  //-- Inicializar variables
  fase=0;
    
  //-- Todos los servos activados
  servos_enable_mask(0xff);

  //-- Habilitar Interrupciones
  GIE=1;

  //----------------------
  //-- Bucle principal
  //----------------------
  while (1) {
      
     //-- Refrescar osciladores
     oscillator_calculate();
  }
}
